<?php

namespace KaliForms\Inc;

if (!defined('ABSPATH')) {
	exit;
}

use KaliForms\Inc\Libs\WebHook_Request;
use KaliForms\Inc\Utils\Condition_Checker;

/**
 * WebHooks handler
 */
class WebHooks_Handler
{
	/**
	 * Plugin slug
	 *
	 * @var string
	 */
	protected $slug = 'kaliforms-webhooks';
	/**
	 * Parent plugin slug
	 *
	 * @var string
	 */
	protected $parent_slug = 'kaliforms';
	/**
	 * Class constructor
	 */
	public function __construct()
	{
		add_action(
			'wp_ajax_kaliforms_test_webhook',
			[$this, 'test_webhook']
		);
		add_action(
			'wp_ajax_nopriv_kaliforms_test_webhook',
			[$this, 'denied']
		);

		add_filter($this->parent_slug . '_before_form_process', [$this, 'send_hook_before']);
		add_filter($this->parent_slug . '_after_form_process', [$this, 'send_hook_after']);
	}
	/**
	 * If the user is not authorized, deny action
	 */
	public function denied()
	{
		wp_die(esc_html__('Denied', 'kaliforms'));
	}
	/**
	 * Test a webhook
	 *
	 * @return void
	 */
	public function test_webhook()
	{
		if (!isset($_POST['args'], $_POST['args']['nonce'])) {
			wp_die(wp_json_encode([
				'success' => false,
				'message' => esc_html__('Denied', 'kaliforms'),
			]));
		}
		if (!wp_verify_nonce(sanitize_key(wp_unslash($_POST['args']['nonce'])), $this->parent_slug . '_nonce')) {
			wp_die(wp_json_encode([
				'success' => false,
				'message' => esc_html__('Denied', 'kaliforms'),
			]));
		}

		if (!isset($_POST['args']['hook'])) {
			wp_die(wp_json_encode([
				'success' => false,
				'message' => esc_html__('No hook to test', 'kaliforms'),
			]));
		}

		$hook = json_encode(stripslashes_deep($_POST['args']['hook']));
		$hook = json_decode($hook);

		if (!filter_var($hook->url, FILTER_VALIDATE_URL)) {
			wp_die(wp_json_encode([
				'success' => false,
				'message' => esc_html__('Not a valid URL', 'kaliforms'),
			]));
		}

		$wh_req = WebHook_Request::get_instance($hook);
		$wh_req->set_payload($this->mock_payload($hook));
		$wh_req->call();

		if ($wh_req->is_error) {
			wp_die(wp_json_encode([
				'success' => false,
				'message' => $wh_req->request_status['message'],
			]));
		}

		wp_die(wp_json_encode([
			'success' => true,
		]));
	}
	/**
	 * Mock a payload object
	 *
	 * @param $hook
	 * @return void
	 */
	public function mock_payload($hook)
	{
		$body = [];
		foreach ($hook->body as $item) {
			if (empty($item->key) || empty($item->value)) {
				continue;
			}
			$body[$item->key] = $this->generate_random_string(mt_rand(3, 10));
		}

		return $body;
	}
	/**
	 * Generate a random stirng
	 *
	 * @param integer $length
	 * @return void
	 */
	public function generate_random_string($length = 10)
	{
		$characters       = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
		$charactersLength = strlen($characters);
		$randomString     = '';
		for ($i = 0; $i < $length; $i++) {
			$randomString .= $characters[rand(0, $charactersLength - 1)];
		}
		return $randomString;
	}
	/**
	 * Send hook before
	 *
	 * @return void
	 */
	public function send_hook_before($data)
	{
		$hooks = $this->get_hook_data($data, 'beforeFormProcess');
		if (!$hooks) {
			return $data;
		}

		$data['kf_hooks'] = [];
		$placeholders     = [];
		foreach ($hooks as $hook) {
			$hook_sent                     = $this->_send_hook($hook);
			$data['kf_hooks'][$hook->name] = $hook_sent;
			if ($hook_sent['success']) {
				$placeholders = array_merge($placeholders, $this->_extract_placeholders($hook->name, $hook_sent));
			}
		}
		$data['kf_hooks'] = $placeholders;

		return $data;
	}
	/**
	 * Send hook after
	 *
	 * @return void
	 */
	public function send_hook_after($data)
	{
		$hooks = $this->get_hook_data($data, 'afterFormProcess');
		if (!$hooks) {
			return $data;
		}
		if (!isset($data['kf_hooks'])) {
			$data['kf_hooks'] = [];
		}
		$placeholders = [];
		foreach ($hooks as $hook) {
			$hook_sent                     = $this->_send_hook($hook);
			$data['kf_hooks'][$hook->name] = $hook_sent;
			if ($hook_sent['success']) {
				$placeholders = array_merge($placeholders, $this->_extract_placeholders($hook->name, $hook_sent));
			}
		}
		$data['kf_hooks'] = $placeholders;
		return $data;
	}
	/**
	 * Send hook
	 *
	 * @param [type] $hook
	 * @return void
	 */
	protected function _send_hook($hook)
	{
		$wh_req = new WebHook_Request($hook);
		$body   = [];
		foreach ($hook->body as $item) {
			if (empty($item->key) || empty($item->value)) {
				continue;
			}

			$body[$item->key] = $item->value;
		}
		$wh_req->set_payload($body);
		return $wh_req->call();
		// return !$wh_req->is_error;
	}
	/**
	 * Get hook data
	 *
	 * @param $formData
	 * @param $event
	 * @return void
	 */
	public function get_hook_data($formData, $event)
	{
		$hooks = get_post_meta($formData['formId'], $this->parent_slug . '_webhooks', true);
		if ($hooks === null || $hooks === '') {
			$hooks = [];
		}

		if (empty($hooks)) {
			return false;
		}

		$to_send = [];
		foreach ($hooks as $hook) {
			if ($hook->event !== $event) {
				continue;
			}

			if (!Condition_Checker::check_conditions($hook->conditions, $formData)) {
				continue;
			};

			// Replace with current values
			foreach ($hook->body as $item) {
				$item->value = isset($formData[$item->value]) ? $formData[$item->value] : '';
			}

			$to_send[] = $hook;
		}

		return $to_send;
	}
	/**
	 * Extracts placeholders
	 *
	 * @param [string] $name
	 * @param [type] $hook
	 * @return void
	 */
	private function _extract_placeholders($name, $hook)
	{
		$placeholders = [];
		if (is_array($hook['content'])) {
			foreach ($hook['content'] as $k => $content) {
				$placeholders["{{$name}:{$k}}"] = $content;
			}

			return $placeholders;
		}

		$placeholders["{{$name}}"] = $hook['content'];

		return $placeholders;
	}
}
