<?php

namespace KaliForms\Inc\Libs;

if (!defined('ABSPATH')) {
	exit;
}

class WebHook_Request
{
	/**
	 * Is error
	 *
	 * @var boolean
	 */
	public $is_error = false;
	/**
	 * Request status
	 *
	 * @var array
	 */
	public $request_status = [];
	/**
	 * Url
	 *
	 * @var string
	 */
	protected $url = '';
	/**
	 * Payload
	 *
	 * @var array
	 */
	protected $payload = [];
	/**
	 * Custom headers
	 *
	 * @var array
	 */
	protected $headers = [];
	/**
	 * Current hook
	 *
	 * @var array
	 */
	private $hook = [];
	/**
	 * Request Method
	 *
	 * @var string
	 */
	protected $method = 'POST';
	/**
	 * Request Format
	 *
	 * @var string
	 */
	protected $format = 'json';
	/**
	 * Class constructor
	 */
	public function __construct($hook)
	{
		$this->hook = $hook;

		$this->url    = $this->hook->url;
		$this->format = $this->hook->format;
		$this->method = $this->hook->method;

		$this->create_headers();
	}
	/**
	 * Get an instance of this class
	 *
	 * @param [type] $hook
	 * @return void
	 */
	public static function get_instance($hook)
	{
		static $inst;
		if (!$inst) {
			$inst = new WebHook_Request($hook);
		}

		return $inst;
	}
	/**
	 * Create custom headers to know that they are coming from us
	 *
	 * @return void
	 */
	public function create_headers()
	{
		$format = '';

		switch ($this->format) {
			case 'json':
				$format = 'application/json; charset=UTF-8';
				break;
			case 'form':
				$format = 'application/x-www-form-urlencoded; charset=UTF-8';
				break;
		}

		if (in_array($this->method, ['POST', 'PUT', 'PATCH', 'DELETE'])) {
			$this->headers['Content-Type'] = $format;
		}

		$this->headers['X-App']       = 'Kali Forms Webhooks/v:' . KALIFORMS_WEBHOOKS_VERSION;
		$this->headers['X-App-Event'] = $this->hook->event;
		if (!empty($this->hook->authentication)) {
			$this->headers['X-Kali-Auth'] = strpos($this->hook->authentication, 'constant') > -1
				? $this->_explode_and_evaluate($this->hook->authentication)
				: $this->hook->authentication;
		}

		foreach ($this->hook->headers as $item) {
			if (empty($item->key) || empty($item->value)) {
				continue;
			}

			if (strpos($item->value, 'constant') > -1) {
				$item->value = $this->_explode_and_evaluate($item->value);
			}

			if (strtolower(trim($item->key)) === 'authorization') {
				$this->headers['Authorization'] = esc_attr($item->value);
				continue;
			}

			$this->headers['X-' . esc_attr(trim($item->key))] = esc_attr($item->value);
		}
	}
	/**
	 * Should return constant if used
	 *
	 * @param [type] $string
	 * @return void
	 */
	protected function _explode_and_evaluate($string)
	{
		$ivalue = preg_split('/\s+/', $string);

		foreach ($ivalue as $idx => $str) {
			$str          = str_replace('constant.', '', $str);
			$ivalue[$idx] = defined($str) ? constant($str) : $str;
		}

		return implode(' ', $ivalue);
	}
	/**
	 * Set payload
	 *
	 * @param [type] $data
	 * @return void
	 */
	public function set_payload($data)
	{
		$this->payload = $data;
	}
	/**
	 * Get payload
	 *
	 * @param [type] $data
	 * @return void
	 */
	public function get_payload($data)
	{
		return $this->payload;
	}
	/**
	 * Set url
	 *
	 * @param [type] $url
	 * @return void
	 */
	public function set_url($url)
	{
		$this->url = $url;
	}
	/**
	 * Get url
	 *
	 * @return void
	 */
	public function get_url()
	{
		return $this->url;
	}
	/**
	 * Call function
	 *
	 * @return void
	 */
	public function call()
	{
		if (empty($this->url)) {
			$this->is_error       = true;
			$this->request_status = [
				'success' => false,
				'message' => esc_html__('No URL specified', 'kaliforms'),
			];
			return;
		}

		$args = [
			'headers' => $this->headers,
			'method'  => $this->method,
			'body'    => $this->format === 'json' ? wp_json_encode($this->payload) : $this->payload,
		];

		$url = html_entity_decode($this->url);

		if ($this->method === 'GET' && is_array($this->payload) && count(array_filter($this->payload)) > 0) {
			$glue = strpos($url, '?') > -1 ? '&' : '?';
			$url  = $url . $glue . http_build_query($this->payload);
		}

		$response = wp_remote_request(
			$url,
			$args
		);

		if (is_wp_error($response)) {
			$this->request_status = [
				'success' => false,
				'message' => $response->get_error_message(),
			];
			$this->is_error = true;
			return;
		}

		if ($response['response']['code'] !== 200) {
			$this->request_status = [
				'success' => false,
				'message' => $response['response']['message'],
			];
			$this->is_error = true;

			return;
		}

		$body = wp_remote_retrieve_body($response);

		$this->request_status = [
			'success' => true,
			'content' => $this->_get_content($body),
		];

		return $this->request_status;
	}

	/**
	 * Gets content from the request
	 *
	 * @param [type] $data
	 * @return void
	 */
	private function _get_content($data)
	{
		return is_object(json_decode($data)) ? json_decode($data, true) : $data;
	}
}
