<?php

// Exit if accessed directly.
if (!defined('ABSPATH')) exit;

/**
 * Class Rdg_Blocks_Helpers
 *
 * This class contains repetitive functions that
 * are used globally within the plugin.
 *
 * @package		RDGBLOCKS
 * @subpackage	Classes/Rdg_Blocks_Helpers
 * @author		RDGCorp
 * @since		1.0.0
 */
class Rdg_Blocks_Helpers
{
    /**
     * ######################
     * ###
     * #### CALLABLE FUNCTIONS
     * ###
     * ######################
     */

    /**
     * HELPER COMMENT START
     *
     * Within this class, you can define common functions that you are 
     * going to use throughout the whole plugin. 
     * 
     * Down below you will find a demo function called output_text()
     * To access this function from any other class, you can call it as followed:
     * 
     * RDGBLOCKS()->helpers->output_text( 'my text' );
     * 
     */

    /**
     * The blocks cache directory
     *
     * @var		array
     * @since   1.0.13
     */

    private $cache_dir;

    /**
     * The blocks cache suffix
     *
     * @var		array
     * @since   1.0.13
     */

    private $cache_suffix;

    /**
     * Our Rdg_Blocks_Settings constructor 
     * to run the plugin logic.
     *
     * @since 1.0.0
     */

    function __construct() {
        $this->cache_dir = wp_upload_dir()['basedir'].'/rdg-blocks/';
        $this->cache_suffix = '-cache.log';
    }

    /**
     * Output some text
     *
     * @param	string	$text	The text to output
     * @since	1.0.0
     *
     * @return	void
     */
    public function output_text( $text = '' ) {
        echo $text;
    }

    public function get_cache_dir() {
        return $this->cache_dir;
    }

    public function get_cache_suffix() {
        return $this->cache_suffix;
    }

    /**
     * HELPER COMMENT END
    */

    public function webpack($file, $buildDirectory = 'dist') {
        static $manifest;
        static $manifestPath;

        if (is_null($manifest) || $manifestPath !== $buildDirectory) {
            $manifest = json_decode(
                file_get_contents(RDGBLOCKS_PLUGIN_URL.$buildDirectory.'/manifest.json'),
                true
            );

            $manifestPath = $buildDirectory;
        }

        if (isset($manifest[$file])) {
            return RDGBLOCKS_PLUGIN_URL.trim($buildDirectory.'/'.$manifest[$file], '/');
        }

        throw new InvalidArgumentException("File {$file} not defined in asset manifest.");
    }

    public function get_blocks_spaces($fields) {
        $styles = [];

        if (!empty($fields['top'])) {
            $styles[] = $fields['top']['top_type'].'-top: '.$fields['top']['top_space'].'rem';
        }

        if (!empty($fields['right'])) {
            $styles[] = $fields['right']['right_type'].'-right: '.$fields['right']['right_space'].'rem';
        }

        if (!empty($fields['bottom'])) {
            $styles[] = $fields['bottom']['bottom_type'].'-bottom: '.$fields['bottom']['bottom_space'].'rem';
        }

        if (!empty($fields['left'])) {
            $styles[] = $fields['left']['left_type'].'-left: '.$fields['left']['left_space'].'rem';
        }

        if (count($styles) > 0) {
            return implode('; ', $styles).';';
        }

        return '';
    }
    
    public function get_title_sizes($fields) {
        $size = (!empty($fields['size'])) ? $fields['size'] : 3;
        $size_lg = (!empty($fields['size_lg'])) ? $fields['size_lg'] : null;
        $size_md = (!empty($fields['size_md'])) ? $fields['size_md'] : null;
        $size_sm = (!empty($fields['size_sm'])) ? $fields['size_sm'] : null;
    
        $style = [];
        
        if (!empty($size) || !empty($size_lg) || !empty($size_md) || !empty($size_sm)) {
            if (!empty($size)) {
                $last = $size;
                $style[] = '--font-size: '.$size.'rem';
                $style[] = '--line-height: '.(intval($size) + 0.5).'rem';
            } else {
                $last = 3;
                $style[] = '--font-size-lg: '.$last.'rem';
                $style[] = '--line-height-lg: '.($last + 0.5).'rem';
            }
    
            if (!empty($size_lg)) {
                $last = $size_lg;
                $style[] = '--font-size-lg: '.$size_lg.'rem';
                $style[] = '--line-height-lg: '.(intval($size_lg) + 0.5).'rem';
            } else {
                $style[] = '--font-size-lg: '.$last.'rem';
                $style[] = '--line-height-lg: '.($last + 0.5).'rem';
            }
    
            if (!empty($size_md)) {
                $last = $size_md;
                $style[] = '--font-size-md: '.$size_md.'rem';
                $style[] = '--line-height-md: '.(intval($size_md) + 0.5).'rem';
            } else {
                $style[] = '--font-size-md: '.$last.'rem';
                $style[] = '--line-height-md: '.($last + 0.5).'rem';
            }
            
            if (!empty($size_sm)) {
                $style[] = '--font-size-sm: '.$size_sm.'rem';
                $style[] = '--line-height-sm: '.(intval($size_sm) + 0.5).'rem';
            } else {
                $style[] = '--font-size-sm: '.$last.'rem';
                $style[] = '--line-height-sm: '.($last + 0.5).'rem';
            }
        }
    
        return $style;
    }

    public function get_api_data($block_name, $attibutes = []) {
        $data = new StdClass();
        $data->json = '';
        $data->message = '';
        $api_request = false;
        $cache_file = $this->cache_dir.$block_name.$this->cache_suffix;

        if (!file_exists($cache_file)) {
            $set_cache_file = $this->set_cache_file($block_name);

            if (empty($set_cache_file->cache_file)) {
                $data->message = $set_cache_file->message;
                
                return $data;
            } else {
                $api_request = true;
                $cache_file = $set_cache_file->cache_file;
            }
        }

        $cache_refresh = (isset($attibutes['cache_refresh']) && !empty($attibutes['cache_refresh'])) 
            ? intval($attibutes['cache_refresh']) : 1;
        
        $cache_time = ((time() - filemtime($cache_file)) / 60) / 60;
        //$cache_time = 50;

        if ((($cache_time < $cache_refresh) && $api_request == false) || (($cache_time > $cache_refresh) && $api_request == true)) {
            $cache = file_get_contents($cache_file);
            $data->json = unserialize($cache);

            //dump('Load cache');
            return $data;
        }

        $items = (isset($attibutes['items']) && !empty($attibutes['items'])) 
            ? intval($attibutes['items']) : 10;
        $request = (isset($attibutes['request']) && !empty($attibutes['request'])) 
            ? $attibutes['request'] : '';
        $request_method = (isset($attibutes['method']) && !empty($attibutes['method'])) 
            ? $attibutes['method'] : 'GET';
        $request_headers = (isset($attibutes['headers']) && !empty($attibutes['headers'])) 
            ? $attibutes['headers'] : [];

        //dump('Load API');
        $response = $this->get_api_request($request, $request_method, $request_headers);

        if ($response->http_code == 200) {
            $response_json = $this->is_json($response->data);

            if ($response_json != null) {
                if (count($response_json) != $items) {
                    array_pop($response_json);
                }

                $data->json = $response_json;

                try {
                    $cache = fopen($cache_file, "w");
                    fwrite($cache, serialize($response_json));
                    fclose($cache);
                } catch (Exception $exception) {
                    if (file_exists($cache_file)) {
                        unlink($cache_file);
                    }

                    $data->message = $this->get_api_error('create_cache', ['block_name' => $block_name]);
                }

                return $data;
            }

            $response->http_code = 'json_response';
        }

        $error = $this->get_api_error($response->http_code, ['block_name' => $block_name]);
        $data->message = $error;
        
        return $data;
    }

    public function set_cache_file($block_name) {
        $file = New StdClass();
        $file->code = 200;
        $file->message = null;
        $file->cache_file = null;
        $cache_file = $this->cache_dir.$block_name.$this->cache_suffix;

        if (!file_exists($this->cache_dir)) {
            try {
                mkdir($this->cache_dir);
            } catch (Exception $exception) {
                if (file_exists($this->cache_dir)) {
                    unlink($this->cache_dir);
                }

                $error = $this->get_api_error('create_directory', ['block_name' => $block_name]);
                $file->code = 'create_directory';
                $file->message = $error;

                return $file;
            }
        }

        if (!file_exists($cache_file)) {
            try {
                $cache = fopen($cache_file, "w");
                fclose($cache);
            } catch (Exception $exception) {
                if (file_exists($cache_file)) {
                    unlink($cache_file);
                }
    
                $error = $this->get_api_error('create_cache', ['block_name' => $block_name]);
                $file->code = 'create_cache';
                $file->message = $error;
    
                return $file;
            }
        }

        $file->cache_file = $cache_file;

        return $file;
    }

    public function get_api_request($request, $method, $headers = []) {
        $curl = curl_init();
        $response = new StdClass();

        curl_setopt_array($curl, [
            CURLOPT_URL => $request,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => $headers,
        ]);

        try {
            $api_response = curl_exec($curl);
            curl_close($curl);
        } catch (Exception $exception) {
            curl_close($curl);
            $response->http_code = 'api_request';

            return $response;
        }

        $response->http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $response->data = $api_response;
        
        return $response;
    }

    public function is_json($string) {
        $encode = json_decode($string);
        $response = (json_last_error() === JSON_ERROR_NONE) ? $encode : null;

        return $response;
    }

    public function get_api_error($error, $attributes = null) {
        if ($error == 0) {
            $message = 'Invalid or unavailable URL.';
            error_log(print_r(['error' => $error, 'message' => $message, 'attributes' => $attributes], true));

            return __($message);
        }

        if ($error == 200) {
            error_log('Block: '. $attributes['block_name'].'. The cache was updated correctly.');

            return;
        }

        if ($error == 404) {
            $message = 'Resource not found.';
            error_log(print_r(['error' => $error, 'message' => $message, 'attributes' => $attributes], true));

            return __($message);
        }

        if ($error == 401) {
            $message = 'Not authorized.';
            error_log(print_r(['error' => $error, 'message' => $message, 'attributes' => $attributes], true));

            return __($message);
        }
        
        if ($error == 'json_response') {
            $message = 'Invalid data format.';
            error_log(print_r(['error' => $error, 'message' => $message, 'attributes' => $attributes], true));

            return __($message);
        }

        if ($error == 'api_request') {
            $message = 'The request could not be processed.';
            error_log(print_r(['error' => $error, 'message' => $message, 'attributes' => $attributes], true));

            return __($message);
        }

        if ($error == 'create_cache') {
            $message = 'Error trying to create cache.';
            error_log(print_r(['error' => $error, 'message' => $message, 'attributes' => $attributes], true));

            return __($message);
        }

        if ($error == 'refresh_cache') {
            $message = 'Error trying to update cache.';
            error_log(print_r(['error' => $error, 'message' => $message, 'attributes' => $attributes], true));

            return __($message);
        }

        if ($error == 'create_directory') {
            $message = 'Error trying to create directory.';
            error_log(print_r(['error' => $error, 'message' => $message, 'attributes' => $attributes], true));

            return __($message);
        }

        $message = 'The data could not be accessed, please try again later.';
        error_log(print_r(['error' => $error, 'message' => $message, 'attributes' => $attributes], true));

        return $message;
    }
}
