<?php

// Exit if accessed directly.
if (!defined('ABSPATH')) exit;
/**
 * HELPER COMMENT START
 * 
 * This class contains all of the plugin related settings.
 * Everything that is relevant data and used multiple times throughout 
 * the plugin.
 * 
 * To define the actual values, we recommend adding them as shown above
 * within the __construct() function as a class-wide variable. 
 * This variable is then used by the callable functions down below. 
 * These callable functions can be called everywhere within the plugin 
 * as followed using the get_plugin_name() as an example: 
 * 
 * RDGBLOCKS->settings->get_plugin_name();
 * 
 * HELPER COMMENT END
 */

/**
 * Class Rdg_Blocks_Settings
 *
 * This class contains all of the plugin settings.
 * Here you can configure the whole plugin data.
 *
 * @package		RDGBLOCKS
 * @subpackage	Classes/Rdg_Blocks_Settings
 * @author		RDGCorp
 * @since		1.0.0
 */

class Rdg_Blocks_Settings
{
    /**
     * The plugin name
     *
     * @var		string
     * @since   1.0.0
     */

    private $plugin_name;

    /**
     * The blocks list
     *
     * @var		array
     * @since   1.0.0
     */

    private $block_list;

    /**
     * Our Rdg_Blocks_Settings constructor 
     * to run the plugin logic.
     *
     * @since 1.0.0
     */

    function __construct() {
        $this->plugin_name = RDGBLOCKS_NAME;
        $this->block_list = $this->blocks_load();
    }

    /**
     * ######################
     * ###
     * #### CALLABLE FUNCTIONS
     * ###
     * ######################
     */

    /**
     * Return the plugin name
     *
     * @access	public
     * @since	1.0.0
     * @return	string The plugin name
     */

    public function get_plugin_name() {
        return apply_filters('RDGBLOCKS/settings/get_plugin_name', $this->plugin_name);
    }

    /**
     * Load plugin blocks
     *
     * @access	public
     * @since	1.0.0
     * @return	array Plugin blocks
     */

    private function blocks_load() {
        add_filter('block_categories_all', [$this, 'custom_blocks'], 10, 2);
        add_filter('render_block', [$this, 'replace_block_class'], 10, 2);
        
        $block_classes = [];
        foreach (glob(RDGBLOCKS_PLUGIN_DIR . "/blocks/*/main.php") as $block_file) {
            $block = include $block_file;
            $block_classes[$block->get_block_name()] = $block;
        }

        return $block_classes;
    }

    public function get_blocks() {
        return $this->block_list;
    }

    public function get_block($block = null) {
        if ($block !== null && isset($this->block_list[$block])) {
            return $this->block_list[$block];
        }

        return null;
    }

    public function custom_blocks($categories) {
        return array_merge(
            $categories,
            array(
                array(
                    'slug' => RDGBLOCKS_NAME_SLUG,
                    'title' => __('RDG Blocks', RDGBLOCKS_NAME_SLUG),
                ),
            )
        );
    }

    public function replace_block_class($block_content, $block) {
        $full_grids = ['full-screen'];
    
        if (isset($block['attrs']['className']) && !empty($block['attrs']['className'])) {
            $classes = explode(' ', $block['attrs']['className']);
    
            foreach ($classes as $class) {
                if (in_array($class, $full_grids)) {
                    $Id = '';

                    if (isset($block['attrs']['uniqueId'])) {
                        $Id = ' block-'.$block['attrs']['uniqueId'];
                    } else {
                        $classes_block = explode(' ', $block['attrs']['className']);
                        foreach ($classes_block as $block) {
                            if (strpos($block, 'block-') !== false) {
                                $Id = ' '.$block;
                            }
                        }
                    }
    
                    return '<div class="full-screen-container'.$Id.'"><div class="container-generate grid-container">'.$block_content.'</div></div>';
                }
            }
        }
    
        return $block_content;
    }
}
