<?php

// Exit if accessed directly.
if (!defined('ABSPATH')) exit;

class Rdg_Blocks_Handicappers_Hub extends Rdg_Blocks_Base implements Rdg_Blocks_Api_Request
{
    protected $block_name = BLOCK_RDG_HANDICAPPERS_HUB;
    private $block_folder_name = BLOCK_RDG_HANDICAPPERS_HUB;
    protected $block_title = 'RDG Handicappers Hub';
    private $block_description = 'This block displays a list of handicappers, filtered by package, sport and date range.';
    private $block_icon = 'admin-links';
    private $block_keywords = ['rdg', 'custom', 'handicapper', 'hub', 'api'];
    private $block_mode = 'preview';
    private $block_category = RDGBLOCKS_NAME_SLUG;
    protected $is_caching = true;
    protected $is_config = true;
    protected $is_styles_load = true;
    protected $config_page_title = 'RDG handicappers Hub';
    protected $config_menu_title = 'Handicappers Hub';
    protected $interval_field = 'handicappers_hub_interval';

    public function init() {
        $this->register_block();
        add_action('acf/init', [$this, 'register_config_menu'], 20);
        add_action('acf/init', [$this, 'register_fields'], 20);
    }

    private function register_block() {
        if (function_exists('acf_register_block_type')) {
            $rdg_helpers = new Rdg_Blocks_Helpers();
            acf_register_block_type([
                'name' => $this->block_name,
                'title' => __($this->block_title),
                'description' => __($this->block_description),
                'category' => $this->block_category,
                'icon' => $this->block_icon,
                'keywords' => $this->block_keywords,
                'mode' => $this->block_mode,
                'render_template' => RDGBLOCKS_PLUGIN_DIR . "/blocks/".$this->block_folder_name."/template.php",
                'enqueue_style' => $rdg_helpers->webpack('rdg_handicappers_hub.css'),
                'enqueue_script' => $rdg_helpers->webpack('rdg_handicappers_hub_script.js'),
                'enqueue_assets' => function() {
                    wp_register_script('rdg_handicappers_hub_script', RDGBLOCKS_PLUGIN_URL.'blocks/rdg-handicappers-hub/assets/js/rdg-handicappers-hub-script.js', [], '1', true);
                    wp_enqueue_script('rdg_handicappers_hub_script');
                    wp_localize_script('rdg_handicappers_hub_script','rdg_handicappers_hub_var',['ajaxurl' => admin_url('admin-ajax.php')]);
                },
                'example' => array(
                    'attributes' => array(
                        'mode' => 'preview',
                        'data' => array(
                            'isExample' => true,
                            'preview_image' => RDGBLOCKS_PLUGIN_URL."dist/images/rdg-handicappers-hub-preview.jpg",
                        ),
                    ),
                ),
                'supports' => [
                    'jsx' => true,
			        'color' => [
                        'background' => true,
                        'gradients'  => false,
                        'text' => false,
                    ],
                ],
                'render_callback' => function($block) use ($rdg_helpers) {
                    global $wp_query;
                    $loop = $wp_query->in_the_loop;
                    $template_path = RDGBLOCKS_PLUGIN_DIR . "/blocks/".$this->block_folder_name."/template.php";

                    if ($loop) {
                        include $template_path;
                        return;
                    }

                    $trace_config = $rdg_helpers->getBacktraceConfig();
                    $traces = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, $trace_config['max_trace']);
                    $trace_valid_admin = $rdg_helpers->checkExistsTraceFunction($traces, $trace_config['max_trace'], $trace_config['functions']);

                    if (is_admin() && !$trace_valid_admin) {
                        include $template_path;
                        return;
                    }
                },
            ]);
        }
    }

    public function register_fields() {
        $sports_all = $this->getSports();
        $sports = (isset($sports_all['all_sports']) && is_array($sports_all['all_sports'])) ? $sports_all['all_sports'] : [];
        $handicappers = (function_exists('shMappingHandicappers')) ? shMappingHandicappers() : [];

        if( function_exists('acf_add_local_field_group') ):
            acf_add_local_field_group(array(
                'key' => 'group_handicappers_hub_form',
                'title' => 'Block_rdg_handicappers_hub',
                'fields' => array(
                    array(
                        'key' => 'key_handicapppers_hub_handicappers',
                        'label' => 'Handicappers Exclude',
                        'name' => 'handicapppers_hub_handicappers',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_handicapppers_hub_handicapper',
                                'label' => 'Handicapper',
                                'name' => 'handicapper',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $handicappers,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'max' => '',
                                'prepend' => '',
                                'append' => '',
                                'allow_null' => 0,
                                'allow_in_bindings' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'allow_custom' => 0,
                                'search_placeholder' => '',
                                'min' => '',
                                'parent_repeater' => 'key_handicapppers_hub_handicappers',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'key_handicapppers_hub_sport',
                        'label' => 'Sports',
                        'name' => 'handicapppers_hub_sport',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => [0 => 'Choose an option'] + $sports,
                        'default_value' => 0,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'key_handicapppers_hub_date_range',
                        'label' => 'Date Range',
                        'name' => 'handicapppers_hub_date_range',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => $this->getHandicappersHubDateRange(),
                        'default_value' => '365',
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'key_handicapppers_hub_package',
                        'label' => 'Packages',
                        'name' => 'handicapppers_hub_package',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => $this->getHandicappersHubPackages(),
                        'default_value' => '0',
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'key_handicapppers_hub_number_of_items',
                        'label' => 'Number of Items',
                        'name' => 'number_of_items',
                        'aria-label' => '',
                        'type' => 'number',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => 8,
                        'min' => '',
                        'max' => '',
                        'placeholder' => '',
                        'step' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'block',
                            'operator' => '==',
                            'value' => 'acf/rdg-handicappers-hub',
                        ),
                    ),
                ),
                'menu_order' => 1,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
            ));

        endif;
    }

    public function register_config_menu() {
        if( function_exists('acf_add_local_field_group') ):
            acf_add_local_field_group(array(
                'key' => 'api_configuration_handicappers_hub',
                'title' => 'API configuration',
                'fields' => array(
                    array(
                        'key' => 'key_handicappers_hub_api_url',
                        'label' => 'URL',
                        'name' => 'handicappers_hub_api_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '70',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_handicappers_hub_interval',
                        'label' => 'Cache Refresh Frequency (Hours)',
                        'name' => 'handicappers_hub_interval',
                        'aria-label' => '',
                        'type' => 'number',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => 1,
                        'min' => '',
                        'max' => '',
                        'placeholder' => '',
                        'step' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_handicappers_hub_api_token',
                        'label' => 'Token',
                        'name' => 'handicappers_hub_api_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_handicappers_hub_date_range',
                        'label' => 'Date Range',
                        'name' => 'handicappers_hub_date_range',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_handicappers_hub_date_range_value',
                                'label' => 'Value',
                                'name' => 'value',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_handicappers_hub_date_range',
                            ),
                            array(
                                'key' => 'key_handicappers_hub_date_range_title',
                                'label' => 'Title',
                                'name' => 'title',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_handicappers_hub_date_range',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'key_handicappers_hub_packages',
                        'label' => 'Packages',
                        'name' => 'handicappers_hub_packages',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_handicappers_hub_package_value',
                                'label' => 'Value',
                                'name' => 'value',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_handicappers_hub_packages',
                            ),
                            array(
                                'key' => 'key_handicappers_hub_package_title',
                                'label' => 'Title',
                                'name' => 'title',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_handicappers_hub_packages',
                            ),
                        ),
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-handicappers-hub',
                        ),
                    ),
                ),
                'menu_order' => 1,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
            ));

        endif;
    }

    public function getApiAttributes($items = null, $attributes = []) {
        $api_url = get_field('handicappers_hub_api_url', 'option');
        $api_token = get_field('handicappers_hub_api_token', 'option');
        $cache_refresh = get_field('handicappers_hub_interval', 'option');

        $items = (!empty($items)) ? $items : 10;
        $package = (isset($attributes['package']) && !empty($attributes['package'])) ? $attributes['package'] : null;
        $duration = (isset($attributes['duration']) && !empty($attributes['duration'])) ? $attributes['duration'] : 365;
        $capper_exclude = (!empty($attributes['capper_exclude'])) ? $attributes['capper_exclude'] : [];

        $api_url = (substr($api_url, -1) == '/') ? substr($api_url, 0, strlen($api_url) -1) : $api_url;
        $api_url .= '?limit='.$items;

        $sport = null;
        if (isset($attributes['sport']) && !empty($attributes['sport'])) {
            $sport = $attributes['sport'];
            $api_url .= '&id_sport='.$sport;
        }

        if (!empty($package)) {
            $api_url .= '&group='.$package;
        }

        $handicappers_string = '';
        if (is_array($capper_exclude) && count($capper_exclude) > 0) {
            $handicappers_string = $this->gerHandicappersExclude($capper_exclude, false);

            if (!empty($handicappers_string)) {
                $api_url .= '&exclude='.$handicappers_string;
            }
        } else {
            if (!empty($capper_exclude)) {
                $handicappers_string = $capper_exclude;
                $api_url .= '&exclude='.$handicappers_string;
            }
        }	

        $api_url .= '&dateRange='.$duration;
        $api_request = $api_url.'&api_token='.$api_token;

        $request = [
            'items' => $items,
            'package' => $package,
            'duration' => $duration,
            'sport' => $sport,
            'capper_exclude' => $handicappers_string,
            'method' => 'GET',
            'cache_refresh' => $cache_refresh,
            'headers' => ['Accept: application/json'],
            'request' => $api_request,
        ];

        return $request;
    }

    public function getApiParams() {
        return [
            'package' => 'handicapppers_hub_package',
            'duration' => 'handicapppers_hub_date_range',
            '_to_transform_' => [
                [
                    'key' => 'sport',
                    'function' => 'getSportID',
                    'params' => ['handicapppers_hub_sport'],
                ],
                [
                    'key' => 'capper_exclude',
                    'function' => 'gerHandicappersExclude',
                    'params' => ['handicapppers_hub_handicappers'],
                    'repeater' => true,
                    'repeater_format' => 'handicapppers_hub_handicappers_%d_handicapper',
                ],
            ],
        ];
    }

    public function getSportID($sport = 0) {
        return preg_replace('/\D/', '', $sport);
    }

    public function gerHandicappersExclude($handicappers, $is_repeater = true) {
        if ($is_repeater) {
            return implode(',', $handicappers);
        }

        if (is_array($handicappers) && count($handicappers) > 0) {
            return implode(',', array_column($handicappers, 'handicapper'));
        }

        return '';
    }

    public function getResourcesAjax($request) {
        $data = new StdClass();
        $data->json = '';
        $data->message = '';

        $rdgb_helpers = New Rdg_Blocks_Helpers;
        $handicappers_hub = $rdgb_helpers->getApiDataAjax($this->block_name, $request);

        if (!empty($handicappers_hub->message)) {
            $data->message = $handicappers_hub->message;
        
            return $data;
        }

        $data->json = $handicappers_hub->json;

        return $data;
    }

    public function getFieldsValidate() {
        return [
            'name' => ['required', 'notnull'],
            'image' => ['required', 'notnull'],
            'bio' => ['required', 'notnull'],
        ];
    }

    public function getHandicappersHubDateRange() {
        $date_range = [];
        $ranges = get_field('handicappers_hub_date_range', 'option');

        if (is_array($ranges) && count($ranges) > 0) {
            foreach ($ranges as $range) {
                $date_range[$range['value']] = $range['title'];
            }
        }

        return $date_range;
    }

    public function getHandicappersHubPackages() {
        $packages_list = ['0' => 'All'];
        $packages = get_field('handicappers_hub_packages', 'option');

        if (is_array($packages) && count($packages) > 0) {
            foreach ($packages as $package) {
                $packages_list[$package['value']] = $package['title'];
            }
        }

        return $packages_list;
    }

    public function getImage($url_image, $attributes = []) {
        $rdgb_helpers = New Rdg_Blocks_Helpers;
        $url_image = $rdgb_helpers->getImageSource($url_image, $attributes);

        return $url_image;
    }

    public function getSports($default_sports = []) {
        $sports_new = [];
        $default_sports_new = [];
        $sports = (function_exists('shMappingHierarchySports')) ? shMappingHierarchySports() : [];

        if (is_array($sports) && count($sports) > 0) {
            foreach ($sports as $key => $sport) {
                if (in_array(strtolower($sport), $default_sports)) {
                    $default_sports_new[$this->getSportID($key)] = strtoupper($sport);
                } else {
                    $sports_new[$this->getSportID($key)] = strtoupper($sport);
                }
            }
        }

        asort($sports_new);

        return [
            'default_sports' => $default_sports_new,
            'all_sports' => $sports_new,
        ];
    }

    public function getPreSavedData($data, $attributes = [])
    {
        return $data;
    }

    public function getColorsToloadGeneratePress() {
        return [
            ['name' => 'color-rdg-handicappers-hub-header','slug' => 'color-rdg-handicappers-hub-header','color' => '#03467d'],
            ['name' => 'color-rdg-handicappers-hub-header-hover','slug' => 'color-rdg-handicappers-hub-header-hover','color' => '#e11414'],
            ['name' => 'color-rdg-handicappers-hub-bg','slug' => 'color-rdg-handicappers-hub-bg','color' => '#ffffff'],
            ['name' => 'color-rdg-handicappers-hub-border','slug' => 'color-rdg-handicappers-hub-border','color' => '#EEF4FC'],
            ['name' => 'color-rdg-handicappers-hub-shadow','slug' => 'color-rdg-handicappers-hub-shadow','color' => '#F3F3F3'],
            ['name' => 'color-rdg-handicappers-hub-line','slug' => 'color-rdg-handicappers-hub-line','color' => '#67748e66'],
            ['name' => 'color-rdg-handicappers-hub-stats','slug' => 'color-rdg-handicappers-hub-stats','color' => '#878787'],
            ['name' => 'color-rdg-handicappers-hub-btn','slug' => 'color-rdg-handicappers-hub-btn','color' => '#03467d'],
            ['name' => 'color-rdg-handicappers-hub-btn-hover','slug' => 'color-rdg-handicappers-hub-btn-hover','color' => '#3a96e0'],
            ['name' => 'color-rdg-handicappers-hub-btn-default','slug' => 'color-rdg-handicappers-hub-btn-default','color' => '#E9ECEF'],
            ['name' => 'color-rdg-handicappers-hub-scrollbar-track','slug' => 'color-rdg-handicappers-hub-scrollbar-track','color' => '#f1f1f1'],
            ['name' => 'color-rdg-handicappers-hub-scrollbar-thumb','slug' => 'color-rdg-handicappers-hub-scrollbar-thumb','color' => '##88888'],
            ['name' => 'color-rdg-handicappers-hub-scrollbar-hover','slug' => 'color-rdg-handicappers-hub-scrollbar-hover','color' => '##55555'],
            ['name' => 'color-rdg-handicappers-hub-filter-sport','slug' => 'color-rdg-handicappers-hub-filter-sport','color' => '#878787'],
            ['name' => 'color-rdg-handicappers-hub-filter-sport-hover','slug' => 'color-rdg-handicappers-hub-filter-sport-hover','color' => '#e9ecef'],
            ['name' => 'color-rdg-handicappers-hub-filter-border','slug' => 'color-rdg-handicappers-hub-filter-border','color' => '#E9ECEF'],
            ['name' => 'color-rdg-handicappers-hub-filter-bg','slug' => 'color-rdg-handicappers-hub-filter-bg','color' => '#ffffff'],
            ['name' => 'color-rdg-handicappers-hub-filter-package','slug' => 'color-rdg-handicappers-hub-filter-package','color' => '#EEF4FC'],
            ['name' => 'color-rdg-handicappers-hub-filter-package-bg','slug' => 'color-rdg-handicappers-hub-filter-package-bg','color' => '#ffffff'],
        ];
    }

    public function getStylesToloadGeneratePress() {
        return [
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .handicapper-header .header-content',
                'lineHeight' => '1.2rem',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .handicapper-header .header-name',
                'fontSize' => '1.06rem',
                'fontWeight' => '700',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .handicapper-header .header-package',
                'fontSize' => '0.95rem',
                'fontWeight' => '400',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .handicapper-header .header-stats',
                'fontSize' => '0.75rem',
                'fontWeight' => '400',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .handicapper-header .header-stats .success, .rdg-handicappers-hub .handicapper .handicapper-header .header-stats .danger',
                'fontWeight' => '700',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .body-content',
                'fontSize' => '0.95rem',
                'fontWeight' => '400',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .handicapper-footer .button-picks',
                'fontSize' => '1.3rem',
                'margin' => '10px 0px',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .handicapper-footer .button-capper',
                'fontSize' => '1.3rem',
                'fontWeight' => '700',
                'margin' => '10px 0px',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .handicapper-footer .footer-button',
                'lineHeight' => '1rem',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .handicapper .handicapper-footer .footer-button.primary',
                'fontSize' => '0.9rem',
                'fontWeight' => '700',
                'lineHeight' => 'initial',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .filter-sport .default-sports .sport',
                'fontSize' => '0.9rem',
                'textTransform' => 'uppercase',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .filter-sport .default-sports .sport.active',
                'fontWeight' => '700',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .sports-list .dropdown:checked+label, .rdg-handicappers-hub .sports-list .dropdown:not(:checked)+label',
                'fontSize' => '1rem',
                'textTransform' => 'uppercase',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .sports-list .dropdown:not(:checked)+label .uil',
                'fontSize' => '1rem',
                'fontWeight' => '700',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .sports-list .dropdown:checked+label .uil',
                'fontSize' => '1rem',
                'fontWeight' => '700',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .sports-list a',
                'fontSize' => '1rem',
                'fontWeight' => '600',
                'textTransform' => 'uppercase',
                'textDecoration' => 'none',
            ],
            [
                'customSelector' => '.rdg-handicappers-hub .filters .filter-packages',
                'fontSize' => '.9rem',
                'fontWeight' => '700',
            ],
        ];
    }
}

add_action('wp_ajax_nopriv_load_handicappers_hub','getHandicappersHubCappers');
add_action('wp_ajax_load_handicappers_hub','getHandicappersHubCappers');

function getHandicappersHubCappers() {
    $sport = $_POST['sport'];
    $items = $_POST['items'];
    $package_id = $_POST['packageId'];
    $duration = $_POST['duration'];
    $capper_exclude = $_POST['exclude'];

    $capper_exclude = str_replace('\"', '"', $capper_exclude);
    $capper_exclude = str_replace('/"', '"', $capper_exclude);
    $capper_exclude = json_decode($capper_exclude, false);

    $message = null;
    $api_url = get_field('handicappers_hub_api_url', 'option');
    $api_token = get_field('handicappers_hub_api_token', 'option');

    if (!empty($api_url) && !empty($api_token)) {
        global $rdgb;
        $rdgb = (!empty($rdgb)) ? $rdgb : Rdg_Blocks::instance();

        $block_handicappers_hub = $rdgb->settings->getBlock(BLOCK_RDG_HANDICAPPERS_HUB);
        $api_attributes = $block_handicappers_hub->getApiAttributes($items, ['sport' => $sport, 'duration' => $duration, 'package' => $package_id, 'capper_exclude' => $capper_exclude]);
        $response = $block_handicappers_hub->getResourcesAjax($api_attributes);

        if (empty($response->message) && is_array($response->json)) {
            $handicappers = $response->json['handicappers'];
        } else {
            $handicappers = null;
            $message = $response->message;
        }
    } else {
        $message = 'Configure API connection correctly.';
    }

    if (!empty($message)) {
        $message = 'No records were found with the requested characteristics.';
        require 'partials/message.php';
        wp_die();
    }

    $fields = $block_handicappers_hub->getFieldsValidate();

    # Default sports and other sports are defined for the filter
    $default_sports = ['basketball','hockey','football','soccer'];
    $sports = $block_handicappers_hub->getSports($default_sports);
    $packages = $block_handicappers_hub->getHandicappersHubPackages();

    require 'partials/api.php';

    wp_die();
}

$rdg_handicappers_hub_class = new Rdg_Blocks_Handicappers_Hub();
$rdg_handicappers_hub_class->init();

return $rdg_handicappers_hub_class;
