<?php

// Exit if accessed directly.
if (!defined('ABSPATH')) exit;

/**
 * HELPER COMMENT START
 * 
 * This class is used to bring your plugin to life. 
 * All the other registered classed bring features which are
 * controlled and managed by this class.
 * 
 * Within the add_hooks() function, you can register all of 
 * your WordPress related actions and filters as followed:
 * 
 * add_action( 'my_action_hook_to_call', array( $this, 'the_action_hook_callback', 10, 1 ) );
 * or
 * add_filter( 'my_filter_hook_to_call', array( $this, 'the_filter_hook_callback', 10, 1 ) );
 * or
 * add_shortcode( 'my_shortcode_tag', array( $this, 'the_shortcode_callback', 10 ) );
 * 
 * Once added, you can create the callback function, within this class, as followed: 
 * 
 * public function the_action_hook_callback( $some_variable ){}
 * or
 * public function the_filter_hook_callback( $some_variable ){}
 * or
 * public function the_shortcode_callback( $attributes = array(), $content = '' ){}
 * 
 * 
 * HELPER COMMENT END
 */

/**
 * Class Rdg_Blocks_Run
 *
 * Thats where we bring the plugin to life
 *
 * @package		RDGBLOCKS
 * @subpackage	Classes/Rdg_Blocks_Run
 * @author		RDGCorp
 * @since		1.0.0
 */
class Rdg_Blocks_Run {
    /**
     * Our Rdg_Blocks_Run constructor 
     * to run the plugin logic.
     *
     * @since 1.0.0
     */

    function __construct() {
        $this->add_hooks();
    }

    /**
     * ######################
     * ###
     * #### WORDPRESS HOOKS
     * ###
     * ######################
     */

    /**
     * Registers all WordPress and plugin related hooks
     *
     * @access	private
     * @since	1.0.0
     * @return	void
     */

    private function add_hooks() {
        add_action('admin_enqueue_scripts', array( $this, 'enqueue_backend_scripts_and_styles'), 20);
        add_action('acf/init', array( $this, 'add_custom_acf_options_page'), 20);
        register_activation_hook(RDGBLOCKS_PLUGIN_FILE, array( $this, 'activation_hook_callback'));
        add_action('acf/input/admin_enqueue_scripts', array( $this, 'custom_acf_admin_enqueue_scripts'));

        add_action('admin_menu', [$this, 'rdg_add_admin_page']);

    }

    public function rdg_add_admin_page() {
        add_menu_page(
            'Blocks Cache',
            'RDG Blocks Cache',
            'manage_options',
            'rdg-cached-blocks',
            [$this, 'rdg_display_cached_blocks'],
            'dashicons-database',
            82
        );
    }

    public function rdg_display_cached_blocks() {
        global $rdgb;
        $connect = $rdgb->cache->getRedisConnect();

        if (!$connect) {
            echo '<div class="wrap">';
            echo '<h1>Redis connection error</h1>';
            echo '<div class="notice notice-warning"><p><strong>Redis:</strong> Redis connection failed.</p></div>';
            echo '</div>';
            return;
        }

        $filter_type = isset($_GET['filter_type']) ? sanitize_text_field($_GET['filter_type']) : null;
        $filter_type_id = isset($_GET['filter_type_id']) ? sanitize_text_field($_GET['filter_type_id']) : null;
        $filter_block_type = isset($_GET['filter_block_type']) ? sanitize_text_field($_GET['filter_block_type']) : null;

        $blocks = $rdgb->cache->getCacheList();

        echo '<div class="wrap">';
        echo '<h1>Cached Blocks in Redis</h1>';
        echo '<table class="widefat fixed filters_cache">';
        echo '<thead><tr><th>';


        echo '<select class="filter_cache" id="filter_type">';
        echo '<option value="">All Post Types</option>';
        echo '</select>';

        echo '<select class="filter_cache" id="filter_type_id">';
        echo '<option value="">All Post IDs</option>';
        echo '</select>';

        echo '<select class="filter_cache" id="filter_block_type">';
        echo '<option value="">All RDG Blocks</option>';
        echo '</select>';

        echo '<a name="redis-cache-delete-all" href="#" class="button">Delete All</a>';

        echo '</th></tr></thead><tbody>';
        echo '</tbody></table>';
        echo '</div>';

        echo '<div class="wrap">';
        echo '<table class="widefat fixed filters_cache_data">';
        echo '<thead><tr><th style="width: 3%;">Time</th><th style="width: 19%;">Name</th><th style="width: 26%;">Request</th><th style="width: 37%;">Data</th><th style="width: 15%;">Actions</th></tr></thead>';
        echo '<tbody>';

        $i = 1;
        $data_filter_type = [];
        $data_filter_type_ids = [];
        $data_filter_block_types = [];

        foreach ($blocks as $block) {
            [$block_name, $block_type, $block_type_id, $cache] = explode(':', $block);

            if (!in_array($block_type, $data_filter_type)) {
                $data_filter_type[] = $block_type;
            }

            if (!in_array($block_type_id, $data_filter_type_ids)) {
                $data_filter_type_ids[] = $block_type_id;
            }

            if (!in_array($block_name, $data_filter_block_types)) {
                $data_filter_block_types[] = $block_name;
            }

            if (!empty($filter_type) && $block_type != $filter_type) {
                continue;
            }
            if (!empty($filter_type_id) && $block_type_id != $filter_type_id) {
                continue;
            }
            if (!empty($filter_block_type) && $block_name != $filter_block_type) {
                continue;
            }

            $data_block_json = $rdgb->cache->getCacheBlock($block);
            $time = $rdgb->cache->getCacheBlockTime($block);

            if (empty($data_block_json)) {
                $rdgb->cache->delCacheBlock($block);
                continue;
            }

            $data_block = json_decode($data_block_json, true);
            
            echo '<tr class="item_'.$block.'">';
            echo '<td>' . $time . '</td>';
            echo '<td>' . esc_html($block) . '</td>';
            echo '<td><pre style="max-height: 5rem;">' . print_r($data_block['api_attributes'], true) . '</pre></td>';
            echo '<td class="api_response_item_'.$block.'"><pre style="max-height: 5rem;">' . print_r($data_block['api_response'], true) . '</pre></td>';
            echo '<td class="actions">';
            echo '<a name="redis-cache-delete" data-cache="'.$block.'" href="#" class="button">Delete</a>';
            echo '<a name="redis-cache-update" data-cache="'.$block.'" href="#" class="button">Update</a>';
            echo '</td>';
            echo '</tr>';

            $i++;
        }

        ?>
        <script>
            let jsonData = <?php echo json_encode([
                'data_filter_type' => $data_filter_type,
                'data_filter_type_ids' => $data_filter_type_ids,
                'data_filter_block_types' => $data_filter_block_types,
                'selected' => [
                    'filter_type' => $filter_type,
                    'filter_type_id' => $filter_type_id,
                    'filter_block_type' => $filter_block_type,
                ],
            ]); ?>;
        </script>
        <?php

        echo '</tbody></table>';
        echo '</div>';
    }


    public function custom_acf_admin_enqueue_scripts() {
        wp_enqueue_script( 'custom-acf-input-js', RDGBLOCKS_PLUGIN_URL.'core/includes/assets/js/ajax_response.js', false, '1.0.0' );
    }

    /**
     * ######################
     * ###
     * #### WORDPRESS HOOK CALLBACKS
     * ###
     * ######################
     */

    /**
     * Enqueue the backend related scripts and styles for this plugin.
     * All of the added scripts andstyles will be available on every page within the backend.
     *
     * @access	public
     * @since	1.0.0
     *
     * @return	void
     */

    public function enqueue_backend_scripts_and_styles($hook) {
        wp_enqueue_style('rdgblocks-backend-styles', RDGBLOCKS_PLUGIN_URL.'core/includes/assets/css/backend-styles.css', [], RDGBLOCKS_VERSION, 'all');
        wp_enqueue_script('rdgblocks-backend-scripts', RDGBLOCKS_PLUGIN_URL.'core/includes/assets/js/backend-scripts.js', [], RDGBLOCKS_VERSION, false);
        wp_localize_script('rdgblocks-backend-scripts', 'rdgblocks', [
            'plugin_name' => __(RDGBLOCKS_NAME, RDGBLOCKS_NAME_SLUG),
        ]);
        wp_localize_script('rdgblocks-backend-scripts', 'rdg_block_var', [
            'ajaxurl' => admin_url('admin-ajax.php'),
        ]);

        if ($hook == 'toplevel_page_rdg-cached-blocks') {
            wp_enqueue_script('rdgblocks-cache-scripts', RDGBLOCKS_PLUGIN_URL.'core/includes/assets/js/rdg-blocks-cache.js', [], RDGBLOCKS_VERSION, false);
        }
    }

    /**
     * Add the Advanced Custom fields
     * options pages
     *
     * @access	public
     * @since	1.0.0
     * @link	https://www.advancedcustomfields.com/resources/acf_add_options_page/
     *
     * @return	void
     */

    public function add_custom_acf_options_page() {
        // Check function exists.
        if(function_exists('acf_add_options_page')) {
            // Register options page.
            acf_add_options_page([
                'page_title' => __('RDG Blocks'),
                'menu_title' => __('RDG Blocks'),
                'menu_slug' => RDGBLOCKS_NAME_SLUG,
                'capability' => 'edit_posts',
                'redirect' => true
            ]);

            global $rdgb;
            if (!empty($rdgb)) {
                $menus = $rdgb->helpers->getMenus();

                $menus[] = [
                    'page_title'  => __('Cache Maintenance'),
                    'menu_title'  => 'Cache Maintenance',
                    'parent_slug' => RDGBLOCKS_NAME_SLUG,
                ];

                $menus[] = [
                    'page_title'  => __('Categories Maintenance'),
                    'menu_title'  => 'Categories Maintenance',
                    'parent_slug' => RDGBLOCKS_NAME_SLUG,
                ];

                $menus[] = [
                    'page_title'  => __('Images Maintenance'),
                    'menu_title'  => 'Images Maintenance',
                    'parent_slug' => RDGBLOCKS_NAME_SLUG,
                ];

                $menus[] = [
                    'page_title'  => 'Resources',
                    'menu_title'  => 'Resources',
                    'parent_slug' => RDGBLOCKS_NAME_SLUG,
                ];

                $menus[] = [
                    'page_title'  => __('Settings'),
                    'menu_title'  => 'Settings',
                    'parent_slug' => RDGBLOCKS_NAME_SLUG,
                ];

                $menus[] = [
                    'page_title'  => 'Sports Categories',
                    'menu_title'  => 'Sports Categories',
                    'parent_slug' => RDGBLOCKS_NAME_SLUG,
                ];

                // $menus[] = [
                //     'page_title'  => 'Redis Cache',
                //     'menu_title'  => 'Redis Cache',
                //     'parent_slug' => RDGBLOCKS_NAME_SLUG,
                // ];

                function compareMenuTitle($a, $b) {
                    return strcmp($a['menu_title'], $b['menu_title']);
                }

                $order_menu = array_values($menus);
                usort($order_menu, 'compareMenuTitle');

                foreach ($order_menu as $key => $menu) {
                    acf_add_options_page(array(
                        'page_title'  => $menu['page_title'],
                        'menu_title'  => $menu['menu_title'],
                        'parent_slug' => RDGBLOCKS_NAME_SLUG,
                    ));
                }
            }
        }
    }

    

    /**
     * ####################
     * ### Activation/Deactivation hooks
     * ####################
     */
        
    /*
    * This function is called on activation of the plugin
    *
    * @access	public
    * @since	1.0.0
    *
    * @return	void
    */

    public function activation_hook_callback() {
        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        if (!is_plugin_active('advanced-custom-fields-pro/acf.php') && !is_plugin_active('advanced-custom-fields/acf.php')) {
            deactivate_plugins(plugin_basename(__FILE__));
            wp_die(
                'RDGBLOCKS requires Advanced Custom Fields Pro to be active.<br /> <b>Please activate ACF before activating this plugin.</  b>',
                'Activation Error',
                array('back_link' => true)
            );
        }
    }
}

add_action('wp_ajax_nopriv_rdg_blocks_delete_cache','deleteRdgBlocksCache');
add_action('wp_ajax_rdg_blocks_delete_cache','deleteRdgBlocksCache');

function deleteRdgBlocksCache() {
    $cache_name = $_POST['cache'];
    if (empty($cache_name)) wp_die();

    $cache = new Rdg_Blocks_Cache();
    // $connect = $cache->getRedisConnect();
    $deleted = $cache->delCacheBlock($cache_name);

    if (!$deleted) {
        error_log(sprintf('%s / deleteRdgBlocksCache: Failed to delete cache for %s', RDGBLOCKS_NAME, $cache_name));
    }

    wp_die();
}

add_action('wp_ajax_nopriv_rdg_blocks_update_cache','updateRdgBlocksCache');
add_action('wp_ajax_rdg_blocks_update_cache','updateRdgBlocksCache');

function updateRdgBlocksCache() {
    $cache_name = $_POST['cache'];
    if (empty($cache_name)) wp_die();

    global $rdgb;
    $rdgb->cache->getRedisConnect();

    $block = $rdgb->cache->getCacheBlock($cache_name);
    $attributes = json_decode($block, true);
    $block_name = explode(':', $cache_name)[0];
    $block = $rdgb->settings->getBlock(str_replace('_', '-', $block_name));

    if (is_array($attributes)) {
        $api_attributes = $attributes['api_attributes'];

        $response = $rdgb->helpers->getApiRequest(
            $api_attributes['request'],
            $api_attributes['method'],
            $api_attributes['headers']
        );

        if ($response['http_code'] != 200) {
            $rdgb->helpers->getApiError($response['http_code'], [
                'block' => $block_name,
                'http_code' => $response['http_code'],
                'api_attributes' => $api_attributes
            ]);

            echo 'RDG Blocks: Could not update cache.';

            wp_die();
        }

        $response_json = $rdgb->helpers->isJson($response['data']);
        if ($response_json == null) {
            $rdgb->helpers->getApiError('json_response', [
                'block' => $block_name,
                'code' => 'json_response',
                'api_attributes' => $api_attributes
            ]);

            echo 'RDG Blocks: Could not update cache.';

            wp_die();
        }

        if ($block->isPreSaved()) {
            $attributes['cache_name'] = $cache_name;
            $attributes['block'] = $block;
            $response_json = $block->getPreSavedData($response_json, $attributes);
        }
        
        $updated = $rdgb->cache->updateCacheBlock($cache_name, $attributes, $response_json);
        if ($updated) {
            $tems = explode(':', $cache_name);
            $rdgb->helpers->getApiError('update_cache', [
                'page_type' => $tems[1],
                'post_id' => $tems[2],
                'block_name' => $block_name,
            ]);

            print_r($response_json);
        } else{
            echo 'RDG Blocks: Could not update cache.';
        }
    }   

    wp_die();
}
