<?php

namespace Rdgcorp\Blocks\TeamSchedules\Utils;

require __DIR__ . '/../MasterFeed/vendor/autoload.php';
require __DIR__ . '/mappers/SchedulesMapperInterface.php';
require __DIR__ . '/mappers/AbstractScheduleMapper.php';
require __DIR__ . '/mappers/DefaultSchedulesMapper.php';
require __DIR__ . '/mappers/FootballSchedulesMapper.php';
require __DIR__ . '/mappers/SoccerSchedulesMapper.php';

use Rdgcorp\Blocks\TeamSchedules\Utils\Mappers\DefaultSchedulesMapper;
use Rdgcorp\Blocks\TeamSchedules\Utils\Mappers\FootballSchedulesMapper;
use Rdgcorp\Blocks\TeamSchedules\Utils\Mappers\SchedulesMapperInterface;
use Rdgcorp\Blocks\TeamSchedules\Utils\Mappers\SoccerSchedulesMapper;
use Rdgcorp\MasterFeed\MasterFeed;

class MasterFeedApiConsumer
{
    private MasterFeed $masterFeed;

    public function __construct(string $token)
    {
        $this->masterFeed = new MasterFeed($token);
    }

    public function getSports(): array
    {
        $data = [];

        foreach ($this->masterFeed->getDirectSports() as $sport) {
            $data[$sport['id']] = $sport['name'];
        }

        return $data;
    }

    public function getLeagues(?string $sportId = null): array
    {
        $data = [];

        foreach ($this->masterFeed->getDirectLeagues($sportId) as $league) {
            $data[$league['id']] = $league['name'];
        }

        return $data;
    }

    public function getTeams(?string $leagueId = null): array
    {
        $data = [];

        foreach ($this->masterFeed->getDirectTeams($leagueId) as $team) {
            $data[$team['id']] = $team['name'];

            if (!ctype_upper($team['nickname']) && !str_contains(strtolower($team['name']), strtolower($team['nickname']))) {
                $data[$team['id']] .= ' ' . $team['nickname'];
            }
        }

        return $data;
    }

    public function getSeasons(?string $leagueId = null): array
    {
        $data = [];

        foreach ($this->masterFeed->getDirectSeasons($leagueId) as $season) {
            $data[$season['id']] = $season['name'];
        }

        return $data;
    }

    public function getSchedules(?string $teamId = null, ?string $seasonId = null): ?array
    {
        $mapperClass = $this->getMapper($teamId);

        if (!$mapperClass) {
            return null;
        }

        $teams = $this->getTeams();
        $rawSchedules = $this->masterFeed->getDirectSchedules($teamId, $seasonId);

        /** @var SchedulesMapperInterface $mapper */
        $mapper = new $mapperClass($teams);

        return $mapper->getMappedData($teamId, $rawSchedules);
    }

    public function hasMapper(string $teamId): bool
    {
        if ($this->getMapper($teamId)) {
            return true;
        }

        return false;
    }

    private function getMapper(string $teamId): ?string
    {
        $data = explode('/', $teamId);

        switch ($data[2]) {
            case 'football':
                return FootballSchedulesMapper::class;
            case 'baseball':
                return DefaultSchedulesMapper::class;
            case 'basketball':
                return DefaultSchedulesMapper::class;
            case 'hockey':
                return DefaultSchedulesMapper::class;
            case 'soccer':
                return SoccerSchedulesMapper::class;
            case 'lacrosse':
                return DefaultSchedulesMapper::class;
            default:
                return null;
        }
    }
}
