<?php

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * HELPER COMMENT START
 * 
 * This class is used to bring your plugin to life. 
 * All the other registered classed bring features which are
 * controlled and managed by this class.
 * 
 * Within the add_hooks() function, you can register all of 
 * your WordPress related actions and filters as followed:
 * 
 * add_action( 'my_action_hook_to_call', array( $this, 'the_action_hook_callback', 10, 1 ) );
 * or
 * add_filter( 'my_filter_hook_to_call', array( $this, 'the_filter_hook_callback', 10, 1 ) );
 * or
 * add_shortcode( 'my_shortcode_tag', array( $this, 'the_shortcode_callback', 10 ) );
 * 
 * Once added, you can create the callback function, within this class, as followed: 
 * 
 * public function the_action_hook_callback( $some_variable ){}
 * or
 * public function the_filter_hook_callback( $some_variable ){}
 * or
 * public function the_shortcode_callback( $attributes = array(), $content = '' ){}
 * 
 * 
 * HELPER COMMENT END
 */

/**
 * Class Sb_Sportsfeed_Run
 *
 * Thats where we bring the plugin to life
 *
 * @package		SBSPORTSFE
 * @subpackage	Classes/Sb_Sportsfeed_Run
 * @author		IT
 * @since		1.0.0
 */
class Sb_Sportsfeed_Run{

	/**
	 * Our Sb_Sportsfeed_Run constructor 
	 * to run the plugin logic.
	 *
	 * @since 1.0.0
	 */
	function __construct(){
		$this->add_hooks();
	}

	/**
	 * ######################
	 * ###
	 * #### WORDPRESS HOOKS
	 * ###
	 * ######################
	 */

	/**
	 * Registers all WordPress and plugin related hooks
	 *
	 * @access	private
	 * @since	1.0.0
	 * @return	void
	 */
	private function add_hooks() {
		//add_action( 'plugin_action_links_' . SBSPORTSFE_PLUGIN_BASE, array( $this, 'add_plugin_action_link' ), 20 );
		add_action('acf/init', array( $this, 'set_taxonomy_post'));
		add_shortcode( 'my_shortcode_tag', array( $this, 'add_shortcode_callback' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_backend_scripts_and_styles' ), 20 );
		add_action( 'wp_dashboard_setup', array( $this, 'manage_dashboard_widgets' ), 20 );
		add_action( 'acf/init', array( $this, 'add_custom_acf_options_page' ), 20 );
		add_action( 'acf/init', array( $this, 'generate_field_groups' ) );
		add_action( 'plugins_loaded', array( $this, 'add_wp_webhooks_integrations' ), 9 );
		add_filter('cron_schedules', array( $this,'sbf_feeds_add_cron_interval'));
		add_action('sbf_feeds_cron_event', array($this, 'sbf_feeds_cron_function'));
		add_action('sbf_feeds_cron_event_shortpixel', array($this, 'sbf_feeds_cron_function_shortpixel'));

        add_action('acf/save_post', [$this, 'save_post_log_maintenance']);
        add_filter('acf/prepare_field/name=number_of_logs', [$this, 'load_field_logs_maintenance'], 10, 3);
        add_filter('acf/load_value/name=delete_logs', [$this, 'load_field_delete_log'], 10, 3);

		register_activation_hook( SBSPORTSFE_PLUGIN_FILE, array( $this, 'activation_hook_callback' ) );
		register_deactivation_hook( SBSPORTSFE_PLUGIN_FILE, array( $this, 'deactivation_hook_callback' ) );
		add_action('acf/save_post', [$this, 'save_post_sb_sport_feed']);
		add_action('acf/save_post', [$this, 'save_post_sbf_feeds_shortpixel']);
		add_filter('acf/load_value/name=reset_last_run', [$this, 'load_field_reset_last_run'], 10, 3);
		add_action('acf/init', array($this, 'generate_field_sports_feed'));
		add_action('acf/init', array($this, 'generate_field_sports_feed_images'));
		add_action('acf/init', array($this, 'generate_field_sports_feed_logs'), 20);
		add_action('acf/init', array($this, 'generate_field_sports_feed_minimum_content_words'), 20);
		add_filter('acf/load_value/name=short_pixel_last_post_optimized_id', [$this, 'load_field_last_post_optimized'], 10, 3);
	}

	public function load_field_last_post_optimized($value) {
		if (!empty(get_option('sb_feeds_last_post_optimized'))) {
			return get_option('sb_feeds_last_post_optimized');
		}

		return $value;
	}

	public function load_field_reset_last_run($value) {
		if ($value == 'reset') {
			update_field('reset_last_run', 'default');
		}

		return 'default';
	}

	public function save_post_sb_sport_feed($post_id) {
		$option = (isset($_GET['page'])) ? $_GET['page'] : null;

		if (!empty($option) && $option == 'sb-sports-feed-options' && isset($_POST['acf']) && count($_POST['acf']) == 8) {
			$fields = [];
			$categories = [];
			$select_sbf_cron_time = current($_POST['acf']);
			$url_request = next($_POST['acf']);
			$post_type = next($_POST['acf']);
			$txt_allowed_entry_types = next($_POST['acf']);
			$txt_allowed_entry_sports = next($_POST['acf']);
			$txt_allowed_entry_leagues = next($_POST['acf']);
			$txt_posts_username_tem = next($_POST['acf']);
			$btn_reset_last_run = next($_POST['acf']);
	
			if (is_array($txt_posts_username_tem) && count($txt_posts_username_tem) > 0) {
				foreach ($txt_posts_username_tem as $key => $value) {
					$tem = array_values($value);
					$category = (!empty($tem[0]) && trim($tem[0]) == '*') ? '*' : sanitize_title(trim($tem[0]));
					$author = (!empty(sanitize_title(trim($tem[1])))) ? sanitize_title(trim($tem[1])) : 1;
					$categories[$category] = $author;
				}
			}

			$txt_posts_username = $categories;
	
			if (empty($select_sbf_cron_time) || empty($txt_allowed_entry_types) || empty($txt_allowed_entry_sports) || empty($txt_allowed_entry_leagues) || empty($txt_posts_username)) {
				error_log(print_r([
					'message' => 'You must enter all the requested fields!',
					'fields' => [
						'select_sbf_cron_time',
						'txt_allowed_entry_types',
						'txt_allowed_entry_sports',
						'txt_allowed_entry_leagues',
						'txt_posts_username',
					]
				], true));
	
				wp_die('You must enter all the requested fields!');
			} else {
				$original_cron = get_option( 'sb_feeds_cron_time');

				if ((!empty($select_sbf_cron_time)) && ($original_cron != $select_sbf_cron_time)) {
					wp_clear_scheduled_hook('sbf_feeds_cron_event');
					wp_schedule_event(time(), $select_sbf_cron_time, 'sbf_feeds_cron_event');
				}

				// Validate the input data
				if ( $txt_allowed_entry_types !== null || $txt_allowed_entry_sports !== null || $txt_allowed_entry_leagues !== null || $txt_posts_username !== null || $select_sbf_cron_time !== null) {
					// Save the settings to the database
					update_option('sb_feeds_allowed_entry_types', $txt_allowed_entry_types );
					update_option('sb_feeds_allowed_entry_sports', $txt_allowed_entry_sports );
					update_option('sb_feeds_allowed_entry_leagues', $txt_allowed_entry_leagues );
					update_option('sb_feeds_author', $txt_posts_username );
					update_option('sb_feeds_cron_time', $select_sbf_cron_time );
					update_option('sb_feed_url_request', $url_request);
					update_option('sb_feed_post_type', $post_type);
				}  

				if (!empty($btn_reset_last_run) && $btn_reset_last_run == 'reset') {
					delete_option('sb_feeds_last_run');
				}
			}
		}
	}

    public function load_field_logs_maintenance($field) {
        $logs = scandir(SBSPORTSFE_PLUGIN_LOG_DIR);

        $count_files = (is_array($logs)) ? count($logs) - 2 : 0; 

        $field['message'] = $count_files.' Files';

        return $field;
    }

    public function load_field_delete_log($value) {
        return 'default';
    }

    public function save_post_log_maintenance($post_id) {
        $option = (isset($_GET['page'])) ? $_GET['page'] : null;

        if (!empty($option) && $option == 'acf-options-log-maintenance' && isset($_POST['acf']) && is_array($_POST['acf']) && count($_POST['acf']) > 0) {
			$fields = [];
            
            foreach ($_POST['acf'] as $key => $value) {
				$fields[] = $value;
            }

            try {
                $logs = scandir(SBSPORTSFE_PLUGIN_LOG_DIR);
                
                if ($fields[1] == 'deletelogs') {
                    if (file_exists(SBSPORTSFE_PLUGIN_LOG_DIR)) {
                        if (is_array($logs) && count($logs) > 0) {
                            foreach ($logs as $log) {
								if ($log != '.' && $log != '..') {
									unlink(SBSPORTSFE_PLUGIN_LOG_DIR . '/' . $log);
								}
                            }

                            error_log('SB SportsFeed: All logs deleted from file.');
                            update_user_option(get_current_user_id(), 'show_message', 'All logs deleted from file.');
                            update_user_option(get_current_user_id(), 'message_type', 'notice-success');
                        } else {
                            error_log('RDG Blocks: Log file is empty.');
                            update_user_option(get_current_user_id(), 'show_message', 'Log file is empty.');
                            update_user_option(get_current_user_id(), 'message_type', 'notice-warning');
                        }
                    }
                }

				$helpers = New Sb_Sportsfeed_Helpers;
				$cron_manager_log = ($helpers->is_install_cron_manager() && isset($fields[2]) && $fields[2] == 'save') ? true : false;
				update_option('sb_feeds_cron_manager_log', $cron_manager_log);
            } catch (\Exception $exception) {
                update_user_option(get_current_user_id(), 'show_message', 'An unexpected error occurred. Try again later.');
                update_user_option(get_current_user_id(), 'message_type', 'notice-error');

                error_log(print_r([
                    'message' => $exception->getMessage(),
                    'fields' => $fields,
                ], true));
            }
        }
    }

	/**
	 * ######################
	 * ###
	 * #### WORDPRESS HOOK CALLBACKS
	 * ###
	 * ######################
	 */

	/**
	* Adds action links to the plugin list table
	*
	* @access	public
	* @since	1.0.0
	*
	* @param	array	$links An array of plugin action links.
	*
	* @return	array	An array of plugin action links.
	*/
	public function add_plugin_action_link( $links ) {

		$links['our_shop'] = sprintf( '<a href="%s" title="Custom Link" style="font-weight:700;">%s</a>', 'https://test.test', __( 'Custom Link', 'sb-sportsfeed' ) );

		return $links;
	}

	/**
	 * Add the shortcode callback for [my_shortcode_tag]
	 *
	 * @access	public
	 * @since	1.0.0
	 *
	 * @param	array	$attr		Additional attributes you have added within the shortcode tag.
	 * @param	string	$content	The content you added between an opening and closing shortcode tag.
	 *
	 * @return	string	The customized content by the shortcode.
	 */
	public function add_shortcode_callback( $attr = array(), $content = '' ) {

		$content .= ' this content is added by the add_shortcode_callback() function';

		return $content;
	}

	/**
	 * Enqueue the backend related scripts and styles for this plugin.
	 * All of the added scripts andstyles will be available on every page within the backend.
	 *
	 * @access	public
	 * @since	1.0.0
	 *
	 * @return	void
	 */
	public function enqueue_backend_scripts_and_styles() {
		wp_enqueue_style( 'sbsportsfe-backend-styles', SBSPORTSFE_PLUGIN_URL . 'core/includes/assets/css/backend-styles.css', array(), SBSPORTSFE_VERSION, 'all' );
	}

	/**
	 * Add custom menu page content for the following
	 * menu item: custom-menu-slug
	 *
	 * @access	public
	 * @since	1.0.0
	 *
	 * @return	void
	 */
	public function custom_admin_menu_page_callback(){
		//
	}

	/**
	 * Adds all plugin related dashbaord widgets
	 *
	 * @access	public
	 * @since	1.0.0
	 *
	 * @return	void
	 */
	public function manage_dashboard_widgets() {

		wp_add_dashboard_widget( 'demo_widget', __( 'Demo Widget', 'sb-sportsfeed' ), array( $this, 'demo_widget_callback' ), null, array() );

	}

	/**
	 * The callback for the "Demo Widget" widget
	 *
	 * @access	public
	 * @since	1.0.0
	 *
	 * @param	object	$object	Often this is the object that's the focus of the current screen, for example a WP_Post or WP_Comment object.
	 * @param	array	$args	Data that should be set as the $args property of the widget array (which is the fifth parameter passed to the wp_add_dashboard_widget function call)
	 *
	 * @return	void
	 */
	public function demo_widget_callback( $object, $args ){
		?>
<p>This is some custom demo text.</p>
		<?php
	}

	/**
	 * Add the Advanced Custom fields
	 * options pages
	 *
	 * @access	public
	 * @since	1.0.0
	 * @link	https://www.advancedcustomfields.com/resources/acf_add_options_page/
	 *
	 * @return	void
	 */
	public function add_custom_acf_options_page() {

		// Check function exists.
		if( function_exists('acf_add_options_page') ) {
			// Register options page.
			$option_page = acf_add_options_page( array(
				'page_title'    => __('SB SportsFeed'),
				'menu_title'    => __('SB SportsFeed Settings'),
				'menu_slug'     => 'sb-sports-feed-options',
				'capability'    => 'edit_posts',
				'icon_url' 		=> 'dashicons-rest-api',
				'redirect'      => false,
				'position' 		=> 5
			));

			acf_add_options_page(array(
				'page_title'  => __('Image Optimization'),
				'menu_title'  => __('Image Optimization'),
				'parent_slug' => 'sb-sports-feed-options',
			));

            acf_add_options_page(array(
				'page_title'  => __('Log Maintenance'),
				'menu_title'  => __('Log Maintenance'),
				'parent_slug' => 'sb-sports-feed-options',
			));
            
            acf_add_options_page(array(
				'page_title'  => __('Minimum Content Words Settings'),
				'menu_title'  => __('Minimum Content Words'),
				'parent_slug' => 'sb-sports-feed-options',
			));
		}
	}

	/**
	 * ####################
	 * ### WP Webhooks 
	 * ####################
	 */

	/*
	 * Register dynamically all integrations
	 * The integrations are available within core/includes/integrations.
	 * A new folder is considered a new integration.
	 *
	 * @access	public
	 * @since	1.0.0
	 *
	 * @return	void
	 */
	public function add_wp_webhooks_integrations(){

		// Abort if WP Webhooks is not active
		if( ! function_exists('WPWHPRO') ){
			return;
		}

		$custom_integrations = array();
		$folder = SBSPORTSFE_PLUGIN_DIR . 'core' . DIRECTORY_SEPARATOR . 'includes' . DIRECTORY_SEPARATOR . 'integrations';

		try {
			$custom_integrations = WPWHPRO()->helpers->get_folders( $folder );
		} catch ( Exception $e ) {
			WPWHPRO()->helpers->log_issue( $e->getTraceAsString() );
		}

		if( ! empty( $custom_integrations ) ){
			foreach( $custom_integrations as $integration ){
				$file_path = $folder . DIRECTORY_SEPARATOR . $integration . DIRECTORY_SEPARATOR . $integration . '.php';
				WPWHPRO()->integrations->register_integration( array(
					'slug' => $integration,
					'path' => $file_path,
				) );
			}
		}
	}

	/**
	 * ####################
	 * ### Activation/Deactivation hooks
	 * ####################
	 */
	 
	/*
	 * This function is called on activation of the plugin
	 *
	 * @access	public
	 * @since	1.0.0
	 *
	 * @return	void
	 */
	public function activation_hook_callback(){
		//Create cronjob
		if (!wp_next_scheduled('sbf_feeds_cron_event')) {
			wp_schedule_event(time(), 'hourly', 'sbf_feeds_cron_event');
			update_option( 'sb_feeds_cron_time', 'hourly' );
		}

		//Create cronjob shortpixel
        if (!wp_next_scheduled('sbf_feeds_cron_event_shortpixel')) {
            wp_schedule_event(time(), 'hourly', 'sbf_feeds_cron_event_shortpixel');
            update_option('sbf_feeds_cron_time_shortpixel', 'hourly');
        }
	}

	public function deactivation_hook_callback(){
		wp_clear_scheduled_hook('sbf_feeds_cron_event');
		wp_clear_scheduled_hook('sbf_feeds_cron_event_shortpixel');
	}

	public function sbf_feeds_cron_function() {
		// Call the WP-CLI command
		//error_log("Executing cronjob... ");
		$command = 'wp sb_sportsfeed pull_feed_data';
		$output = shell_exec($command);

		$helpers = New Sb_Sportsfeed_Helpers;
		if (get_option('sb_feeds_cron_manager_log') && $helpers->is_install_cron_manager()) {
			$vars = $helpers->get_cron_vars($output);

			if ($vars !== null) {
				add_action('advanced-cron-manager/log', function($logger) use ($vars) {
					$logger->log($vars);
				});
			}
		}
	}

	public function sbf_feeds_cron_function_shortpixel() {
        // Call the WP-CLI command
        $command = 'wp sb_sportsfeed optimize_post_images check_active';
        $output = shell_exec($command);
    }
	
	public function sbf_feeds_add_cron_interval($schedules) {
		if(!array_key_exists("every_five_minutes", $schedules)) {
            $schedules['every_five_minutes'] = array(
                'interval' => 60 * 5,
                'display'  => __('Every Five Minutes'),
            );
        }

		if(!array_key_exists("quarter_hour", $schedules)) {
            $schedules['quarter_hour'] = array(
                'interval' => 60 * 15,
                'display'  => __('Every 15 Minutes'),
            );
        }

		if(!array_key_exists("half_hour", $schedules)) {
            $schedules['half_hour'] = array(
                'interval' => 60 * 30,
                'display'  => __('Every 30 Minutes'),
            );
        }

		if(!array_key_exists("hourly", $schedules)) {
            $schedules['hourly'] = array(
                'interval' => 60 * 60,
                'display'  => __('Every Hour'),
            );
        }

		if(!array_key_exists("every_six_hour", $schedules)) {
            $schedules['every_six_hour'] = array(
                'interval' => 60 * 60 * 6,
                'display'  => __('Every 6 Hour'),
            );
        }

        if(!array_key_exists("daily", $schedules)) {
            $schedules['daily'] = array(
                'interval' => 60 * 60 * 24,
                'display'  => __('Every Day'),
            );
        }

		return $schedules;
	}

	public function generate_field_groups() {
		$post_type = get_option('sb_feed_post_type', 'post');
		$post_type_taxonomy = get_option('sb_feed_post_type_taxonomy', 'category');

		//SBF Post Fields
		if( function_exists('acf_add_local_field_group') ):
			acf_add_local_field_group(array(
				'key' => 'group_6406f02fb6ca4',
				'title' => 'SBF Automatic Posts',
				'fields' => array(
					array(
						'key' => 'field_6406f02f3ebd2',
						'label' => 'Type',
						'name' => 'sbf_type',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_642261cecd1cb',
						'label' => 'External ID',
						'name' => 'sbf_external_id',
						'aria-label' => '',
						'type' => 'number',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'min' => '',
						'max' => '',
						'placeholder' => '',
						'step' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6406f0813ebd3',
						'label' => 'Sport ID',
						'name' => 'sbf_sport_id',
						'aria-label' => '',
						'type' => 'number',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'min' => '',
						'max' => '',
						'placeholder' => '',
						'step' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6406f1363ebd9',
						'label' => 'Sport Name',
						'name' => 'sbf_sport_name',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6406f0923ebd4',
						'label' => 'League ID',
						'name' => 'sbf_league_id',
						'aria-label' => '',
						'type' => 'number',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'min' => '',
						'max' => '',
						'placeholder' => '',
						'step' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6406f1483ebda',
						'label' => 'League Name',
						'name' => 'sbf_league_name',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_640850854a72c',
						'label' => 'League Nickname',
						'name' => 'sbf_league_nickname',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6406f0a23ebd5',
						'label' => 'Home Team ID',
						'name' => 'sbf_home_team_id',
						'aria-label' => '',
						'type' => 'number',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'min' => '',
						'max' => '',
						'placeholder' => '',
						'step' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6406f0cc3ebd7',
						'label' => 'Home Team Name',
						'name' => 'sbf_home_team_name',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6406f0bc3ebd6',
						'label' => 'Visitor Team ID',
						'name' => 'sbf_visitor_team_id',
						'aria-label' => '',
						'type' => 'number',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'min' => '',
						'max' => '',
						'placeholder' => '',
						'step' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6406f1073ebd8',
						'label' => 'Visitor Team Name',
						'name' => 'sbf_visitor_team_name',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6406f17c3ebdb',
						'label' => 'Featured Image URL',
						'name' => 'sbf_featured_image_url',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6407f3de2b193',
						'label' => 'Image Headline',
						'name' => 'sbf_image_headline',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_6407f40e2b194',
						'label' => 'Image Credit',
						'name' => 'sbf_image_credit',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
				),
				'location' => array(
					array(
						array(
							'param' => 'post_type',
							'operator' => '==',
							'value' => $post_type,
						),
					),
				),
				'menu_order' => 0,
				'position' => 'normal',
				'style' => 'default',
				'label_placement' => 'top',
				'instruction_placement' => 'label',
				'hide_on_screen' => '',
				'active' => true,
				'description' => '',
				'show_in_rest' => 0,
			));

			endif;		

			//SBF Categories
			if( function_exists('acf_add_local_field_group') ):

				acf_add_local_field_group(array(
					'key' => 'group_6422620ada123',
					'title' => 'SBF Categories',
					'fields' => array(
						array(
							'key' => 'field_6422620a4a276',
							'label' => 'Category Type',
							'name' => 'sbf_category_type',
							'aria-label' => '',
							'type' => 'text',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'maxlength' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => '',
						),
					),
					'location' => array(
						array(
							array(
								'param' => 'taxonomy',
								'operator' => '==',
								'value' => $post_type_taxonomy,
							),
						),
					),
					'menu_order' => 0,
					'position' => 'normal',
					'style' => 'default',
					'label_placement' => 'top',
					'instruction_placement' => 'label',
					'hide_on_screen' => '',
					'active' => true,
					'description' => '',
					'show_in_rest' => 0,
				));
				
				endif;		
	}

	public function generate_field_sports_feed(){
		//SBF Post Fields
		if( function_exists('acf_add_local_field_group') ):
			acf_add_local_field_group(array(
				'key' => 'group_64etf34r34rd3r34r34r',
				'title' => 'SB SportsFeed Settings',
				'fields' => array(
					array(
						'key' => 'field_646f7a0a36940',
						'label' => 'Last Feed Pull Datetime',
						'name' => '',
						'aria-label' => '',
						'type' => 'message',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '50',
							'class' => '',
							'id' => '',
						),
						'message' => get_option('sb_feeds_last_run', 'never'),
						'new_lines' => '',
						'esc_html' => 0,
					),
					array(
						'key' => 'field_646e9a1f84fe1',
						'label' => 'Cronjob run time',
						'name' => 'sbf_cron_time',
						'aria-label' => '',
						'type' => 'select',
						'instructions' => '',
						'required' => 1,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '50',
							'class' => '',
							'id' => '',
						),
						'choices' => array(
							'every_five_minutes' => 'Every 5 Minutes',
							'hourly' => 'Once Hourly',
							'twicedaily' => 'Twice Daily',
							'daily' => 'Once Daily',
							'weekly' => 'Once Weekly',
						),
						'default_value' => 'hourly',
						'return_format' => 'value',
						'multiple' => 0,
						'allow_null' => 0,
						'ui' => 0,
						'ajax' => 0,
						'placeholder' => '',
					),
					array(
						'key' => 'field_64ef64609cee3',
						'label' => 'URL Request',
						'name' => 'sb_feed_url_request',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '65',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_64ef641c9cee2',
						'label' => 'Create as Post Type',
						'name' => 'create_as_post_type',
						'aria-label' => '',
						'type' => 'select',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '35',
							'class' => '',
							'id' => '',
						),
						'choices' => $this->get_post_types_clear(),
						'default_value' => 'post',
						'return_format' => 'value',
						'multiple' => 0,
						'allow_null' => 1,
						'ui' => 0,
						'ajax' => 0,
						'placeholder' => '',
					),
					array(
						'key' => 'field_646e9ae584fe2',
						'label' => 'Name of entry type names to process',
						'name' => 'txt_allowed_entry_types',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => 'Sepparated by comma',
						'required' => 1,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '33',
							'class' => '',
							'id' => 'txt_allowed_entry_types',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_646e9b7b84fe3',
						'label' => 'IDs of sports to process',
						'name' => 'txt_allowed_entry_sports',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => 'Sepparated by comma',
						'required' => 1,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '34',
							'class' => '',
							'id' => 'txt_allowed_entry_sports',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_646e9bc284fe4',
						'label' => 'IDs of leagues to process',
						'name' => 'txt_allowed_entry_leagues',
						'aria-label' => '',
						'type' => 'text',
						'instructions' => 'Sepparated by comma',
						'required' => 1,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '33',
							'class' => '',
							'id' => 'txt_allowed_entry_leagues',
						),
						'default_value' => '',
						'maxlength' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
					),
					array(
						'key' => 'field_646cfeef60fcf',
						'label' => 'Category Authors',
						'name' => 'sb-sports-feed-authors',
						'aria-label' => '',
						'type' => 'repeater',
						'instructions' => '',
						'required' => 1,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'layout' => 'block',
						'pagination' => 0,
						'min' => 0,
						'max' => 0,
						'collapsed' => '',
						'button_label' => 'Add Row',
						'rows_per_page' => 20,
						'sub_fields' => array(
							array(
								'key' => 'field_646cff2460fd1',
								'label' => 'Category',
								'name' => 'sb-sports-feed-category',
								'aria-label' => '',
								'type' => 'text',
								'instructions' => '',
								'required' => 1,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '50',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'maxlength' => '',
								'placeholder' => '',
								'prepend' => '',
								'append' => '',
								'parent_repeater' => 'field_646cfeef60fcf',
							),
							array(
								'key' => 'field_646cff1160fd0',
								'label' => 'Author',
								'name' => 'sb-sports-feed-user',
								'aria-label' => '',
								'type' => 'text',
								'instructions' => '',
								'required' => 1,
								'conditional_logic' => 0,
								'wrapper' => array(
									'width' => '50',
									'class' => '',
									'id' => '',
								),
								'default_value' => '',
								'maxlength' => '',
								'placeholder' => '',
								'prepend' => '',
								'append' => '',
								'parent_repeater' => 'field_646cfeef60fcf',
							),
						),
					),
					array(
						'key' => 'field_646ec0e1bb72a',
						'label' => 'Reset Last Run',
						'name' => 'reset_last_run',
						'aria-label' => '',
						'type' => 'radio',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'choices' => array(
							'default' => 'Default',
							'reset' => 'Reset',
						),
						'default_value' => '',
						'return_format' => 'value',
						'allow_null' => 0,
						'other_choice' => 0,
						'layout' => 'horizontal',
						'save_other_choice' => 0,
					),
				),
				'location' => array(
					array(
						array(
							'param' => 'options_page',
							'operator' => '==',
							'value' => 'sb-sports-feed-options',
						),
					),
				),
				'menu_order' => 0,
				'position' => 'normal',
				'style' => 'default',
				'label_placement' => 'top',
				'instruction_placement' => 'label',
				'hide_on_screen' => '',
				'active' => true,
				'description' => '',
				'show_in_rest' => 0,
			));

			endif;
	}

	public function get_post_types_clear() {
		$args = array(
			'public' => true,
		);
		
		$all_post_types = get_post_types($args, 'objects');
		$post_types_array = array();

		foreach ($all_post_types as $post_type_slug => $post_type_obj) {
			$post_types_array[$post_type_slug] = $post_type_obj->labels->name;
		}
		
		return $post_types_array;
	}

	public function generate_field_sports_feed_images(){
		//SBF Post Fields
		if( function_exists('acf_add_local_field_group') ):
			acf_add_local_field_group(array(
				'key' => 'group_6wdfwwwewerrewr',
				'title' => 'SB SportsFeed Settings',
				'fields' => array(
					array(
						'key' => 'field_649ad571f526e',
						'label' => 'Short Pixel Active?',
						'name' => 'short_pixel_active',
						'aria-label' => '',
						'type' => 'radio',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '20',
							'class' => '',
							'id' => '',
						),
						'choices' => array(
							'none' => 'None',
							'active' => 'Active',
						),
						'default_value' => 'none',
						'return_format' => 'value',
						'allow_null' => 0,
						'other_choice' => 0,
						'layout' => 'horizontal',
						'save_other_choice' => 0,
					),
					array(
						'key' => 'field_649ad5c1f526f',
						'label' => 'Max Chunks',
						'name' => 'max_chunks',
						'aria-label' => '',
						'type' => 'select',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_649ad571f526e',
									'operator' => '==',
									'value' => 'active',
								),
							),
						),
						'wrapper' => array(
							'width' => '30',
							'class' => '',
							'id' => '',
						),
						'choices' => array(
							10 => '10 Images',
							20 => '20 Images',
							30 => '30 Images',
							40 => '40 Images',
							50 => '50 Images',
							100 => '100 Images',
						),
						'default_value' => 10,
						'return_format' => 'value',
						'multiple' => 0,
						'allow_null' => 0,
						'ui' => 0,
						'ajax' => 0,
						'placeholder' => '',
					),
					array(
						'key' => 'field_649ad62df5270',
						'label' => 'Cronjob run time',
						'name' => 'short_pixel_cronjob_run_time',
						'aria-label' => '',
						'type' => 'select',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_649ad571f526e',
									'operator' => '==',
									'value' => 'active',
								),
							),
						),
						'wrapper' => array(
							'width' => '30',
							'class' => '',
							'id' => '',
						),
						'choices' => array(
							'every_five_minutes' => 'Every 5 Minutes',
							'quarter_hour' => 'Every 15 Minutes',
							'half_hour' => 'Every 30 Minutes',
							'hourly' => 'Once Hourly',
							'every_six_hour' => 'Every 6 Hours',
							'daily' => 'Once Daily',
						),
						'default_value' => false,
						'return_format' => 'value',
						'multiple' => 0,
						'allow_null' => 0,
						'ui' => 0,
						'ajax' => 0,
						'placeholder' => '',
					),
					array(
						'key' => 'field_64ae9535eb6a3',
						'label' => 'Last Post Optimized ID',
						'name' => 'short_pixel_last_post_optimized_id',
						'aria-label' => '',
						'type' => 'number',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_649ad571f526e',
									'operator' => '==',
									'value' => 'active',
								),
							),
						),
						'wrapper' => array(
							'width' => '20',
							'class' => '',
							'id' => '',
						),
						'default_value' => 0,
						'min' => 0,
						'max' => '',
						'placeholder' => '',
						'step' => '',
						'prepend' => '',
						'append' => '',
					),
				),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-image-optimization',
                        ),
                    ),
                ),
				'menu_order' => 0,
				'position' => 'normal',
				'style' => 'default',
				'label_placement' => 'top',
				'instruction_placement' => 'label',
				'hide_on_screen' => '',
				'active' => true,
				'description' => '',
				'show_in_rest' => 0,
			));

			endif;			
	}

    public function generate_field_sports_feed_logs() {
            if ( ! function_exists( 'acf_add_local_field_group' ) ) {
                return;
            }

			$helpers = New Sb_Sportsfeed_Helpers;
			
			$width_default = ($helpers->is_install_cron_manager()) ? '40' : '25';
			if ($helpers->is_install_cron_manager()) {
				$width_default = '25';
			}

			$fields = [
				[
                    'key' => 'field_64d3ead891acb',
                    'label' => 'Number of Logs',
                    'name' => 'number_of_logs',
                    'aria-label' => '',
                    'type' => 'message',
                    'instructions' => '',
                    'required' => 0,
                    'conditional_logic' => 0,
                    'wrapper' => array(
                        'width' => '20',
                        'class' => '',
                        'id' => '',
                    ),
                    'message' => '200 Files',
                    'new_lines' => 'wpautop',
                    'esc_html' => 0,
                ],
                [
                    'key' => 'field_64d3ead895534',
                    'label' => 'Save Logs',
                    'name' => 'logs',
                    'aria-label' => '',
                    'type' => 'radio',
                    'instructions' => '',
                    'required' => 0,
                    'conditional_logic' => 0,
                    'wrapper' => array(
                        'width' => $width_default,
                        'class' => '',
                        'id' => '',
                    ),
                    'choices' => array(
                        'none' => 'Don\'t Save Logs',
                        'save' => 'Save Logs',
                    ),
                    'default_value' => 'none',
                    'return_format' => 'value',
                    'allow_null' => 0,
                    'other_choice' => 0,
                    'layout' => 'horizontal',
                    'save_other_choice' => 0,
                ],
                [
                    'key' => 'field_64d52e4d27722',
                    'label' => 'Delete Logs',
                    'name' => 'delete_logs',
                    'aria-label' => '',
                    'type' => 'radio',
                    'instructions' => '',
                    'required' => 0,
                    'conditional_logic' => 0,
                    'wrapper' => array(
                        'width' => $width_default,
                        'class' => '',
                        'id' => '',
                    ),
                    'choices' => array(
                        'default' => 'Default',
                        'deletelogs' => 'Delete All Logs',
                    ),
                    'default_value' => 'default',
                    'return_format' => 'value',
                    'allow_null' => 0,
                    'other_choice' => 0,
                    'layout' => 'horizontal',
                    'save_other_choice' => 0,
				]
			];

			if ($helpers->is_install_cron_manager()) {
				$fields[] = [
					'key' => 'field_64d52gytuu6t7g',
					'label' => 'Cron Manager Logs',
					'name' => 'cron_manager_log',
					'aria-label' => '',
					'type' => 'radio',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '30',
						'class' => '',
						'id' => '',
					),
					'choices' => array(
						'none' => 'Don\'t Create Logs',
						'save' => 'Create Logs',
					),
					'default_value' => 'none',
					'return_format' => 'value',
					'allow_null' => 0,
					'other_choice' => 0,
					'layout' => 'horizontal',
					'save_other_choice' => 0,
				];
			}
        
            acf_add_local_field_group( array(
            'key' => 'group_64d3ead88cfe7',
            'title' => 'SB SportsFeed Settings Log',
            'fields' => $fields,
            'location' => array(
                array(
                    array(
                        'param' => 'options_page',
                        'operator' => '==',
                        'value' => 'acf-options-log-maintenance',
                    ),
                ),
            ),
            'menu_order' => 0,
            'position' => 'normal',
            'style' => 'default',
            'label_placement' => 'top',
            'instruction_placement' => 'label',
            'hide_on_screen' => '',
            'active' => true,
            'description' => '',
            'show_in_rest' => 0,
        ) );
                           
    }

    public function generate_field_sports_feed_minimum_content_words() {

        if ( ! function_exists( 'acf_add_local_field_group' ) ) {
            return;
        }            
        
        acf_add_local_field_group( array(
        'key' => 'group_650348b6938cf',
        'title' => 'Minimum Content Words Settings',
        'fields' => array(
            array(
                'key' => 'field_650348b78794d',
                'label' => 'Activate function?',
                'name' => 'activate_function',
                'aria-label' => '',
                'type' => 'radio',
                'instructions' => '',
                'required' => 0,
                'conditional_logic' => 0,
                'wrapper' => array(
                    'width' => '50',
                    'class' => '',
                    'id' => '',
                ),
                'choices' => array(
                    'on' => 'On',
                    'off' => 'Off',
                ),
                'default_value' => 'off',
                'return_format' => 'value',
                'allow_null' => 0,
                'other_choice' => 0,
                'layout' => 'vertical',
                'save_other_choice' => 0,
            ),
            array(
                'key' => 'field_65034ac18794e',
                'label' => 'Minimum Content Words',
                'name' => 'minimum_content_words',
                'aria-label' => '',
                'type' => 'number',
                'instructions' => '',
                'required' => 0,
                'conditional_logic' => 0,
                'wrapper' => array(
                    'width' => '50',
                    'class' => '',
                    'id' => '',
                ),
                'default_value' => '',
                'min' => '',
                'max' => '',
                'placeholder' => '',
                'step' => '',
                'prepend' => '',
                'append' => '',
            ),
        ),
        'location' => array(
            array(
                array(
                    'param' => 'options_page',
                    'operator' => '==',
                    'value' => 'acf-options-minimum-content-words',
                ),
            ),
        ),
        'menu_order' => 0,
        'position' => 'normal',
        'style' => 'default',
        'label_placement' => 'top',
        'instruction_placement' => 'label',
        'hide_on_screen' => '',
        'active' => true,
        'description' => '',
        'show_in_rest' => 0,
    ) );
    
    }    
    

	public function save_post_sbf_feeds_shortpixel($post_id) {
        $option = (isset($_GET['page'])) ? $_GET['page'] : null;

        if (!empty($option) && $option == 'acf-options-image-optimization' && isset($_POST['acf']) && is_array($_POST['acf']) && count($_POST['acf']) > 0) {
            try {
                foreach ($_POST['acf'] as $key => $value) {
                    $fields[] = $value;
                }

                $short_pixel_active = (isset($fields[0])) ? $fields[0] : 'none';
                $short_pixel_cronjob = (isset($fields[3])) ? $fields[3] : 'hourly';
                $last_post_optimized_id = (isset($fields[4])) ? $fields[4] : 0;

				if (!empty($last_post_optimized_id) && $last_post_optimized_id > 0) {
					update_option('sb_feeds_last_post_optimized', $last_post_optimized_id);
				} else {
					update_option('sb_feeds_last_post_optimized', 0);
				}

                if ($short_pixel_active == 'active') {
                    wp_clear_scheduled_hook('sbf_feeds_cron_event_shortpixel');
                    wp_schedule_event(time(), $short_pixel_cronjob, 'sbf_feeds_cron_event_shortpixel');
                    update_option('sbf_feeds_cron_time_shortpixel', $short_pixel_cronjob);
                } else {
                    wp_clear_scheduled_hook('sbf_feeds_cron_event_shortpixel');
                }
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => $exception->getMessage(),
                ], true));
            }
        }
    }

	public function set_taxonomy_post() {
		$post_type = get_option('sb_feed_post_type', 'post');
		$taxonomies = get_object_taxonomies(['post_type' => $post_type]);

		if (is_array($taxonomies) && count($taxonomies) > 0 && !empty($taxonomies[0])) {
			update_option('sb_feed_post_type_taxonomy', $taxonomies[0]);

			return;
		}

		$new_taxonomy = $this->create_taxonomy_post($post_type);
		if (!empty($new_taxonomy)) {
			update_option('sb_feed_post_type_taxonomy', $new_taxonomy);

			return;
		}

		update_option('sb_feed_post_type_taxonomy', 'category');
	}

	public function create_taxonomy_post($post_type) {
		$taxonomy_name = ucwords($post_type.' category');
		$slug = sanitize_title($taxonomy_name);

		$labels = [
			'name'              => __($taxonomy_name),
			'singular_name'     => __($taxonomy_name),
			'search_items'      => __('Search '.$taxonomy_name),
			'all_items'         => __('All '.$taxonomy_name),
			'parent_item'       => __('Parent '.$taxonomy_name),
			'parent_item_colon' => __('Parent '.$taxonomy_name),
			'edit_item'         => __('Edit '.$taxonomy_name),
			'update_item'       => __('Update '.$taxonomy_name),
			'add_new_item'      => __('Add '.$taxonomy_name),
			'new_item_name'     => __('New '.$taxonomy_name.' Name'),
			'menu_name'         => __($taxonomy_name)
		];
	
		$args = [
			'labels'            => $labels,
			'public'      		=> true,
			'hierarchical'      => true,
			'show_admin_column' => true,
			'show_in_rest'      => true,
			'rewrite' => [
				'slug' => $slug,
				'with_front' => true,
			],
		];

		register_taxonomy($slug, $post_type, $args);
		wp_insert_term('Uncategorized', $slug);

		return $slug;
	}
}
