<?php
// Exit if accessed directly
if (!defined('ABSPATH')) exit;

//require_once( ABSPATH . 'wp-includes/pluggable.php' );
//include_once('wp-includes\option.php');

/**
 * HELPER COMMENT START
 * 
 * This class contains all of the available CLI commands for your plugin. 
 * Down below, we added a command that allows you to display the current 
 * details about your plugin within the console. 
 * 
 * To test the command, please head over to your WordPress instance and type in the following
 * 
 * wp sb_sportsfeed details
 * 
 * HELPER COMMENT END
 */

WP_CLI::add_command('sb_sportsfeed', 'Sb_Sportsfeed_CLI');

/**
 * Class Sb_Sportsfeed_CLI
 *
 * This class contains all WP CLI related features.
 *
 * @package		SBSPORTSFE
 * @subpackage	Classes/Sb_Sportsfeed_CLI
 * @author		IT
 * @since		1.0.0
 */
class Sb_Sportsfeed_CLI extends WP_CLI_Command
{

	/**
	 * Our Sb_Sportsfeed_CLI constructor 
	 * to run the plugin CLI logic.
	 *
	 * @since 1.0.0
	 */
	function __construct()
	{
	}


	/**
	 * Get plugin details
	 *
	 * ## OPTIONS
	 *
	 * None. Returns basic info regarding the plugin instance.
	 *
	 * ## EXAMPLES
	 *
	 * To access this feature, type the following into your console:
	 * wp sb_sportsfeed details
	 *
	 * @param		array $args
	 * @param		array $assoc_args
	 * @return		void
	 */
	public function details($args, $assoc_args)
	{
		WP_CLI::line(sprintf(__('Plugin name: %s', 'SB SportsFeed'), SBSPORTSFE_NAME));
		WP_CLI::line(sprintf(__('Plugin version: %s', 'SB SportsFeed'), SBSPORTSFE_VERSION));
	}


	/**
	 * Pull data from an endpoint and store in a file on a wordpress uploads subfolder
	 *
	 * ## OPTIONS
	 *
	 * None. For now
	 *
	 * ## EXAMPLES
	 *
	 * To access this feature, type the following into your console:
	 * wp sb_sportsfeed pull_feed_data
	 *
	 * @param		array $args
	 * @param		array $assoc_args
	 * @return		void
	 */
	public function pull_feed_data($args, $assoc_args)
	{
		$author_general_id = 1;
		$author_general = null;
		$author_categories = get_option('sb_feeds_author');

		if(is_array($author_categories) && count($author_categories) > 0) {
			foreach ($author_categories as $category_field => $author_field) {
				if ($category_field == '*') {
					$author_general = $author_field;
					unset($author_categories['*']);
					break;
				}
			}
		}

		$author_general = (!empty($author_general)) ? get_user_by('login', $author_general) : null;
		if (!empty($author_general)) {
			$author_general_id = $author_general->ID;
		}

		// set the filename and path for the JSON response file
		$filename =  'NEWS_FEED_PULL_' . date('Y-m-d-H-i-s') . '.json';
		$file_path = SBSPORTSFE_PLUGIN_LOG_DIR . '/' . $filename;

        // if the uploads directory does not exist, create it and the plugin subfolder
        if (!file_exists(SBSPORTSFE_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SBSPORTSFE_PLUGIN_LOG_DIR);
        } 

		$last_run = get_option('sb_feeds_last_run', null);
		if ($last_run != null) {
			$date_pars = '&start_date=' . urlencode($last_run);
		} else {
			$date_pars = '';
		}

		// set the URL of the REST API
		$url = SBSPORTSFE_PLUGIN_DEAULT_ENDPOINT_URL .  $date_pars;
		//MPHILLIPS
		//$url = SBSPORTSFE_PLUGIN_DEAULT_ENDPOINT_URL;

		WP_CLI::line(sprintf(__('Attempting to retrieve data from %s', 'SB SportsFeed'), $url));
		// initialize curl
		$ch = curl_init();

		// set curl options
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

		// execute the curl request and store the response
		$response = curl_exec($ch);

        $keep_logs = get_field('logs', 'options');

        if ($keep_logs != 'none') {
            // check if the curl request was successful
            if ($response === false) {
                WP_CLI::line(sprintf(__('ERROR: %s', 'SB SportsFeed'), curl_error($ch)));
            } else {
                // write the response to a file
                file_put_contents($file_path, $response);
                WP_CLI::line(sprintf(__('JSON response saved to: %s', 'SB SportsFeed'), $file_path));
            }
        }	

		// close curl
		curl_close($ch);

		//Parse the file
		$parse_results = $this->parse_feed_file($file_path);

		if (isset($parse_results["news"])) {
			$new_posts_list = array();

			foreach ($parse_results["news"] as $news_row) {

				$featured_image_url = isset($news_row["images"][0]["preview_url"]) ? $news_row["images"][0]["preview_url"] : "";
				$featured_image_headline = isset($news_row["images"][0]["headline"]) ? $news_row["images"][0]["headline"] : "";
				$featured_image_credit = isset($news_row["images"][0]["credit"]) ? $news_row["images"][0]["credit"] : "";

				$post_content = $this->create_block('paragraph',  array('align' => 'left'), $news_row['body']);

				$args = array(
					'post_title'   => $news_row['title'],
					'post_type'	   => 'post',
					'post_status'  => 'publish',
					'post_author'  => $author_general_id,
					'post_content' => $post_content,
					'post_modified' => $news_row['updated_at'],
					'meta_input'   => array(
						'sbf_type' 				 => $news_row["type"],
						'sbf_featured_image_url' => $featured_image_url,
						'sbf_image_headline' 	 => $featured_image_headline,
						'sbf_image_credit' 		 => $featured_image_credit,
					),
					'post_date' => $news_row['date'],
				);

				if (isset($news_row["id"])) {
					WP_CLI::line(__('External ID: '. $news_row["id"], 'SB SportsFeed'));
					$args['meta_input']['sbf_external_id'] = $news_row["id"];
				}

				if (isset($news_row["sport_id"])) {
					$sport_cat = sanitize_title($parse_results["sports"][$news_row["sport_id"]]["name"]);
					$author = (isset($author_categories[$sport_cat])) ? get_user_by('login', $author_categories[$sport_cat]) : null;
					
					if (!empty($author)) {
						$author_id = $author->ID;
					} else {
						$author_id = $author_general_id;
					}

					$args['post_author'] = $author_id;
					$categories[] = $this->get_category($parse_results["sports"][$news_row["sport_id"]]["name"], 'sport');
					$args['meta_input']['sbf_sport_id'] = $news_row["sport_id"];
					$args['meta_input']['sbf_sport_name'] = $parse_results["sports"][$news_row["sport_id"]]["name"];
				}

				if (isset($news_row["league_id"])) {
					$categories[] = $this->get_category($parse_results["leagues"][$news_row["league_id"]]["nickname"], 'league', $parse_results["sports"][$news_row["sport_id"]]["name"]);
					$args['meta_input']['sbf_league_id'] = $news_row["league_id"];
					$args['meta_input']['sbf_league_name'] = $parse_results["leagues"][$news_row["league_id"]]["name"];
					$args['meta_input']['sbf_league_nickname'] = $parse_results["leagues"][$news_row["league_id"]]["nickname"];
				}

				if (isset($news_row["home_team_id"])) {
					$categories[] = $this->get_category($parse_results["teams"][$news_row["home_team_id"]]["name"], 'team', $parse_results["sports"][$news_row["sport_id"]]["name"]);
					$args['meta_input']['sbf_home_team_id'] = $news_row["home_team_id"];
					$args['meta_input']['sbf_home_team_name'] = $parse_results["teams"][$news_row["home_team_id"]]["name"];
				}

				if (isset($news_row["visitor_team_id"])) {
					$categories[] = $this->get_category($parse_results["teams"][$news_row["visitor_team_id"]]["name"], 'team', $parse_results["sports"][$news_row["sport_id"]]["name"]);
					$args['meta_input']['sbf_visitor_team_id'] = $news_row["visitor_team_id"];
					$args['meta_input']['sbf_visitor_team_name'] = $parse_results["teams"][$news_row["visitor_team_id"]]["name"];
				}
				
				$args['post_category'] = $categories;

				$response = $this->insert_auto_post($args, $featured_image_url, $post_content);
				unset($categories);
				if (!is_string($response)) {
					$new_posts_list[] = "New Post ID = " . $response;
				} else {
					WP_CLI::Line($response);
				}
				
			}

			WP_CLI::line(sprintf(__('%s succesfully inserted posts', 'SB SportsFeed'), sizeof($new_posts_list)));
		} else {
			WP_CLI::line(__('No data available for this pull', 'SB SportsFeed'));
		}
		//update last run date 
		update_option('sb_feeds_last_run', $parse_results['time_start']);
	}

	function get_category($category_name, $category_type, $category_sport=""){

		if($category_type == 'sport'){

			$category = get_term_by( 'name', $category_name, 'category' );

			if ( $category ) {
				// The category was found, you can use its properties like:
				$category_name = $category->name;
				$category_id = $category->term_id;
				// ...
			  } else {
				$category_args = array(
					'cat_name' => $category_name,
					'category_nicename' => sanitize_title($category_name)
				  );
				
				  $category_id = $this->create_category($category_args, $category_type);

				  WP_CLI::line(__('Category Sport: ' . $category_name . ' inserted', 'SB SportsFeed'));
			  }

		}elseif($category_type == 'league'){

			if ( !empty($category_sport) ) {
				
				$parent_category = get_term_by( 'name', $category_sport, 'category' );
				$category = get_term_by( 'name', $category_name, 'category' );

				if ( $category ) {
					// The category was found, you can use its properties like:
					$category_name = $category->name;
					$category_id = $category->term_id;
			  	} else {
					$category_args = array(
						'cat_name' => $category_name,
						'category_nicename' => sanitize_title($category_name),
						'category_parent' => $parent_category->term_id,
					);
					
					$category_id = $this->create_category($category_args, $category_type);

					WP_CLI::line(__('Category League: ' . $category_name . ' inserted', 'SB SportsFeed'));
			  	}
			}

		}elseif($category_type == 'team'){
			if ( !empty($category_sport) ) {
				
				$parent_category = get_term_by( 'name', $category_sport, 'category' );
				$category = get_term_by( 'name', $category_name, 'category' );

				if ( $category ) {
					// The category was found, you can use its properties like:
					$category_name = $category->name;
					$category_id = $category->term_id;
			  	} else {
					$category_args = array(
						'cat_name' => $category_name,
						'category_nicename' => sanitize_title($category_name),
						'category_parent' => $parent_category->term_id,
					);
					
					$category_id = $this->create_category($category_args, $category_type);

					WP_CLI::line(__('Category Team: ' . $category_name . ' inserted', 'SB SportsFeed'));
			  	}
			}

		}

		return $category_id;


	}

	public function create_category($category_args, $category_type){

		$category_id = wp_insert_category($category_args);
		update_field('sbf_category_type', $category_type, "category_".$category_id);

		return $category_id;

	}

	/**
	 * Proces / Parse the feed file
	 *
	 * ## OPTIONS
	 *
	 * return an array with all the important data from the file filtered if needed
	 *
	 * @param		array $filepath ( the path to a json file containing the feed data)
	 * @return		array
	 */
	function parse_feed_file($file_path)
	{

		//Define local and return variables 
		$final_data = [];

		// Read the contents of the file
		$file_contents = file_get_contents($file_path);

		// Check if reading the file was successful
		if ($file_contents === false) {
			WP_CLI::line(__('Error: Failed to read the file.', 'SB SportsFeed'));
			return null;
		} else {
			// Decode the JSON
			$json_data = json_decode($file_contents, true);

			// Check if decoding the JSON was successful
			if ($json_data === null) {
				WP_CLI::line(__('Error: Failed to decode the JSON.', 'SB SportsFeed'));
				return null;
			} else {
				// Access the JSON data
				foreach ($json_data as $content) {
					foreach ($content["data-content"] as $data) {
						// Access the "full-load" value						
						$final_data['full_load'] = $data["full-load"];

						// Access the "time-start" value
						$final_data['time_start'] = $data["time-start"];

						// Access the "time-end" value
						$final_data['time_end'] = $data["time-end"];

						// Access the "time-zone" value
						$final_data['time_zone'] = $data["time-zone"];

						// Access the sources array
						$sources = $data[0]["sources"];
						foreach ($sources as $source) {
							// Access the "id" value							
							$id = $source["id"];

							// Access the "name" value
							$name = $source["name"];

							//Set the value in a multi dimentional array for future reference
							$final_data['sources'][$id] = $name;
						}

						// Access the sports array
						$sports = $data[0]["sports"];
						foreach ($sports as $sport) {
							// Access the "id" value							
							$id = $sport["id"];

							// Access the "name" value
							$name = $sport["name"];

							// Access the "updated_at" value
							$updated_at = $sport["updated_at"];

							//Set the value in a multi dimentional array for future reference
							$final_data['sports'][$id] = ['name' => $name, 'updated_at' => $updated_at];
						}

						// Access the leagues array
						$leagues = $data[0]["leagues"];
						foreach ($leagues as $league) {
							// Access the "id" value							
							$id = $league["id"];

							// Access the "sport_id" value
							$sport_id = $league["sport_id"];

							// Access the "name" value
							$name = $league["name"];

							// Access the "updated_at" value
							$updated_at = $league["updated_at"];

							// Access the "nickname" value
							$nickname = $league["nickname"];

							//Set the value in a multi dimentional array for future reference
							$final_data['leagues'][$id] = ["sport_id" => $sport_id, 'name' => $name, 'updated_at' => $updated_at, 'nickname' => $nickname];
						}

						// Access the teams array
						$teams = $data[0]["teams"];
						foreach ($teams as $team) {
							// Access the "id" value							
							$id = $team["id"];

							// Access the "name" value
							$name = $team["name"];

							// Access the "short_name" value
							$short_name = $team["short_name"];

							// Access the "updated_at" value
							$updated_at = $team["updated_at"];

							// Access the "nickname" value
							$nickname = $team["nickname"];

							//Set the value in a multi dimentional array for future reference
							$final_data['teams'][$id] = ["name" => $name, 'short_name' => $short_name, 'updated_at' => $updated_at, 'nickname' => $nickname];
						}

						// Access the teams array
						$news_data = $data[0]["news"];
						foreach ($news_data as $news) {

							// Access the "sport_id" value
							$sport_id = $news["sport_id"];

							$allowed_sports = get_option('sb_feeds_allowed_entry_sports', array('244'));
							$allowed_sports_array =  explode(',', $allowed_sports);

							if (!in_array($sport_id, $allowed_sports_array)) {
								continue;
							}

							// Access the "league_id" value
							$league_id = $news["league_id"];

							$allowed_leagues = get_option('sb_feeds_allowed_entry_leagues', array('6064'));
							$allowed_leagues_array =  explode(',', $allowed_leagues);

							if (!in_array($league_id, $allowed_leagues_array)) {
								continue;
							}


							// Access the "type" value
							$type = $news["type"];

							//restrict the kind of news data available for storage
							$allowed_entry_types = get_option('sb_feeds_allowed_entry_types', 'news');
							$allowed_entry_types_array =  explode(',', $allowed_entry_types);

							if (!in_array($type, $allowed_entry_types_array)) {
								continue;
							}

							// Access the "id" value							
							$id = $news["id"];

							// Access the "source_id" value
							$source_id = $news["source_id"];

							// Access the "title" value
							$title = $news["title"];

							// Access the "language" value
							$language = $news["language"];

							// Access the "author" value
							$author = $news["author"];

							// Access the "date" value
							$date = $news["date"];

							// Access the "synopsis" value
							$synopsis = $news["synopsis"];

							// Access the "body" value
							$body = $news["body"];

							// Access the "home_team_id" value
							$home_team_id = $news["home_team_id"];

							// Access the "visitor_team_id" value
							$visitor_team_id = $news["visitor_team_id"];

							// Access the "tags" value
							$tags = isset($news["extra_details"]["tags"]) ? $news["extra_details"]["tags"] : '';

							// Access the "importance" value
							$importance = isset($news["extra_details"]["importance"]) ? $news["extra_details"]["importance"] : '';

							// Access the "slug" value
							$slug = isset($news["extra_details"]["slug"]) ? $news["extra_details"]["slug"] : '';

							// Access the "top_story" value
							$top_story = isset($news["extra_details"]["top_story"]) ? $news["extra_details"]["top_story"] : '';

							// Access the "images" value
							$images = isset($news["extra_details"]["images"]) ? $news["extra_details"]["images"] : [];

							// Access the "updated_at" value
							$updated_at = $news["updated_at"];

							//Set the value in a multi dimentional array for future reference
							$final_data['news'][$id] = [
								//"source_id" => $source_id,
								"id" => $id,
								"type" => $type, //
								"title" => $title,
								//"language" => $language,
								//"source" => $source,
								//"author" => $author,
								"date" => $date,
								//"synopsis" => $synopsis,
								"body" => $body,
								"sport_id" => $sport_id, //
								"league_id" => $league_id, //
								"home_team_id" => $home_team_id, //
								"visitor_team_id" => $visitor_team_id, //
								//"tags" => $tags,
								//"importance" => $importance,
								//"slug" => $slug,
								//"top_story" => $top_story,
								"images" => $images,
								"updated_at" => $updated_at
							];
						}
					}
				}
			}
		}

		//return the array
		return $final_data;
	}


	/**
	 * Wrapp the content of a post on block code
	 *
	 * ## OPTIONS
	 *
	 * The content wrapped on the corresponding block string
	 *
	 *
	 * @param		string $block_name
	 * @param		array  $attributes
	 * @param		string $content
	 * @return		string
	 */
	function create_block($block_name, $attributes = array(), $content = '')
	{

		$block_content = '';
		$attributes_string = json_encode($attributes);

		preg_match_all('/(?:\A|\\r\\n\\r\\n)(.*?)(?=\\r\\n\\r\\n|\z)/s', $content, $matches);

		foreach ($matches[1] as $paragraph) {
			if (strpos($paragraph, "Field Level Media") !== false) {
				continue;
			}
			$block_content .= '<!-- wp:' . $block_name . ' ' . $attributes_string . ' --><p class="has-text-align-left">' . $paragraph . '</p><!-- /wp:' . $block_name . ' -->';
		}

		return $block_content;
	}

	/**
	 * Inserts the data entry as a blog post on the site
	 *
	 * ## OPTIONS
	 *
	 * The content wrapped on the corresponding block string
	 *
	 *
	 * @param		array  $args
	 * @param		string $featured_image_url
	 * @return		the new post id or an error string
	 */
	function insert_auto_post($args, $featured_image_url, $post_content)
	{
		$args_search = array(
			'post_type' => 'post',
			// 'post_status' => 'publish',
			'meta_query' => array(
				array(
					'key' => 'sbf_external_id',
					'value' => $args['meta_input']['sbf_external_id'],
					'compare' => '=',
				),
			),
		);
			
		$query = new WP_Query($args_search);

		if (!$query->have_posts()) {
			$post_id = wp_insert_post($args);

			// Download the image and upload it to the media library
			$attachment_id = media_sideload_image($featured_image_url, $post_id, '', 'id');
	
			// Set the uploaded image as the featured image for the post
			set_post_thumbnail($post_id, $attachment_id);
	
			if (!is_wp_error($post_id)) {
				return $post_id;
			} else {
				return $post_id->get_error_message();
			}
		} else {
			while ($query->have_posts()) {
				$query->the_post();
				$post_id = get_the_ID();
			}
			WP_CLI::line(__('Post already in the DB, ID: '. $post_id, 'SB SportsFeed'));
			return $post_id;
		}

		wp_reset_postdata();
	}

	public function optimize_post_images($args, $assoc_args)
	{
		$helpers = New Sb_Sportsfeed_Helpers;

		if (!$helpers->is_install_shortpixel()) {
			error_log(print_r([
				'mesage' => 'The shortpixel plugin must be installed and active.',
				'plugin_name' => 'shortpixel-image-optimiser/wp-shortpixel.php',
				'active_plugins' => apply_filters('active_plugins', get_option('active_plugins')),
			], true));

			WP_CLI::line(sprintf(__('SB Feeds: The shortpixel plugin must be installed and active.')));

			return;
		}

		if (empty($helpers->get_key_shortpixel())) {
			error_log('SB Feeds: You must set the api key to the shortpixel plugin.');
			WP_CLI::line(sprintf(__('SB Feeds: You must set the api key to the shortpixel plugin.')));

			return;
		}

		# Parameter that indicates the amount of images to be processed at the same time
		$max_chunk = (isset($assoc_args['chunk'])) ? intval($assoc_args['chunk']) : 20;
		$max_chunk = ($max_chunk <= 100) ? $max_chunk : 100;

		# Parameter that indicates if all images are replaced or only those that are not webp
		$replace_images = (in_array('replace', $args)) ? true : false;

		# Parameter that indicates if the process is executed if the optimization is deactivated
		$check_active_cron = (in_array('check_active', $args)) ? true : false;

		if ($check_active_cron && !$helpers->is_active_shortpixel()) {
			error_log('SB Feeds: Shortpixel plugin disabled.');
			WP_CLI::line(sprintf(__('SB Feeds: Shortpixel plugin disabled.')));

			return;
		}

		# Last post optimized
		$last_post_optimized_id = 0;

		if (!empty(get_option('sb_feeds_last_post_optimized')) && get_option('sb_feeds_last_post_optimized') > 0) {
			$last_post_optimized_id = get_option('sb_feeds_last_post_optimized');
		}
		
		# All the ids of posts are consulted, ordering them by id, to make the chunk
		global $wpdb;
		$sql = $wpdb->prepare("SELECT ID FROM $wpdb->posts WHERE post_type = 'post' AND post_status = 'publish' AND ID > %d ORDER BY ID ASC", $last_post_optimized_id);
		$posts = $wpdb->get_results($sql, 'ARRAY_A');

		# Query chunks are created
		$chunks = array_chunk($posts, $max_chunk);

		# Processing of post chunks starts
		foreach ($chunks as $chunk_ids) {
			# Prepare posts ids
			$prepare_ids = array_map(function ($ids) {
				return $ids['ID'];
			}, $chunk_ids);

			# Consult posts by the ids of each chunk
			$args_search = array(
				'posts_per_page' => -1,
				'post_parent' => '0',
				'post__in' => array_values($prepare_ids)
			);

			# initialize variables
			$delete_attachment = [];
			$new_data = [];
			$new_images = [];
			$new_metadata = [];
			$delete_images = [];
			$query = new WP_Query($args_search);

			# Loop through the results of each chunk of data
			if ($query->have_posts()) {
				while ($query->have_posts()) {
					$query->the_post();
					$post_id = get_the_ID();
					$meta = get_post_meta($post_id);

					$thumbnail_id = get_post_thumbnail_id($post_id);
					$attachment = wp_get_attachment_metadata($thumbnail_id);

					# The replace parameter is defined, even the webp images of each post will be replaced
					if (!$replace_images) {
						$featured_image = null;

						if (is_array($attachment) && isset($attachment['file']) && !empty($attachment['file'])) {
							$featured_image = $this->getNameFile($attachment['file']);

							if ($featured_image['extension'] == 'webp') {
								error_log('The webp image is kept in the post: '.$post_id);
								continue;
							} else {
								$delete_attachment[$post_id] = $thumbnail_id;
							}
						}
					} else {
						$delete_attachment[$post_id] = $thumbnail_id;
					}

					$delete_images[] = $post_id;
					$sbf_image_headline = (isset($meta['sbf_image_headline'][0])) ? $meta['sbf_image_headline'][0] : '';
					$sbf_featured_image_url = (isset($meta['sbf_featured_image_url'][0])) ? $meta['sbf_featured_image_url'][0] : '';
					
					# The image url of each post defined in sbf_featured_image_url is taken, to be converted to webp
					if (!empty($sbf_featured_image_url)) {
						$sbf_image = $this->getNameFile($sbf_featured_image_url);

						if (!empty($sbf_image['name']) && !empty($sbf_image['extension'])) {
							$new_metadata[$post_id] = [
								'headline' => $sbf_image_headline,
								'name' => $sbf_image['name'],
								'extension' => $sbf_image['extension'],
							];
							$new_images[$sbf_featured_image_url] = $post_id;
						}
					}
				}
			}

			# If there are posts with images defined in the sbf_featured_image_url field
			if (is_array($new_images) && count($new_images) > 0) {
				# The new images are created in webp format of the chunk of data to be processed
				$new_images_webp = [];
				$new_images_webp = $this->shortPixelReducer($new_images);

				# If the new optimized images were created
				if (is_array($new_images_webp) && count($new_images_webp) > 0) {
					foreach ($new_images_webp as $key => $new_data) {
						try {
							# The image assigned to the post is deleted, before assigning the new optimized image
							if (isset($delete_attachment[$new_data['post_id']])) {
								wp_delete_attachment($delete_attachment[$new_data['post_id']], true);
							}

							# Download the image and upload it to the media library
							$attachment_id = media_sideload_image($new_data['new_image_url'], $new_data['post_id'], $new_metadata[$new_data['post_id']]['headline'], 'id');

							# Update image attachment with alt text
							update_post_meta($attachment_id, '_wp_attachment_image_alt', $new_metadata[$new_data['post_id']]['headline']);
							
							# Set the uploaded image as the featured image for the post
							set_post_thumbnail($new_data['post_id'], $attachment_id);

							if (!is_wp_error($new_data['post_id'])) {
								update_option('sb_feeds_last_post_optimized', $new_data['post_id']);
								error_log('Assign new optimized image in post: '.$new_data['post_id']);
							} else {
								error_log(print_r([
									'message' => 'Error trying to assign the new optimized image',
									'post_id' => $new_data['post_id'],
									'exception' => $post_id->get_error_message(),
								], true));
							}
						} catch (\Exception $exception) {
							error_log(print_r([
								'message' => 'Error trying to update post image.',
								'delete_images' => $delete_images,
								'exception' => $exception->getMessage(),
							], true));
						}
					}
				}
			}
		}
	}

	private function shortPixelReducer($array_images)
	{
		$reducer_images = [];
		$clear_array = array_keys($array_images);
		$helpers = New Sb_Sportsfeed_Helpers;

		$params = json_encode([
			'plugin_version' => 'rdg10',
			'key' => $helpers->get_key_shortpixel(),
			'lossy' => 1,
			'resize' => 0,
			'cmyk2rgb' => 1, 
			'keep_exif'=> 0, 
			'convertto'=> '+webp', 
			'refresh'=> 0, 
			'wait'=> 30,
			'urllist' => array_map('rawurlencode', $clear_array)
		]);

		$context = stream_context_create([
			'http' => [
				'method' => 'POST',
				'header' => "Content-Type: application/json\r\n" . "Accept: application/json\r\n" . "Content-Length: " . strlen($params) ,
				'content' => $params
			]
		]);

		try {
			$responses = json_decode(file_get_contents($helpers->get_request_shortpixel(), false, $context), true);

			foreach ($responses as $response) {
				if (isset($response['Status']['Code']) && $response['Status']['Code'] == 2) {
					$reducer_images[$response['OriginalURL']] = [
						'post_id' => $array_images[$response['OriginalURL']],
						'original_image_url' => $response['OriginalURL'],
						'new_image_url' => $response['WebPLossyURL'],
					];
				} else {
					error_log(print_r([
						'shortpixel_response' => $response,
					], true));
				}
			}
		} catch (\Exception $exception) {
			error_log(print_r([
				'message' => 'Error trying to optimize images.',
				'array_images' => $array_images,
				'reducer_images' => $reducer_images,
				'exception' => $exception->getMessage(),
			], true));
		}

        return (is_array($reducer_images) && count($reducer_images) > 0) ? $reducer_images : null;
    }

	private function getNameFile($image_url)
	{
		$file_name = '';
		$file_extension = '';

		try {
			$parts = explode('/', $image_url);
			$image_name = $parts[count($parts) - 1];
			$image_name_parts = explode('.', $image_name);
			$file_extension = $image_name_parts[count($image_name_parts) - 1];
			unset($image_name_parts[count($image_name_parts) - 1]);
			$file_name = implode('.', $image_name_parts);
		} catch (\Exception $exception) {
			error_log(print_r([
				'image_url' => $image_url,
				'exception' => $exception->getMessage(),
			], true));
		}

		return ['name' => $file_name, 'extension' => $file_extension];
	}
}
