<?php

require __DIR__ . '/../../../vendor/autoload.php';

use Rdgcorp\OpenAI\OpenAI;

class OpenaiPostOptimizer
{
    public static function optimize(int $postId): bool
    {
        try {
            $active = (bool)get_field('openai_active', 'options');
            $token = get_field('openai_token', 'options');

            if (!$active || !$token) {
                return false;
            }

            $post = get_post($postId);
            $thumbnailId = get_post_thumbnail_id($post);
            $thumbnailUrl = wp_get_attachment_url($thumbnailId);

            if (100 > strlen($post->post_content)) {
                return false;
            }

            if (!$thumbnailId || !$thumbnailUrl) {
                return false;
            }

            $openai = new OpenAI($token);

            $newTitle = $openai->generatePostTitle(strip_tags($post->post_content));
            $newThumbnailAlt = $openai->generateFeaturedImageName(self::encodeFileToUrl($thumbnailUrl));

            $newTitle = str_replace('"', '', $newTitle);
            $newThumbnailAlt = str_replace('"', '', $newThumbnailAlt);
            $newThumbnailName = sanitize_title($newThumbnailAlt);

            wp_update_post([
                'ID' => $post->ID,
                'post_title' => $newTitle,
                'post_name' => sanitize_title($newTitle),
            ]);

            $thumbnailName = pathinfo($thumbnailUrl)['filename'];
            $newThumbnailUrl = str_replace($thumbnailName, $newThumbnailName, $thumbnailUrl);

            $realThumbnailPath = wp_get_upload_dir()['basedir'] . explode('/uploads', $thumbnailUrl)[1];
            $newRealThumbnailPath = wp_get_upload_dir()['basedir'] . explode('/uploads', $newThumbnailUrl)[1];

            rename($realThumbnailPath, $newRealThumbnailPath);

            update_attached_file($thumbnailId, explode('/uploads/', $newThumbnailUrl)[1]);

            $thumbnailMeta = wp_get_attachment_metadata($thumbnailId);

            $thumbnailMeta['file'] = str_replace($thumbnailName, $newThumbnailName, $thumbnailMeta['file']);

            $precessedSizes = [];

            foreach ($thumbnailMeta['sizes'] as $key => $size) {
                $thumbnailMeta['sizes'][$key]['file'] = str_replace($thumbnailName, $newThumbnailName, $thumbnailMeta['sizes'][$key]['file']);

                $sizeDimension = $size['width'] . 'x' . $size['height'];

                if (!in_array($sizeDimension, $precessedSizes)) {
                    $thumbnailNameSize = $thumbnailName . '-' . $sizeDimension;
                    $newThumbnailNameSize = $newThumbnailName . '-' . $sizeDimension;

                    $sizePath = str_replace($thumbnailName, $thumbnailNameSize, $realThumbnailPath);
                    $newSizePath = str_replace($newThumbnailName, $newThumbnailNameSize, $newRealThumbnailPath);

                    rename($sizePath, $newSizePath);

                    $precessedSizes[] = $sizeDimension;
                }
            }

            update_post_meta(get_post_thumbnail_id($post), '_wp_attachment_metadata', $thumbnailMeta);

            update_post_meta(get_post_thumbnail_id($post), '_wp_attachment_image_alt', $newThumbnailAlt);
        } catch (\Throwable $exception) {
            return false;
        }

        return true;
    }

    private static function encodeFileToUrl(string $filePath): string
    {
        $type = pathinfo($filePath, PATHINFO_EXTENSION);

        $data = file_get_contents($filePath);

        return 'data:image/' . $type . ';base64,' . base64_encode($data);
    }
}
