<?php

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * HELPER COMMENT START
 * 
 * This is the main class that is responsible for registering
 * the core functions, including the files and setting up all features. 
 * 
 * To add a new class, here's what you need to do: 
 * 1. Add your new class within the following folder: core/includes/classes
 * 2. Create a new variable you want to assign the class to (as e.g. public $helpers)
 * 3. Assign the class within the instance() function ( as e.g. self::$instance->helpers = new Sportshub_Recaps_Helpers();)
 * 4. Register the class you added to core/includes/classes within the includes() function
 * 
 * HELPER COMMENT END
 */

if (!class_exists('Sportshub_Recaps')):

    /**
     * Main Sportshub_Recaps Class.
     *
     * @package         SHRECAPS
     * @subpackage      Classes/Sportshub_Recaps
     * @since           1.0.0
     * @author          IT
     */
    final class Sportshub_Recaps {

        /**
         * The real instance
         *
         * @access      private
         * @since       1.0.0
         * @var         object|Sportshub_Recaps
         */
        private static $instance;

        /**
         * SHRECAPS helpers object.
         *
         * @access      public
         * @since       1.0.0
         * @var         object|Sportshub_Recaps_Helpers
         */
        public $helpers;

        /**
         * SHRECAPS settings object.
         *
         * @access      public
         * @since       1.0.0
         * @var         object|Sportshub_Recaps_Settings
         */
        public $settings;

        public $slug;

        /**
         * Throw error on object clone.
         *
         * Cloning instances of the class is forbidden.
         *
         * @access      public
         * @since       1.0.0
         * @return      void
         */
        public function __clone() {
            _doing_it_wrong(__FUNCTION__, __('You are not allowed to clone this class.', SHRECAPS_NAME_SLUG), '1.0.0');
        }

        /**
         * Disable unserializing of the class.
         *
         * @access      public
         * @since       1.0.0
         * @return      void
         */
        public function __wakeup() {
            _doing_it_wrong(__FUNCTION__, __('You are not allowed to unserialize this class.', SHRECAPS_NAME_SLUG), '1.0.0' );
        }

        /**
         * Main Sportshub_Recaps Instance.
         *
         * Insures that only one instance of Sportshub_Recaps exists in memory at any one
         * time. Also prevents needing to define globals all over the place.
         *
         * @access      public
         * @since       1.0.0
         * @static
         * @return      object|Sportshub_Recaps    The one true Sportshub_Recaps
         */
        public static function instance() {
            if (!isset(self::$instance) && !(self::$instance instanceof Sportshub_Recaps)) {
                self::$instance = new Sportshub_Recaps;
                self::$instance->base_hooks();
                self::$instance->includes();
                self::$instance->helpers = new Sportshub_Recaps_Helpers();
                self::$instance->settings = new Sportshub_Recaps_Settings();

                if (defined('SHR_SLUG')) {
                    $slug = json_decode(SHR_SLUG, true);
                } else {
                    error_log(print_r([
                        'message' => 'Sportshub Recaps: The constant SHR_SLUG is not defined.',
                    ], true));

                    $slug = [
                        'slug' => 'recap',
                        'label' => 'Recap',
                    ];
                }

                self::$instance->slug = $slug;

                //Fire the plugin logic
                new Sportshub_Recaps_Run();

                /**
                 * Fire a custom action to allow dependencies
                 * after the successful plugin setup
                 */
                do_action('SHRECAPS/plugin_loaded');
            }

            return self::$instance;
        }

        /**
         * Include required files.
         *
         * @access  private
         * @since   1.0.0
         * @return  void
         */
        private function includes() {
            require_once SHRECAPS_PLUGIN_DIR . 'core/includes/classes/class-sportshub-recaps-helpers.php';
            require_once SHRECAPS_PLUGIN_DIR . 'core/includes/classes/class-sportshub-recaps-settings.php';

            if (defined( 'WP_CLI' ) && WP_CLI ){
                require_once SHRECAPS_PLUGIN_DIR . 'core/includes/classes/class-sportshub-recaps-cli.php';
            }

            require_once SHRECAPS_PLUGIN_DIR . 'core/includes/classes/class-sportshub-recaps-run.php';
            require_once SHRECAPS_PLUGIN_DIR . 'core/includes/groups/form-groups.php';
        }

        /**
         * Add base hooks for the core functionality
         *
         * @access  private
         * @since   1.0.0
         * @return  void
         */
        private function base_hooks() {
            add_action('plugins_loaded', array(self::$instance, 'load_textdomain'));
        }

        /**
         * Loads the plugin language files.
         *
         * @access  public
         * @since   1.0.0
         * @return  void
         */
        public function load_textdomain() {
            load_plugin_textdomain(SHRECAPS_NAME_SLUG, FALSE, dirname(plugin_basename(SHRECAPS_PLUGIN_FILE)) . '/languages/');
        }

        public function get_slug($array = true) {
            if ($array) {
                return $this->slug;
            } else {
                return $this->slug['slug'];
            }
        }
    }

endif; // End if class_exists check.