<?php

// Exit if accessed directly.
if ( ! defined('ABSPATH') ) exit;

/**
 * Class Sportshub_Recaps_Run
 * Thats where we bring the plugin to life
 * @package     SHRECAPS
 * @subpackage  Classes/Sportshub_Recaps_Run
 * @author      IT
 * @since       1.0.0
 */

class Sportshub_Recaps_Run {
    /**
     * Our Sportshub_Recaps_Run constructor 
     * to run the plugin logic.
     * @since 1.0.0
     */

    function __construct() {
        $this->add_hooks();
    }

    /**
     * ######################
     * ###
     * #### WORDPRESS HOOKS
     * ###
     * ######################
     */

    /**
     * Registers all WordPress and plugin related hooks
     * @access  private
     * @since   1.0.0
     * @return  void
     */

    private function add_hooks() {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts_and_styles'], 20);
        add_action('admin_menu', [$this, 'register_sportshub_recaps_menu'], 20);
        add_action('acf/init', [$this, 'generate_form_sportshub_recaps'], 20);
        add_action('acf/init', [$this, 'generate_form_sportshub_recaps_contents'], 20);
        add_action('acf/save_post', [$this, 'save_options_sportshub_recaps']);
        add_action('acf/save_post', [$this, 'save_options_contents_types']);
        add_filter('cron_schedules', [$this,'sportshub_add_cron_interval']);
        add_action('shr_recaps_cron_event', [$this, 'shr_recaps_cron_function']);
        register_activation_hook(SHRECAPS_PLUGIN_FILE, [$this, 'activation_hook_callback']);
        register_deactivation_hook(SHRECAPS_PLUGIN_FILE, [$this, 'deactivation_hook_callback']);

        if (empty(get_option('shr_source')) || get_option('shr_source') == 'members') {
            add_action('acf/init', [$this, 'generate_form_cpt_recap']);
        } else {
            add_action('acf/init', [$this, 'generate_form_cpt_recap_simple']);
        }

        add_filter('acf/load_value/name=shr_reset_last_run', [$this, 'load_field_shr_reset_last_run'], 10, 3);
        
        if (get_option('shr_status') == 1) {
            add_action('init', [$this, 'add_custom_post_type'], 10);
        }

        add_filter('mime_types', [$this, 'shr_add_jfif_files'], 10, 3);
        add_action('rank_math/vars/register_extra_replacements', [$this, 'shr_register_rankmath_vars']);
        add_filter('acf/load_value/name=shr_recaps_blocks_load_colors_styles', [$this, 'load_field_shr_blocks_load_colors_styles'], 10, 3);
    }

    public function load_field_shr_blocks_load_colors_styles($value) {
        return 'none';
    }

    public function shr_register_rankmath_vars() {
        if (!function_exists('rank_math_register_var_replacement')) return;

        rank_math_register_var_replacement(
            'home_team_name',
            [
                'name' => esc_html__('Home Team ', 'rank-math'),
                'description' => esc_html__('Home team name', 'rank-math'),
                'variable' => 'home_team_name',
                'example' => 'Bulls',
            ],
            [$this, 'shr_home_team_name_callback']
        );
    
        rank_math_register_var_replacement(
            'visitor_team_name',
            [
                'name' => esc_html__('Visitor Team ', 'rank-math'),
                'description' => esc_html__('Visitor team name', 'rank-math'),
                'variable' => 'visitor_team_name',
                'example' => 'Lakers',
            ],
            [$this, 'shr_visitor_team_name_callback']
        );
    
        rank_math_register_var_replacement(
            'sport_team',
            [
                'name' => esc_html__('Sport', 'rank-math'),
                'description' => esc_html__('Sport', 'rank-math'),
                'variable' => 'sport_team',
                'example' => 'Basketball',
            ],
            [$this, 'shr_sport_team_callback']
        );
    
        rank_math_register_var_replacement(
            'league_team',
            [
                'name' => esc_html__('League', 'rank-math'),
                'description' => esc_html__('League', 'rank-math'),
                'variable' => 'league_team',
                'example' => 'NBA',
            ],
            [$this, 'shr_league_team_callback']
        );
    
        rank_math_register_var_replacement(
            'date_team',
            [
                'name' => esc_html__('Date', 'rank-math'),
                'description' => esc_html__('Date', 'rank-math'),
                'variable' => 'date_team',
                'example' => 'March 18, 2024',
            ],
            [$this, 'shr_date_callback']
        );
    
        rank_math_register_var_replacement(
            'league_or_sport_team',
            [
                'name' => esc_html__('Sport or League', 'rank-math'),
                'description' => esc_html__('Sport or League', 'rank-math'),
                'variable' => 'league_or_sport_team',
                'example' => 'Basketball|NBA',
            ],
            [$this, 'shr_league_or_sport_team_callback']
        );
    }

    public function shr_league_or_sport_team_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shr_league_name'][0]) && !empty($meta['shr_league_name'][0])) {
                return $meta['shr_league_name'][0];
            }

            if (isset($meta['shr_sport_name'][0]) && !empty($meta['shr_sport_name'][0])) {
                return $meta['shr_sport_name'][0];
            }
        }

        return '';
    }

    public function shr_date_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shr_game_date'][0]) && !empty($meta['shr_game_date'][0])) {
                try {
                    $d = DateTime::createFromFormat('Y-m-d H:i:s', $meta['shr_game_date'][0]);
                    $date_status = $d && $d->format('Y-m-d H:i:s') == $meta['shr_game_date'][0];

                    if ($date_status) {
                        return $d->format('F j, Y');
                    }
                } catch (\Exception $exception) {
                    //
                }
            }
        }

        return '';
    }

    public function shr_league_team_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shr_league_name'][0]) && !empty($meta['shr_league_name'][0])) {
                return $meta['shr_league_name'][0];
            }
        }

        return '';
    }

    public function shr_sport_team_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shr_sport_name'][0]) && !empty($meta['shr_sport_name'][0])) {
                return $meta['shr_sport_name'][0];
            }
        }

        return '';
    }

    public function shr_visitor_team_name_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shr_visitor_team'][0]) && !empty($meta['shr_visitor_team'][0])) {
                return $meta['shr_visitor_team'][0];
            }
        }

        return '';
    }

    public function shr_home_team_name_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shr_home_team'][0]) && !empty($meta['shr_home_team'][0])) {
                return $meta['shr_home_team'][0];
            }
        }

        return '';
    }

    public function shr_add_jfif_files($mimes) {
        $mimes['jfif'] = "image/jpeg";

        return $mimes;
    }

    public function enqueue_scripts_and_styles() {
        $shr = Sportshub_Recaps::instance();
        $shrecaps_styles = $shr->helpers->webpack('shrecaps_styles.css');

        wp_register_script('shrecaps_scripts', SHRECAPS_PLUGIN_URL.'core/includes/assets/js/shrecaps-scripts.js', [], SHRECAPS_VERSION, true);
        wp_register_style('shrecaps_styles', $shrecaps_styles, [], SHRECAPS_PLUGIN_URL);
    }

    public function load_field_shr_reset_last_run($value) {
        if ($value == 'reset') {
            update_field('shr_reset_last_run', 'default');
        }

        return 'default';
    }

    public function sportshub_add_cron_interval($schedules) {
        if(!array_key_exists("hourly", $schedules)) {
            $schedules['hourly'] = array(
                'interval' => 60 * 60,
                'display'  => __('Every Hour', SHRECAPS_NAME_SLUG),
            );
        }

        if(!array_key_exists("every_2_hours", $schedules)) {
            $schedules['every_2_hours'] = array(
                'interval' => (60 * 60) * 2,
                'display'  => __('Every 2 Hours', SHRECAPS_NAME_SLUG),
            );
        }

        if(!array_key_exists("every_3_hours", $schedules)) {
            $schedules['every_3_hours'] = array(
                'interval' => (60 * 60) * 3,
                'display'  => __('Every 3 Hours', SHRECAPS_NAME_SLUG),
            );
        }

        if(!array_key_exists("twicedaily", $schedules)) {
            $schedules['twicedaily'] = array(
                'interval' => 60 * 60 * 12,
                'display'  => __('Every 12 hours', SHRECAPS_NAME_SLUG),
            );
        }

        if(!array_key_exists("daily", $schedules)) {
            $schedules['daily'] = array(
                'interval' => 60 * 60 * 24,
                'display'  => __('Every Day', SHRECAPS_NAME_SLUG),
            );
        }

        if(!array_key_exists("weekly", $schedules)) {
            $schedules['weekly'] = array(
                'interval' => 60 * 60 * 24 * 7,
                'display'  => __('Once Weekly', SHRECAPS_NAME_SLUG),
            );
        }

        return $schedules;
    }

    public function shr_recaps_cron_function() {
        // Call the WP-CLI command
        $command = 'wp sportshub_recaps pull';
        $output = shell_exec($command);
    }

    public function add_plugin_action_link($links) {
        $links['our_shop'] = sprintf('<a href="%s" title="Custom Link" style="font-weight:700;">%s</a>', 'https://test.test', __('Custom Link', SHRECAPS_NAME_SLUG) );

        return $links;
    }

    public function register_sportshub_recaps_menu() {
        acf_add_options_page([
            'page_title'    => __('Sports Hub Recaps Settings', SHRECAPS_NAME_SLUG),
            'menu_title'    => __('SH Recaps', SHRECAPS_NAME_SLUG),
            'menu_slug'     => SHRECAPS_NAME_SLUG,
            'capability'    => 'edit_posts',
            'icon_url'      => 'dashicons-games',
            'redirect'      => true,
        ]);

        acf_add_options_page([
            'page_title'  => __('Sports Hub Recaps Contents', SHRECAPS_NAME_SLUG),
            'menu_title'  => __('Contents', SHRECAPS_NAME_SLUG),
            'menu_slug'     => 'sh-recaps-contents',
            'parent_slug' => SHRECAPS_NAME_SLUG,
        ]);

        acf_add_options_page([
            'page_title'  => __('Sports Hub Recaps Settings', SHRECAPS_NAME_SLUG),
            'menu_title'  => __('Settings', SHRECAPS_NAME_SLUG),
            'menu_slug'     => 'sh-recaps-settings',
            'parent_slug' => SHRECAPS_NAME_SLUG,
        ]);
    }

    public function add_custom_post_type() {
        global $shr;
        $shr = (!empty($shr)) ? $shr : Sportshub_Recaps::instance();

        $labels = [
            'name'               => __($shr->slug['label'], SHRECAPS_NAME_SLUG),
            'singular_name'      => __('Recap', SHRECAPS_NAME_SLUG),
            'menu_name'          => __($shr->slug['menu'], SHRECAPS_NAME_SLUG),
            'all_items'          => __('All Recaps', SHRECAPS_NAME_SLUG),
            'search_items'       => __('Search Recaps', SHRECAPS_NAME_SLUG),
            'not_found_in_trash' => __('No recaps found in Trash.', SHRECAPS_NAME_SLUG),
            'not_found'          => __('No recaps found.', SHRECAPS_NAME_SLUG),
        ];

        $supports = [
            'title',
            'editor',
            'author',
            'thumbnail',
            'excerpt',
            'custom-fields',
            'revisions',
            'page-attributes'
        ];

        $taxonomyCategory = get_option('shr_slug') . '_categories';
        $taxonomyTag = get_option('shr_slug') .'_tags';

        $args = [
            'labels'            => $labels,
            'supports'          => $supports,
            'public'            => true,
            'has_archive'       => true,
            'show_in_rest'      => true,
            'hierarchical'      => true,
            'capability_type'   => 'post',
            'taxonomies'        => [$taxonomyCategory, $taxonomyTag],
        ];

        add_rewrite_rule('^'.get_option('shr_slug').'/?$', 'index.php?pagename='.get_option('shr_slug'), 'top');
        add_rewrite_rule('^'.get_option('shr_slug').'/page/([0-9]+)/?$', 'index.php?pagename='.get_option('shr_slug').'&paged=$matches[1]', 'top');

        register_post_type(get_option('shr_slug'), $args);

        $args_taxonomy = [
            'label'         => __('Categories', SHRECAPS_NAME_SLUG),
            'singular_name' => __('Category', SHRECAPS_NAME_SLUG),
            'public'        => true,
            'hierarchical'  => true,
            'show_in_rest'  => true,
            'rewrite'       => true,
            'query_var'     => true,
        ];

        register_taxonomy($taxonomyCategory, get_option('shr_slug'), $args_taxonomy);

        $args_tag = [
            'label'         => __('Tags', SHRECAPS_NAME_SLUG),
            'singular_name' => __('Tag', SHRECAPS_NAME_SLUG),
            'public'        => true,
            'hierarchical'  => false,
            'show_in_rest'  => true,
            'rewrite'       => true,
            'query_var'     => true,
        ];

        register_taxonomy($taxonomyTag, get_option('shr_slug'), $args_tag);
    }

    public function activation_hook_callback() {
        if (!wp_next_scheduled('shr_recaps_cron_event')) {
            wp_schedule_event(time(), 'twicedaily', 'shr_recaps_cron_event');
            update_option('shr_recaps_cron_time', 'twicedaily');
        }
    }

    public function deactivation_hook_callback() {
        wp_clear_scheduled_hook('shr_recaps_cron_event');
    }

    public function generate_form_sportshub_recaps() {
        $helpers = New Sportshub_Recaps_Helpers;
        $helpers->setRedisGroup(REDIS_GROUP_SHRECAPS);
        $sports = (function_exists('shMappingSportsLeaguesOption')) ? shMappingSportsLeaguesOption() : [];
        $sports_clear = $sports;

        if (is_array($sports)) {
            $sports = ['0' => 'All Sports'] + $sports;
        }

        if(function_exists('acf_add_local_field_group')):
            acf_add_local_field_group(array(
                'key' => 'key_sportshub_recaps_settings',
                'title' => 'Sports Hub Recaps Settings',
                'fields' => array(
                    array(
                        'key' => 'key_shr_recaps_status',
                        'label' => 'Status',
                        'name' => 'shr_status',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '15',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'inactive' => 'Inactive',
                            'active' => 'Active',
                        ),
                        'default_value' => 'inactive',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_shr_recaps_data_source',
                        'label' => 'Data Source',
                        'name' => 'shr_data_source',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '18',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'members' => 'Members',
                            'masterfeed' => 'Masterfeed',
                        ),
                        'default_value' => 'members',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_shr_recaps_debug_block',
                        'label' => 'Debug Blocks Games Active?',
                        'name' => 'shr_debug_block_games',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '16',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'none' => 'None',
                            'active' => 'Active',
                        ),
                        'default_value' => 'none',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_shr_recaps_status_games',
                        'label' => 'Game Status',
                        'name' => 'shr_status_games',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '14',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'draft' => 'Draft',
                            'publish' => 'Publish',
                        ),
                        'default_value' => 'draft',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_shr_recaps_cron_time',
                        'label' => 'Cronjob run time',
                        'name' => 'shr_cron_time',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '13',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'hourly' => 'Once Hourly',
                            'every_2_hours' => 'Once Every 2 Hours',
                            'every_3_hours' => 'Once Every 3 Hours',
                            'twicedaily' => 'Twice Daily',
                            'daily' => 'Once Daily',
                            'weekly' => 'Once Weekly',
                        ),
                        'default_value' => 'twicedaily',
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'key_shr_recaps_range_days',
                        'label' => 'Days range',
                        'name' => 'shr_range_days',
                        'type' => 'number',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => 7,
                        'min' => 1,
                    ),
                    array(
                        'key' => 'key_shr_recaps_last_run_message',
                        'label' => 'Last Sports Pull Datetime',
                        'name' => 'shr_last_run_message',
                        'aria-label' => '',
                        'type' => 'message',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '14',
                            'class' => '',
                            'id' => '',
                        ),
                        'message' => get_option('shr_recaps_last_run', 'never'),
                        'new_lines' => 'wpautop',
                        'esc_html' => 0,
                    ),
                    array(
                        'key' => 'key_shr_recaps_teams_url',
                        'label' => 'URL Teams',
                        'name' => 'shr_teams_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_shr_recaps_games_url',
                        'label' => 'URL Games',
                        'name' => 'shr_games_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_shr_recaps_api_url',
                        'label' => 'URL Members',
                        'name' => 'shr_api_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_data_source',
                                    'operator' => '==',
                                    'value' => 'members',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_shr_recaps_api_token',
                        'label' => 'Token Members',
                        'name' => 'shr_api_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_data_source',
                                    'operator' => '==',
                                    'value' => 'members',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_shr_recaps_api_masterfeed_url',
                        'label' => 'URL Masterfeed',
                        'name' => 'sportshub_api_masterfeed_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_data_source',
                                    'operator' => '==',
                                    'value' => 'masterfeed',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_shr_recaps_api_masterfeed_token',
                        'label' => 'Token Masterfeed',
                        'name' => 'shr_api_masterfeed_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_data_source',
                                    'operator' => '==',
                                    'value' => 'masterfeed',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_shr_title_format',
                        'label' => 'Title Format',
                        'name' => 'shr_title_format',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_shr_recaps_sports_leagues',
                        'label' => 'Sports/Leagues',
                        'name' => 'shr_sports_leagues',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_shr_recaps_masterfeed_sport_leagues',
                                'label' => 'League',
                                'name' => 'shr_sport_league',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '100',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports_clear,
                                'default_value' => 1,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'key_shr_recaps_sports_leagues',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'key_shr_recaps_sport_leagues',
                        'label' => 'Sports / Authors',
                        'name' => 'shr_authors',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_shr_recaps_sport',
                                'label' => 'Sport',
                                'name' => 'shr_sport',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'key_shr_recaps_sport_leagues',
                            ),
                            array(
                                'key' => 'key_shr_recaps_user_author',
                                'label' => 'Author',
                                'name' => 'shr_sport_user',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $this->get_authors(),
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'key_shr_recaps_sport_leagues',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'key_shr_recaps_main_games',
                        'label' => 'Sports / Main Games',
                        'name' => 'shr_sports_main_games',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_shr_recaps_main_sport',
                                'label' => 'Sport',
                                'name' => 'shr_sport',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'key_shr_recaps_main_games',
                            ),
                            array(
                                'key' => 'key_shr_recaps_main_game',
                                'label' => 'Game',
                                'name' => 'shr_game',
                                'aria-label' => '',
                                'type' => 'post_object',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'post_type' => array(
                                    0 => get_option('shr_slug'),
                                ),
                                'post_status' => '',
                                'taxonomy' => '',
                                'return_format' => 'id',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 1,
                                'parent_repeater' => 'key_shr_recaps_main_games',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'key_shr_recaps_featured_images',
                        'label' => 'Sports / Featured Images',
                        'name' => 'shr_sports_featured_images',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_shr_recaps_featured_sport',
                                'label' => 'Sport',
                                'name' => 'shr_sport',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'key_shr_recaps_featured_images',
                            ),
                            array(
                                'key' => 'key_shr_recaps_featured_image',
                                'label' => 'Featued Image',
                                'name' => 'shr_image',
                                'aria-label' => '',
                                'type' => 'image',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'acfe_save_meta' => 0,
                                'uploader' => '',
                                'return_format' => 'id',
                                'upload_folder' => 'sh-games',
                                'acfe_thumbnail' => 0,
                                'min_width' => '',
                                'min_height' => '',
                                'min_size' => '',
                                'max_width' => '',
                                'max_height' => '',
                                'max_size' => '',
                                'mime_types' => '',
                                'preview_size' => 'thumbnail',
                                'library' => 'all',
                                'parent_repeater' => 'key_shr_recaps_featured_images',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'key_shr_recaps_blocks_load_colors_styles',
                        'label' => 'Load Colors / Styles',
                        'name' => 'shr_recaps_blocks_load_colors_styles',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'none' => 'None',
                            'colors' => 'Colors',
                            'styles' => 'Styles',
                            'both' => 'Both',
                        ),
                        'default_value' => 'none',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_shr_recaps_last_run',
                        'label' => 'Reset Last Run',
                        'name' => 'shr_reset_last_run',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '75',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'default' => 'Default',
                            'reset' => 'Reset',
                        ),
                        'default_value' => '',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'sh-recaps-settings',
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            endif;    
    }

    public function get_authors() {
        try {
            $users_list = [];
            $roles = ['contributor', 'author', 'editor', 'administrator'];
            $users = get_users(['role__in' => $roles]);

            if (is_array($users) && count($users) > 0) {
                foreach ($users as $user) {
                    if (isset($user->data->user_login) && !empty($user->data->user_login)) {
                        $users_list[$user->data->ID] = $user->data->user_login;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'Sportshub Recaps: '.$exception->getMessage(),
                'roles' => $roles,
            ], true));
        }

        return $users_list;
    }

    public function generate_form_cpt_recap() {
        if( function_exists('acf_add_local_field_group') ):
            acf_add_local_field_group(array(
                'key' => 'group_sh_recaps_automatic_full',
                'title' => 'SH Recaps - Automatic',
                'fields' => array(
                    array(
                        'key' => 'key_sh_recaps_external_id',
                        'label' => 'Game External ID',
                        'name' => 'shr_external_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_game_date',
                        'label' => 'Game Date',
                        'name' => 'shr_game_date',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_links_block',
                        'label' => 'Links Block',
                        'name' => 'shr_game_links_block',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'acfe_save_meta' => 0,
                        'choices' => array(
                            'show' => 'Show',
                            'hide' => 'Hide',
                        ),
                        'default_value' => 'show',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_sh_recaps_sport_id',
                        'label' => 'Sport ID',
                        'name' => 'shr_sport_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_sport_name',
                        'label' => 'Sport Name',
                        'name' => 'shr_sport_name',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_league',
                        'label' => 'League ID',
                        'name' => 'shr_league',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_league_name',
                        'label' => 'League Name',
                        'name' => 'shr_league_name',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '15',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_picks_count',
                        'label' => 'Picks Count',
                        'name' => 'shr_game_picks_count',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_home_team_id',
                        'label' => 'ID Home Team',
                        'name' => 'shr_id_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_home_team',
                        'label' => 'Home Team',
                        'name' => 'shr_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_home_score',
                        'label' => 'Home Score',
                        'name' => 'shr_home_score',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_home_api_image',
                        'label' => 'Home API Image',
                        'name' => 'shr_home_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_home_local_image',
                        'label' => 'Home Local Image',
                        'name' => 'shr_home_local_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_visitor_team_id',
                        'label' => 'ID Visitor Team',
                        'name' => 'shr_id_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_visitor_team',
                        'label' => 'Visitor Team',
                        'name' => 'shr_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_visitor_score',
                        'label' => 'Visitor Score',
                        'name' => 'shr_visitor_score',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_visitor_api_image',
                        'label' => 'Visitor API Image',
                        'name' => 'shr_visitor_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_visitor_local_image',
                        'label' => 'Visitor Local Image',
                        'name' => 'shr_visitor_local_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_game_url',
                        'label' => 'Game URL',
                        'name' => 'shr_game_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_top_handicapper',
                        'label' => 'Top Handicapper',
                        'name' => '',
                        'aria-label' => '',
                        'type' => 'accordion',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'open' => 1,
                        'multi_expand' => 1,
                        'endpoint' => 0,
                    ),
                    array(
                        'key' => 'key_sh_recaps_capper_id',
                        'label' => 'ID',
                        'name' => 'shr_game_top_handicapper_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_capper_name',
                        'label' => 'Name',
                        'name' => 'shr_game_top_handicapper_name',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_capper_api_image',
                        'label' => 'API Image',
                        'name' => 'shr_game_top_handicapper_api_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '35',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_capper_local_image',
                        'label' => 'Local Image',
                        'name' => 'shr_game_top_handicapper_local_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '35',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_capper_description',
                        'label' => 'Description',
                        'name' => 'shr_game_top_handicapper_description',
                        'aria-label' => '',
                        'type' => 'textarea',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'acfe_textarea_code' => 0,
                        'maxlength' => '',
                        'rows' => 4,
                        'placeholder' => '',
                        'new_lines' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_results',
                        'label' => 'Results',
                        'name' => '',
                        'aria-label' => '',
                        'type' => 'accordion',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'open' => 1,
                        'multi_expand' => 1,
                        'endpoint' => 0,
                    ),
                    array(
                        'key' => 'key_sh_recaps_results_rank',
                        'label' => 'Rank',
                        'name' => 'shr_game_results_rank',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_results_yesterday',
                        'label' => 'Yesterday',
                        'name' => 'shr_game_results_yesterday',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_results_last_three_days',
                        'label' => 'Last Three Days',
                        'name' => 'shr_game_results_lastthreedays',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_results_last_seven_days',
                        'label' => 'Last Seven Days',
                        'name' => 'shr_game_results_lastsevendays',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_results_last_thirty_days',
                        'label' => 'Last Thirty Days',
                        'name' => 'shr_game_results_lastthirtydays',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_capper_picks',
                        'label' => 'Picks Handicapper',
                        'name' => 'shr_picks',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_sh_recaps_capper_picks_date',
                                'label' => 'Date',
                                'name' => 'shr_pick_date',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '40',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_capper_picks',
                            ),
                            array(
                                'key' => 'key_sh_recaps_capper_picks_status',
                                'label' => 'Status',
                                'name' => 'shr_pick_status',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_capper_picks',
                            ),
                            array(
                                'key' => 'key_sh_recaps_capper_picks_home_name',
                                'label' => 'Home Name',
                                'name' => 'shr_pick_home_name',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '40',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_capper_picks',
                            ),
                            array(
                                'key' => 'key_sh_recaps_capper_picks_home_image_api',
                                'label' => 'Home Image Api',
                                'name' => 'shr_pick_home_image_api',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_capper_picks',
                            ),
                            array(
                                'key' => 'key_sh_recaps_capper_picks_home_image_local',
                                'label' => 'Home Image Local',
                                'name' => 'shr_pick_home_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_capper_picks',
                            ),
                            array(
                                'key' => 'key_sh_recaps_capper_picks_visitor_name',
                                'label' => 'Visitor Name',
                                'name' => 'shr_pick_visitor_name',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '40',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_capper_picks',
                            ),
                            array(
                                'key' => 'key_sh_recaps_capper_picks_visitor_image',
                                'label' => 'Visitor Image Api',
                                'name' => 'shr_pick_visitor_image_api',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_capper_picks',
                            ),
                            array(
                                'key' => 'key_sh_recaps_capper_picks_visitor_image_local',
                                'label' => 'Visitor Image Local',
                                'name' => 'shr_pick_visitor_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_capper_picks',
                            ),
                            array(
                                'key' => 'key_sh_recaps_capper_picks_game_url',
                                'label' => 'Game Url',
                                'name' => 'shr_pick_game_url',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_capper_picks',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'key_sh_recaps_odds',
                        'label' => 'Game Odds',
                        'name' => '',
                        'aria-label' => '',
                        'type' => 'accordion',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'open' => 1,
                        'multi_expand' => 1,
                        'endpoint' => 0,
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_game_id',
                        'label' => 'Game ID',
                        'name' => 'shr_game_id_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_game_external_id',
                        'label' => 'External ID',
                        'name' => 'shr_external_id_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_sportsbook',
                        'label' => 'Sportsbook',
                        'name' => 'shr_sportsbook_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_date',
                        'label' => 'Date',
                        'name' => 'shr_date_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_home_spread_handicap',
                        'label' => 'Spread Home Handicap',
                        'name' => 'shr_home_handicap_spread_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_home_spread_odds',
                        'label' => 'Spread Home Odds',
                        'name' => 'shr_home_spread_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_home_money_odds',
                        'label' => 'Money Line Home Odds',
                        'name' => 'shr_home_money_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_home_handicap_overunder_odds',
                        'label' => 'Over/Under Line Home Handicap',
                        'name' => 'shr_home_handicap_overunder_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_home_overunder_odds',
                        'label' => 'Over/Under Line Home Odds',
                        'name' => 'shr_home_overunder_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_visitor_spread',
                        'label' => 'Spread Visitor Handicap',
                        'name' => 'shr_away_handicap_spread_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_visitor_spread_odds',
                        'label' => 'Spread Visitor Odds',
                        'name' => 'shr_away_spread_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_visitor_money_odds',
                        'label' => 'Money Line Visitor Odds',
                        'name' => 'shr_away_money_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_visitor_handicap_overunder_odds',
                        'label' => 'Over/Under Line Visitor Handicap',
                        'name' => 'shr_away_handicap_overunder_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_odd_visitor_overunder_odds',
                        'label' => 'Over/Under Line Visitor Odds',
                        'name' => 'shr_away_overunder_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_picks_game',
                        'label' => 'Picks Game',
                        'name' => 'shr_picks_game',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_sh_recaps_picks_capper_name',
                                'label' => 'Handicapper Name',
                                'name' => 'shr_pick_game_handicapper_name',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_capper_package',
                                'label' => 'Handicapper Package',
                                'name' => 'shr_pick_game_handicapper_package',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_capper_image',
                                'label' => 'Handicapper Image',
                                'name' => 'shr_pick_game_handicapper_image',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_capper_name_local',
                                'label' => 'Handicapper Image Local',
                                'name' => 'shr_pick_game_handicapper_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_sport_name',
                                'label' => 'Sport Name',
                                'name' => 'shr_pick_game_sport_name',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_sport_acronym',
                                'label' => 'Acronym Sport',
                                'name' => 'shr_pick_game_sport_acronym',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '10',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_bet',
                                'label' => 'Bet',
                                'name' => 'shr_pick_game_bet',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '10',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_win',
                                'label' => 'Win',
                                'name' => 'shr_pick_game_handicapper_win',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '10',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_lose',
                                'label' => 'Lose',
                                'name' => 'shr_pick_game_handicapper_lose',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '10',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_teaser',
                                'label' => 'Teaser',
                                'name' => 'shr_pick_game_teaser',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_date',
                                'label' => 'Date',
                                'name' => 'shr_pick_game_date',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_home_image_api',
                                'label' => 'Home Image Api',
                                'name' => 'shr_pick_game_home_image_api',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_home_image_local',
                                'label' => 'Home Image Local',
                                'name' => 'shr_pick_game_home_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_visitor_image_api',
                                'label' => 'Visitor Image Api',
                                'name' => 'shr_pick_game_visitor_image_api',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_visitor_image_local',
                                'label' => 'Visitor Image Local',
                                'name' => 'shr_pick_game_visitor_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                            array(
                                'key' => 'key_sh_recaps_picks_url_profile',
                                'label' => 'Profile Url',
                                'name' => 'shr_pick_game_url_profile',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_sh_recaps_picks_game',
                            ),
                        ),
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'post_type',
                            'operator' => '==',
                            'value' => get_option('shr_slug'),
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));
            endif;        
    }

    public function generate_form_cpt_recap_simple() {
        if( function_exists('acf_add_local_field_group') ):
            acf_add_local_field_group(array(
                'key' => 'group_sh_recaps_automatic_simple',
                'title' => 'SH Recaps - Automatic',
                'fields' => array(
                    array(
                        'key' => 'key_sh_recaps_external_id',
                        'label' => 'Game External ID',
                        'name' => 'shr_external_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_sport_id',
                        'label' => 'Sport ID',
                        'name' => 'shr_sport_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_game_date',
                        'label' => 'Game Date',
                        'name' => 'shr_game_date',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_home_id',
                        'label' => 'ID Home Team',
                        'name' => 'shr_id_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_home_name',
                        'label' => 'Home Team',
                        'name' => 'shr_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_visitor_id',
                        'label' => 'ID Visitor Team',
                        'name' => 'shr_id_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_visitor_name',
                        'label' => 'Visitor Team',
                        'name' => 'shr_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_game_url',
                        'label' => 'Game URL',
                        'name' => 'shr_game_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_image_api',
                        'label' => 'Game API Image',
                        'name' => 'shr_game_api_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_recaps_image_local',
                        'label' => 'Game Local Image',
                        'name' => 'shr_game_local_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'post_type',
                            'operator' => '==',
                            'value' => get_option('shr_slug'),
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));
            endif;        
    }

    public function generate_form_sportshub_recaps_contents() {
        if(function_exists('acf_add_local_field_group')):
            acf_add_local_field_group(array(
                'key' => 'key_sportshub_recaps_contents',
                'title' => 'Sports Hub Recaps Contents',
                'fields' => array(
                    array(
                        'key' => 'key_shr_recaps_content_status',
                        'label' => 'Status',
                        'name' => 'shr_content_status',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'inactive' => 'Inactive',
                            'active' => 'Active',
                        ),
                        'default_value' => 'inactive',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_shr_recaps_content_validate_content',
                        'label' => 'Check if post is duplicated by text search on title and content (heavy impact on DB)',
                        'name' => 'shr_content_validate_content',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '80',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            '0' => 'No',
                            '1' => 'Yes',
                        ),
                        'default_value' => '0',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_shr_recaps_content_url',
                        'label' => 'URL',
                        'name' => 'shr_content_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_content_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_shr_recaps_content_token',
                        'label' => 'Token',
                        'name' => 'shr_content_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_content_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_shr_recaps_content_types_members',
                        'label' => 'Content Types Members',
                        'name' => 'shr_content_types_members',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_content_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_shr_recaps_content_type_slug_members',
                                'label' => 'Type Slug',
                                'name' => 'shr_content_type_slug_members',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_shr_recaps_content_type_block_title_members',
                            ),
                            array(
                                'key' => 'key_shr_recaps_content_type_block_title_members',
                                'label' => 'Block Game Links Title',
                                'name' => 'shr_content_type_block_title_members',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_shr_recaps_content_types_members',
                            ),
                            array(
                                'key' => 'key_shr_recaps_content_type_block_icon_members',
                                'label' => 'Icon Block Game Links',
                                'name' => 'shr_content_type_block_icon_members',
                                'aria-label' => '',
                                'type' => 'image',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'acfe_save_meta' => 0,
                                'uploader' => '',
                                'return_format' => 'array',
                                'upload_folder' => '',
                                'acfe_thumbnail' => 0,
                                'min_width' => '',
                                'min_height' => '',
                                'min_size' => '',
                                'max_width' => '',
                                'max_height' => '',
                                'max_size' => '',
                                'mime_types' => '',
                                'preview_size' => 'medium',
                                'library' => 'all',
                            ),
                            array(
                                'key' => 'key_shr_recaps_content_type_title_members',
                                'label' => 'Content Heading Title',
                                'name' => 'shr_content_type_title_members',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '40',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_shr_recaps_content_types_members',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'key_shr_recaps_masterfeed_content_types',
                        'label' => 'Content Types Masterfeed',
                        'name' => 'shr_content_types',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'key_shr_recaps_content_status',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_shr_recaps_masterfeed_content_type_slug',
                                'label' => 'Type Slug',
                                'name' => 'shr_content_type_slug',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_shr_recaps_masterfeed_content_types',
                            ),
                            array(
                                'key' => 'key_shr_recaps_masterfeed_content_type_title',
                                'label' => 'Content Heading Title',
                                'name' => 'shr_content_type_title',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'key_shr_recaps_masterfeed_content_types',
                            ),
                        ),
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'sh-recaps-contents',
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            endif;    
    }

    public function save_options_contents_types() {
        try {
            $option = (isset($_GET['page'])) ? $_GET['page'] : null;

            if (!empty($option) && $option == 'sh-recaps-contents' && isset($_POST['acf']) && is_array($_POST['acf']) && count($_POST['acf']) > 0) {
                $helpers = New Sportshub_Recaps_Helpers;
                $helpers->setRedisGroup(REDIS_GROUP_SHRECAPS);

                if ($helpers->initRedis()) {
                    try {
                        $key_sports_contents = $helpers->getRedisKeySportsContents();
                        $key_sports_contents_members = $helpers->getRedisKeySportsContentsMembers();
                        $helpers->setRedis($key_sports_contents, serialize([]), SHRECAPS_REDIS_TIMEOUT);
                        $helpers->setRedis($key_sports_contents_members, serialize([]), SHRECAPS_REDIS_TIMEOUT);
                    } catch (\Exception $exception) {
                        error_log(print_r([
                            'message' => 'Sportshub Recaps: '.$exception->getMessage(),
                            'keys' => [$key_sports_contents, $key_sports_contents_members],
                        ], true));
                    }
                } else {
                    error_log(print_r([
                        'message' =>  'Sportshub Recaps: Redis activation is necessary to store the sports contents cache.',
                    ], true));
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'Sportshub Recaps: '.$exception->getMessage(),
                'option' => $option,
            ], true));
        }
    }

    public function save_options_sportshub_recaps() {
        try {
            $option = (isset($_GET['page'])) ? $_GET['page'] : null;

            if ($option != 'sh-recaps-settings') {
                return;
            }

            if (isset($_POST['acf']['key_shr_recaps_status']) && $_POST['acf']['key_shr_recaps_status'] == 'active') {
                global $shr;

                $fields = $_POST['acf'];
                $cpt_status = $fields['key_shr_recaps_status'];
                $cpt_range_days = $fields['key_shr_recaps_range_days'];
                $cpt_source = $fields['key_shr_recaps_data_source'];
                $cron = $fields['key_shr_recaps_cron_time'];
                $reset = $fields['key_shr_recaps_last_run'];
                $styles = $fields['key_shr_recaps_blocks_load_colors_styles'];

                $set_option_status = (!empty($cpt_status) && $cpt_status == 'active') ? true : false;
                $set_option_source = (!empty($cpt_source)) ? $cpt_source : 'members';

                update_option('shr_status', $set_option_status);
                update_option('shr_source', $set_option_source, 'members');
                update_option('shr_slug', $shr->get_slug(false));

                if (!empty($cpt_range_days) && $cpt_range_days >= 1) {
                    update_option('shr_recaps_range_days', absint($cpt_range_days));
                }

                $original_cron = get_option('shr_recaps_cron_time');

                if ((!empty($cron)) && ($original_cron != $cron)) {
                    wp_clear_scheduled_hook('shr_recaps_cron_event');
                    wp_schedule_event(time(), $cron, 'shr_recaps_cron_event');
                }

                update_option('shr_recaps_cron_time', $cron);

                if (!empty($reset) && $reset == 'reset') {
                    delete_option('shr_recaps_last_run');
                }

                $this->loadColorsAndStyles($styles);
            } else {
                // Set inactive plugin
                update_option('shr_status', false);
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'Sportshub Recaps: '.$exception->getMessage(),
                'option' => $option,
            ], true));
        }
    }

    public function loadColorsAndStyles($styles) {
        try {
            global $shr;
            $shr = (!empty($shr)) ? $shr : Sportshub_Recaps::instance();

            $blocks = $shr->helpers->getBlocksStylesLoad();
            $load_colors_styles = (!empty($styles)) ? $styles : 'none';

            // error_log(print_r([
            //     'load_colors_styles' => $load_colors_styles,
            // ], true));

            if ($load_colors_styles != 'none') {
                if ($shr->helpers->isInstallGeneratePress()) {
                    $generatepress_options = get_option('generate_settings');

                    if (!$generatepress_options) {
                        $generatepress_options = [
                            'global_colors' => $shr->helpers->getDefaultColors(),
                            'typography' => [],
                        ];
                    }

                    if (!isset($generatepress_options['global_colors'])) {
                        $generatepress_options['global_colors'] = [];
                    }

                    $saved_colors = [];
                    if (is_array($generatepress_options['global_colors']) && count($generatepress_options['global_colors']) > 0) {
                        foreach ($generatepress_options['global_colors'] as $key => $color) {
                            $saved_colors[] = $color['slug'];
                        }
                    }
            
                    if (!isset($generatepress_options['typography'])) {
                        $generatepress_options['typography'] = [];
                    }

                    $saved_styles = [];
                    if (is_array($generatepress_options['typography']) && count($generatepress_options['typography']) > 0) {
                        foreach ($generatepress_options['typography'] as $key => $style) {
                            if ($style['selector'] == 'custom') {
                                $saved_styles[] = $style['customSelector'];
                            }
                        }
                    }

                    $base_colors = $shr->helpers->getBaseColors();
                    $generatepress_options = $shr->helpers->registerGeneratePressColors($base_colors, $saved_colors, $generatepress_options);

                    foreach ($blocks as $key => $block) {
                        if (in_array($load_colors_styles, ['colors', 'both'])) {
                            $colors = $block->getColorsToloadGeneratePress();
                            if (is_array($colors) && count($colors) > 0) {
                                $generatepress_options = $shr->helpers->registerGeneratePressColors($colors, $saved_colors, $generatepress_options);
                            }
                        }

                        if (in_array($load_colors_styles, ['styles', 'both'])) {
                            $styles = $block->getStylesToloadGeneratePress();
                            if (is_array($styles) && count($styles) > 0) {
                                $generatepress_options = $shr->helpers->registerGeneratePressStyles($styles, $saved_styles, $generatepress_options);
                            }
                        }
                    }
                } else {
                    foreach ($blocks as $key => $block) {
                        if (in_array($load_colors_styles, ['styles', 'both'])) {
                            $styles = $block->getStylesToloadGeneratePress();
                            if (is_array($styles) && count($styles) > 0) {
                                $shr->helpers->registerStylesCss($block, $styles);
                            }
                        }
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'Sportshub Recaps: '.$exception->getMessage(),
                'load_colors_styles' => $load_colors_styles,
            ], true));
        }
    }
}
