<?php

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * HELPER COMMENT START
 * 
 * This class contains all of the available CLI commands for your plugin. 
 * Down below, we added a command that allows you to display the current 
 * details about your plugin within the console. 
 * 
 * To test the command, please head over to your WordPress instance and type in the following
 * 
 * wp sportshub_games details
 * 
 * HELPER COMMENT END
 */

WP_CLI::add_command('sportshub_games', 'Sportshub_Games_CLI');

/**
 * Class Sportshub_Games_CLI
 *
 * This class contains all WP CLI related features.
 *
 * @package         SHGAMES
 * @subpackage      Classes/Sportshub_Games_CLI
 * @author          IT
 * @since           1.0.0
 */
class Sportshub_Games_CLI extends WP_CLI_Command {
    function __construct() {
    }

    public function details($args, $assoc_args) {
        WP_CLI::line(sprintf(__('Plugin name: %s', 'SportsHub Games'), SHGAMES_NAME));
        WP_CLI::line(sprintf(__('Plugin version: %s', 'SportsHub Games'), SHGAMES_VERSION));
    }

    public function pull($args, $assoc_args)
    {
        if (get_option('shg_status') != 1) {
            error_log(sprintf(__('Sportshub Games: Plugin disabled.', 'SportsHub Games')));
            WP_CLI::line(sprintf(__('Plugin disabled.', 'SportsHub Games')));

            return;
        }

        $author_general_id = 1;
        $author_general = null;
        $sports_authors = get_field('shg_authors', 'options');
        $authors = [];

        if (is_array($sports_authors) && count($sports_authors) > 0) {
            foreach ($sports_authors as $sport => $author) {
                $post_author = get_user_by('ID', $author['shg_sport_user']);

                if (empty($post_author)) continue;

                if ($author['shg_sport'] == 0) {
                    $author_general = $author['shg_sport_user'];
                    continue;
                } else {
                    $authors[$author['shg_sport']] = $author['shg_sport_user'];
                }
            }
        }

        $author_general = (isset($author_general)) ? get_user_by('id', $author_general) : null;
        if (!empty($author_general)) {
            $author_general_id = $author_general->ID;
        }

        $main_games = [];
        $main_game_id = null;
        $sports_main_games = get_field('shg_sports_main_games', 'options');

        if (is_array($sports_main_games) && count($sports_main_games) > 0) {
            foreach ($sports_main_games as $sport => $game) {
                $post_game = get_post($game['shg_game']);
                if (empty($game['shg_game']) || empty($post_game) || 'trash' == $post_game->post_status) continue;

                if ($game['shg_sport'] == 0) {
                    $main_game_id = $game['shg_game'];
                    continue;
                } else {
                    $main_games[$game['shg_sport']] = $game['shg_game'];
                }
            }
        }

        $sports_leagues_parts = [];
        $sports_leagues_data = (is_array(get_field('shgame_sports_and_leagues', 'options')) && count(get_field('shgame_sports_and_leagues', 'options')) > 0)
            ? get_field('shgame_sports_and_leagues', 'options')
            : null;

        if (is_array($sports_leagues_data) && count($sports_leagues_data) > 0) {
            foreach ($sports_leagues_data as $sport_part) {
                if (isset($sport_part['shg_sport_id']) && !empty($sport_part['shg_sport_id']) && isset($sport_part['shg_game_parts']) && !empty($sport_part['shg_game_parts'])) {
                    $sports_leagues_parts[$sport_part['shg_sport_id']] = $sport_part['shg_game_parts'];
                }
            }
        }

        $sports_leagues_array = [];
        $sports_leagues = (is_array(get_field('shg_sports_leagues', 'options')) && count(get_field('shg_sports_leagues', 'options')) > 0)
            ? get_field('shg_sports_leagues', 'options')
            : null;

        if (is_array($sports_leagues) && count($sports_leagues) > 0) {
            foreach ($sports_leagues as $sport) {
                if ($sport['shg_sport_league'] == 0) continue;

                if (isset($sport['shg_sport_league']) && !empty($sport['shg_sport_league'])) {
                    if (strpos($sport['shg_sport_league'], '-')) {
                        list($sport_id, $league_id) = explode('-', $sport['shg_sport_league']);

                        if (!isset($sports_leagues_array[$sport_id])) {
                            $sports_leagues_array[$sport_id] = [];
                            array_push($sports_leagues_array[$sport_id], $league_id);
                        } else {
                            array_push($sports_leagues_array[$sport_id], $league_id);
                        }
                    } else {
                        $league_id = null;
                        $sport_id = $sport['shg_sport_league'];
                        
                        if (!isset($sports_leagues_array[$sport_id])) {
                            $sports_leagues_array[$sport_id] = [];
                        }
                    }
                }
            }
        }

        if (is_array($sports_leagues_array) && count($sports_leagues_array) == 0) {
            $sports_leagues_array[0] = '';
        }

        $filename =  'SPORTS_HUB_GAMES_PULL_'. date('Y-m-d-H-i-s') . '.json';
        $file_path = SHGAMES_PLUGIN_LOG_DIR . '/' . $filename;

        if (!file_exists(SHGAMES_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHGAMES_PLUGIN_LOG_DIR);
        }

        $shg_request = $this->get_api_request($sports_leagues_array, $sports_leagues_parts);

        WP_CLI::line(sprintf(__('Attempting to retrieve data from %s', 'SportsHub Games'), $shg_request));

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $shg_request);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);

        if ($response === false) {
            WP_CLI::line(sprintf(__('ERROR: %s', 'SportsHub Games'), curl_error($ch)));
        } else {
            file_put_contents($file_path, $response);
            WP_CLI::line(sprintf(__('JSON response saved to: %s', 'SportsHub Games'), $file_path));
        }

        curl_close($ch);

        $parse_results = $this->parse_feed_file($file_path);

        if (!empty($parse_results)) {
            $new_post = 0;
            $existing_post = 0;
            $cpt_slug = get_option('shg_slug');

            foreach ($parse_results as $game_picks) {
                $post_title = 'Game Title';
                if (isset($game_picks['visitor_team']) && !empty($game_picks['visitor_team']) && isset($game_picks['home_team']) && !empty($game_picks['visitor_team'])) {
                    $post_title = ucfirst(strtolower($game_picks['visitor_team'])).' vs '.ucfirst(strtolower($game_picks['home_team'])).' Picks and Predictions';
                }

                $title_date = '';

                try {
                    $title_date = date('F jS Y', strtotime($game_picks['game_date']));
                } catch (\Exception $exception) {
                    error_log(print_r([
                        'game_date', $game_picks['game_date'],
                        'exception', $exception->getMessage()
                    ], true));
                }

                if (!empty($title_date)) {
                    $post_title .= ' '.$title_date;
                }

                $args = [
                    'post_title'    => $post_title,
                    'post_type'     => $cpt_slug,
                    'post_status'   => 'publish',
                    'post_author'   => $author_general_id,
                    'post_parent'   => $main_game_id,
                ];

                $post_id = '';
                if (isset($game_picks['id_game'])) {
                    WP_CLI::line(__('External ID: '. $game_picks['id_game'], 'SportsHub Games'));
                    $args['meta_input']['shg_external_id'] = $game_picks['id_game'];
                    $post_id = $game_picks['id_game'];
                }

                $attributes = [
                    'data' => $game_picks,
                    'authors' => $authors,
                    'main_games' => $main_games,
                ];

                $args = $this->parse_data($post_id, $args, $attributes);
                $response = $this->insert_auto_post($args, $cpt_slug);

                if (is_array($response) && isset($response['type'])) {
                    if ($response['type'] == 'new') {
                        $new_post++;
                    } elseif ($response['type'] == 'duplicate') {
                        $existing_post++;
                    } else {
                        WP_CLI::Line($response['message']);
                    }
                }
            }

            error_log(sprintf(__('Sportshub Games: %s Succesfully inserted posts.', 'SportsHub Games'), $new_post));
            WP_CLI::line(sprintf(__('%s Succesfully inserted posts.', 'SportsHub Games'), $new_post));

            if ($existing_post > 0) {
                error_log(sprintf(__('Sportshub Games: %s Existing posts.', 'SportsHub Games'), $existing_post));
                WP_CLI::line(sprintf(__('%s Existing posts.', 'SportsHub Games'), $existing_post));
            }
        } else {
            error_log('Sportshub Games: No data available for this pull.');
            WP_CLI::line(__('No data available for this pull', 'SportsHub Games'));
        }

        update_option('sportshub_last_run', date('Y-m-d H:i:s'));
    }

    public function parse_data($post_id, $args, $attributes = [])
    {
        if (isset($attributes['data']['game_date'])) {
            $args['meta_input']['shg_game_date'] = $attributes['data']['game_date'];
        }

        if (isset($attributes['data']['visitor_team']) && !empty($attributes['data']['visitor_team']) && isset($attributes['data']['home_team']) && !empty($attributes['data']['visitor_team'])) {
            $args['post_name'] = ucfirst($attributes['data']['visitor_team']).' vs '.strtolower($attributes['data']['home_team']).'/'.$post_id;
        }

        if (isset($attributes['data']['sport'])) {
            $args['meta_input']['shg_sport_id'] = $attributes['data']['sport'];

            if (is_array($attributes['authors']) && isset($attributes['authors'][$attributes['data']['sport']]) && !empty($attributes['authors'][$attributes['data']['sport']])) {
                $args['post_author'] = $attributes['authors'][$attributes['data']['sport']];
            }

            if (is_array($attributes['main_games']) && count($attributes['main_games']) > 0) {
                $parent = $this->get_parent_game($attributes);
                if (!empty($parent)) {
                    $args['post_parent'] = $parent;
                }
            }
        }

        if (isset($attributes['data']['league'])) {
            $args['meta_input']['shg_league'] = $attributes['data']['league'];
        }

        if (isset($attributes['data']['category'])) {
            $args['meta_input']['shg_category'] = $attributes['data']['category'];
        }

        if (isset($attributes['data']['game_stat'])) {
            $args['meta_input']['shg_game_stat'] = $attributes['data']['game_stat'];
        }

        if (isset($attributes['data']['game_picks_count'])) {
            $args['meta_input']['shg_game_picks_count'] = $attributes['data']['game_picks_count'];
        }

        if (isset($attributes['data']['game_url'])) {
            $args['meta_input']['shg_game_url'] = $attributes['data']['game_url'];
        }

        if (isset($attributes['data']['home_team']) && $attributes['data']['home_team'] != '') {
            $args['meta_input']['shg_home_team'] = $attributes['data']['home_team'];
        }

        if (isset($attributes['data']['home_score']) && $attributes['data']['home_score'] != '') {
            $args['meta_input']['shg_home_score'] = $attributes['data']['home_score'];
        }

        if (isset($attributes['data']['home_image']) && $attributes['data']['home_image'] != '') {
            $args['meta_input']['shg_home_image'] = $attributes['data']['home_image'];
        }

        if (isset($attributes['data']['visitor_team']) && $attributes['data']['visitor_team'] != '') {
            $args['meta_input']['shg_visitor_team'] = $attributes['data']['visitor_team'];
        }

        if (isset($attributes['data']['visitor_score']) && $attributes['data']['visitor_score'] != '') {
            $args['meta_input']['shg_visitor_score'] = $attributes['data']['visitor_score'];
        }

        if (isset($attributes['data']['visitor_image']) && $attributes['data']['visitor_image'] != '') {
            $args['meta_input']['shg_visitor_image'] = $attributes['data']['visitor_image'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['money_line_home_odds']) && $attributes['data']['current_game_odds'][0]['money_line_home_odds'] != '') {
            $args['meta_input']['shg_current_money_line_home_odds'] = $attributes['data']['current_game_odds'][0]['money_line_home_odds'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['money_line_visitor_odds']) && $attributes['data']['current_game_odds'][0]['money_line_visitor_odds'] != '') {
            $args['meta_input']['shg_current_money_line_visitor_odds'] = $attributes['data']['current_game_odds'][0]['money_line_visitor_odds'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['total_over_points']) && $attributes['data']['current_game_odds'][0]['total_over_points'] != '') {
            $args['meta_input']['shg_current_total_over_points'] = $attributes['data']['current_game_odds'][0]['total_over_points'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['total_over_odds']) && $attributes['data']['current_game_odds'][0]['total_over_odds'] != '') {
            $args['meta_input']['shg_current_total_over_odds'] = $attributes['data']['current_game_odds'][0]['total_over_odds'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['total_under_points']) && $attributes['data']['current_game_odds'][0]['total_under_points'] != '') {
            $args['meta_input']['shg_current_total_under_points'] = $attributes['data']['current_game_odds'][0]['total_under_points'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['total_under_odds']) && $attributes['data']['current_game_odds'][0]['total_under_odds'] != '') {
            $args['meta_input']['shg_current_total_under_odds'] = $attributes['data']['current_game_odds'][0]['total_under_odds'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['spread_home_points']) && $attributes['data']['current_game_odds'][0]['spread_home_points'] != '') {
            $args['meta_input']['shg_current_spread_home_points'] = $attributes['data']['current_game_odds'][0]['spread_home_points'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['spread_home_odds']) && $attributes['data']['current_game_odds'][0]['spread_home_odds'] != '') {
            $args['meta_input']['shg_current_spread_home_odds'] = $attributes['data']['current_game_odds'][0]['spread_home_odds'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['spread_visitor_points']) && $attributes['data']['current_game_odds'][0]['spread_visitor_points'] != '') {
            $args['meta_input']['shg_current_spread_visitor_points'] = $attributes['data']['current_game_odds'][0]['spread_visitor_points'];
        }

        if (isset($attributes['data']['current_game_odds'][0]['spread_visitor_odds']) && $attributes['data']['current_game_odds'][0]['spread_visitor_odds'] != '') {
            $args['meta_input']['shg_current_spread_visitor_odds'] = $attributes['data']['current_game_odds'][0]['spread_visitor_odds'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['money_line_home_odds']) && $attributes['data']['opener_game_odds'][0]['money_line_home_odds'] != '') {
            $args['meta_input']['shg_opener_money_line_home_odds'] = $attributes['data']['opener_game_odds'][0]['money_line_home_odds'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['money_line_visitor_odds']) && $attributes['data']['opener_game_odds'][0]['money_line_visitor_odds'] != '') {
            $args['meta_input']['shg_opener_money_line_visitor_odds'] = $attributes['data']['opener_game_odds'][0]['money_line_visitor_odds'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['total_over_points']) && $attributes['data']['opener_game_odds'][0]['total_over_points'] != '') {
            $args['meta_input']['shg_opener_total_over_points'] = $attributes['data']['opener_game_odds'][0]['total_over_points'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['total_over_odds']) && $attributes['data']['opener_game_odds'][0]['total_over_odds'] != '') {
            $args['meta_input']['shg_opener_total_over_odds'] = $attributes['data']['opener_game_odds'][0]['total_over_odds'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['total_under_points']) && $attributes['data']['opener_game_odds'][0]['total_under_points'] != '') {
            $args['meta_input']['shg_opener_total_under_points'] = $attributes['data']['opener_game_odds'][0]['total_under_points'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['total_under_odds']) && $attributes['data']['opener_game_odds'][0]['total_under_odds'] != '') {
            $args['meta_input']['shg_opener_total_under_odds'] = $attributes['data']['opener_game_odds'][0]['total_under_odds'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['spread_home_points']) && $attributes['data']['opener_game_odds'][0]['spread_home_points'] != '') {
            $args['meta_input']['shg_opener_spread_home_points'] = $attributes['data']['opener_game_odds'][0]['spread_home_points'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['spread_home_odds']) && $attributes['data']['opener_game_odds'][0]['spread_home_odds'] != '') {
            $args['meta_input']['shg_opener_spread_home_odds'] = $attributes['data']['opener_game_odds'][0]['spread_home_odds'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['spread_visitor_points']) && $attributes['data']['opener_game_odds'][0]['spread_visitor_points'] != '') {
            $args['meta_input']['shg_opener_spread_visitor_points'] = $attributes['data']['opener_game_odds'][0]['spread_visitor_points'];
        }

        if (isset($attributes['data']['opener_game_odds'][0]['spread_visitor_odds']) && $attributes['data']['opener_game_odds'][0]['spread_visitor_odds'] != '') {
            $args['meta_input']['shg_opener_spread_visitor_odds'] = $attributes['data']['opener_game_odds'][0]['spread_visitor_odds'];
        }

        return $args;
    }

    public function get_parent_game($attributes) {
        if (isset($attributes['data']['sport']) && !empty($attributes['data']['sport']) && isset($attributes['data']['category']) && !empty($attributes['data']['category'])) {
            $key = $attributes['data']['sport'].'-'.$attributes['data']['category'];

            if (isset($attributes['main_games'][$key])) {
                return $attributes['main_games'][$key];
            }
        }

        if (isset($attributes['data']['sport']) && !empty($attributes['data']['sport'])) {
            $key = $attributes['data']['sport'];

            if (isset($attributes['main_games'][$key])) {
                return $attributes['main_games'][$key];
            }
        }

        return null;
    }

    public function get_api_request($sports_leagues, $parts)
    {
        $api_request = '';
        $api_url = get_field('sportshub_api_url', 'options');
        $api_token = get_field('sportshub_api_token', 'options');
        $date = date('Y-m-d');

        $api_url = (substr($api_url, -1) == '/') ? substr($api_url, 0, strlen($api_url) -1) : $api_url;
        $sports_categories = $this->get_sports_categories_url_params($sports_leagues, $parts);

        $api_url .= '?date='.$date;

        if (!empty($sports_categories) ) {
            $api_url .= '&'.$sports_categories;
        }

        //$api_url .= '&recordsQty=5';
        $api_url .= '&completeGames=1';
        $api_request = $api_url.'&api_token='.$api_token;
        
        return $api_request;
    }

    public function get_sports_categories_url_params($sports_leagues, $parts) {
        if (!is_array($sports_leagues) || count($sports_leagues) == 0) return null;

        $sports = [];
        $categories = [];
        $leagues = [];
        $game_parts = [];
        
        foreach($sports_leagues as $key => $sport) {
            $category_param = 'NULL';
            $league_param = 'NULL';

            if (is_array($sport) && count($sport) > 0) {
                $category_param = $sport[0];
            }

            if (isset($parts[$key])) {
                $game_parts[] = $parts[$key];
            }

            $sports[] = $key;
            $categories[] = $category_param;
            $leagues[] = $league_param;
        }
        
        $params = [
            'sport='.implode(',', $sports),
            'category='.implode(',', $categories),
            'league='.implode(',', $leagues),
            'gamePart='.implode(',', $game_parts),
        ];
        
        return implode('&', $params);
    }

    function insert_auto_post($args, $cpt_slug)
    {
        wp_reset_postdata();
        $helpers = New Sportshub_Games_Helpers;

        $args_search = [
            'post_type' => $cpt_slug,
            'meta_query' => [
                [
                    'key' => 'shg_external_id',
                    'value' => $args['meta_input']['shg_external_id'],
                    'compare' => '=',
                ],
            ],
        ];

        $query = new WP_Query($args_search);

        if (!$query->have_posts()) {
            # Get local team image data
            $home_image = $helpers->getNameFile($args['meta_input']['shg_home_image']);

            if (is_array($home_image) && isset($home_image['is_image']) && $home_image['is_image'] == true) {
                $attachment_home_check = $this->get_attachment('post_title', $home_image['name']);

                if (empty($attachment_home_check)) {
                    # Download the image and upload it to the media library
                    $home_attachment_id = media_sideload_image($home_image['image_url'], 0, $home_image['name'], 'id');

                    # Update image attachment with alt text
                    update_post_meta($home_attachment_id, '_wp_attachment_image_alt', $home_image['alt']);

                    # Get url of already stored image
                    $attachment_home_check = $this->get_attachment('ID', $home_attachment_id);
                    $args['meta_input']['shg_home_local_url'] = $attachment_home_check[0]['guid'];
                } else {
                    $args['meta_input']['shg_home_local_url'] = $attachment_home_check[0]['guid'];
                }
            }

            # Get visitor team image data
            $visitor_image = $helpers->getNameFile($args['meta_input']['shg_visitor_image']);

            if (is_array($visitor_image) && isset($visitor_image['is_image']) && $visitor_image['is_image'] == true) {
                $attachment_visitor_check = $this->get_attachment('post_title', $visitor_image['name']);

                if (empty($attachment_visitor_check)) {
                    # Download the image and upload it to the media library
                    $visitor_attachment_id = media_sideload_image($visitor_image['image_url'], 0, $visitor_image['name'], 'id');

                    # Update image attachment with alt text
                    update_post_meta($visitor_attachment_id, '_wp_attachment_image_alt', $visitor_image['alt']);

                    # Get url of already stored image
                    $attachment_visitor_check = $this->get_attachment('ID', $visitor_attachment_id);
                    $args['meta_input']['shg_visitor_local_url'] = $attachment_visitor_check[0]['guid'];
                } else {
                    $args['meta_input']['shg_visitor_local_url'] = $attachment_visitor_check[0]['guid'];
                }
            }

            # Create CPT Game
            $post_id = wp_insert_post($args);

            if (!is_wp_error($post_id)) {
                return ['type' => 'new', 'post_id' => $post_id, 'message' => ''];
            } else {
                return ['type' => 'error', 'post_id' => null, 'message' => $post_id->get_error_message()];
            }
        } else {
            while ($query->have_posts()) {
                $query->the_post();
                $post_id = get_the_ID();
            }

            WP_CLI::line(__('Post already in the DB, ID: '. $post_id, 'SportsHub Games'));
            return ['type' => 'duplicate', 'post_id' => $post_id, 'message' => ''];
        }
    }

    public function get_attachment($field, $value)
    {
        global $wpdb;
        $value = esc_sql($value);
        $attachment = $wpdb->get_results( "SELECT * FROM $wpdb->posts WHERE $field = '$value' AND post_type = 'attachment'", ARRAY_A);

        if (is_array($attachment)) {
            return $attachment;
        }

        return null;
    }

    function parse_feed_file($file_path)
    {
        $file_contents = file_get_contents($file_path);

        if ($file_contents === false) {
            WP_CLI::line(__('Sportshub Games Error: Failed to read the file.', 'SportsHub Games'));

            return null;
        }

        $json_data = json_decode($file_contents, true);

        if ($json_data === null) {
            WP_CLI::line(__('Sportshub Games Error: Failed to decode the JSON.', 'SportsHub Games'));

            return null;
        }

        return $json_data;
    }
}
