<?php

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * HELPER COMMENT START
 * 
 * This is the main class that is responsible for registering
 * the core functions, including the files and setting up all features. 
 * 
 * To add a new class, here's what you need to do: 
 * 1. Add your new class within the following folder: core/includes/classes
 * 2. Create a new variable you want to assign the class to (as e.g. public $helpers)
 * 3. Assign the class within the instance() function ( as e.g. self::$instance->helpers = new Sportshub_Games_Helpers();)
 * 4. Register the class you added to core/includes/classes within the includes() function
 * 
 * HELPER COMMENT END
 */

if ( ! class_exists( 'Sportshub_Games' ) ) :

    /**
     * Main Sportshub_Games Class.
     *
     * @package         SHGAMES
     * @subpackage      Classes/Sportshub_Games
     * @since           1.0.0
     * @author          IT
     */
    final class Sportshub_Games {

        /**
         * The real instance
         *
         * @access      private
         * @since       1.0.0
         * @var         object|Sportshub_Games
         */
        private static $instance;

        /**
         * SHGAMES helpers object.
         *
         * @access      public
         * @since       1.0.0
         * @var         object|Sportshub_Games_Helpers
         */
        public $helpers;

        /**
         * SHGAMES settings object.
         *
         * @access      public
         * @since       1.0.0
         * @var         object|Sportshub_Games_Settings
         */
        public $settings;

        /**
         * Throw error on object clone.
         *
         * Cloning instances of the class is forbidden.
         *
         * @access      public
         * @since       1.0.0
         * @return      void
         */
        public function __clone() {
            _doing_it_wrong( __FUNCTION__, __( 'You are not allowed to clone this class.', 'sportshub-games' ), '1.0.0' );
        }

        /**
         * Disable unserializing of the class.
         *
         * @access      public
         * @since       1.0.0
         * @return      void
         */
        public function __wakeup() {
            _doing_it_wrong( __FUNCTION__, __( 'You are not allowed to unserialize this class.', 'sportshub-games' ), '1.0.0' );
        }

        /**
         * Main Sportshub_Games Instance.
         *
         * Insures that only one instance of Sportshub_Games exists in memory at any one
         * time. Also prevents needing to define globals all over the place.
         *
         * @access      public
         * @since       1.0.0
         * @static
         * @return      object|Sportshub_Games    The one true Sportshub_Games
         */
        public static function instance() {
            if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Sportshub_Games ) ) {
                self::$instance                    = new Sportshub_Games;
                self::$instance->base_hooks();
                self::$instance->includes();
                self::$instance->helpers        = new Sportshub_Games_Helpers();
                self::$instance->settings        = new Sportshub_Games_Settings();

                //Fire the plugin logic
                new Sportshub_Games_Run();

                /**
                 * Fire a custom action to allow dependencies
                 * after the successful plugin setup
                 */
                do_action( 'SHGAMES/plugin_loaded' );
            }

            return self::$instance;
        }

        /**
         * Include required files.
         *
         * @access  private
         * @since   1.0.0
         * @return  void
         */
        private function includes() {
            require_once SHGAMES_PLUGIN_DIR . 'core/includes/classes/class-sportshub-games-helpers.php';
            require_once SHGAMES_PLUGIN_DIR . 'core/includes/classes/class-sportshub-games-settings.php';
            if ( defined( 'WP_CLI' ) && WP_CLI ){
                require_once SHGAMES_PLUGIN_DIR . 'core/includes/classes/class-sportshub-games-cli.php';
            }

            require_once SHGAMES_PLUGIN_DIR . 'core/includes/classes/class-sportshub-games-run.php';
            require_once SHGAMES_PLUGIN_DIR . 'core/includes/groups/form-groups.php';
        }

        /**
         * Add base hooks for the core functionality
         *
         * @access  private
         * @since   1.0.0
         * @return  void
         */
        private function base_hooks() {
            add_action( 'plugins_loaded', array( self::$instance, 'load_textdomain' ) );
        }

        /**
         * Loads the plugin language files.
         *
         * @access  public
         * @since   1.0.0
         * @return  void
         */
        public function load_textdomain() {
            load_plugin_textdomain( 'sportshub-games', FALSE, dirname( plugin_basename( SHGAMES_PLUGIN_FILE ) ) . '/languages/' );
        }

    }

endif; // End if class_exists check.