<?php

// Exit if accessed directly.
if ( ! defined('ABSPATH') ) exit;

/**
 * Class Sportshub_Games_Run
 * Thats where we bring the plugin to life
 * @package     SHGAMES
 * @subpackage  Classes/Sportshub_Games_Run
 * @author      IT
 * @since       1.0.0
 */

class Sportshub_Games_Run {
    /**
     * Our Sportshub_Games_Run constructor 
     * to run the plugin logic.
     * @since 1.0.0
     */

    function __construct() {
        $this->add_hooks();
    }

    /**
     * ######################
     * ###
     * #### WORDPRESS HOOKS
     * ###
     * ######################
     */

    /**
     * Registers all WordPress and plugin related hooks
     * @access  private
     * @since   1.0.0
     * @return  void
     */

    private function add_hooks() {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_backend_scripts_and_styles'], 20);
        add_action('admin_menu', [$this, 'register_sportshub_games_menu'], 20);
        add_action('acf/init', [$this, 'generate_form_sportshub_games'], 20);
        add_action('acf/init', [$this, 'generate_form_sportshub_games_contents'], 20);
        add_action('acf/init', [$this, 'generate_form_sportshub_sports_leagues']);
        add_action('acf/save_post', [$this, 'save_options_sports_leagues']);
        add_action('acf/save_post', [$this, 'save_options_sportshub_games']);
        add_action('acf/save_post', [$this, 'save_options_contents_types']);
        add_filter('cron_schedules', [$this,'sportshub_add_cron_interval']);
        add_action('sportshub_cron_event', [$this, 'sportshub_cron_function']);
        register_activation_hook(SHGAMES_PLUGIN_FILE, [$this, 'activation_hook_callback']);
        register_deactivation_hook(SHGAMES_PLUGIN_FILE, [$this, 'deactivation_hook_callback']);

        if (empty(get_option('shg_source')) || get_option('shg_source') == 'members') {
            add_action('acf/init', [$this, 'generate_form_cpt_game']);
        } else {
            add_action('acf/init', [$this, 'generate_form_cpt_game_simple']);
        }

        add_filter('acf/load_value/name=shg_reset_last_run', [$this, 'load_field_shg_reset_last_run'], 10, 3);

        if (get_option('shg_status') == 1) {
            add_action('init', [$this, 'add_custom_post_type'], 10);
        }
        
        //add_action('plugin_action_links_' . SHGAMES_PLUGIN_BASE, [$this, 'add_plugin_action_link'], 20);
    }

    public function enqueue_backend_scripts_and_styles() {
        wp_register_script('shgames_backend_scripts', SHGAMES_PLUGIN_URL.'core/includes/assets/js/backend-scripts.js', [], SHGAMES_VERSION, true);
        wp_enqueue_script('shgames_backend_scripts');
    }

    public function load_field_shg_reset_last_run($value) {
        if ($value == 'reset') {
            update_field('shg_reset_last_run', 'default');
        }

        return 'default';
    }

    public function sportshub_add_cron_interval($schedules) {
        if(!array_key_exists("hourly", $schedules)) {
            $schedules['hourly'] = array(
                'interval' => 60 * 60,
                'display'  => __('Every Hour'),
            );
        }

        if(!array_key_exists("twicedaily", $schedules)) {
            $schedules['twicedaily'] = array(
                'interval' => 60 * 60 * 12,
                'display'  => __('Every 12 hours'),
            );
        }

        if(!array_key_exists("daily", $schedules)) {
            $schedules['daily'] = array(
                'interval' => 60 * 60 * 24,
                'display'  => __('Every Day'),
            );
        }

        if(!array_key_exists("weekly", $schedules)) {
            $schedules['weekly'] = array(
                'interval' => 60 * 60 * 24 * 7,
                'display'  => __('Once Weekly'),
            );
        }

        return $schedules;
    }

    public function sportshub_cron_function() {
        // Call the WP-CLI command
        $command = 'wp sportshub_games pull';
        $output = shell_exec($command);
    }

    public function add_plugin_action_link($links) {
        $links['our_shop'] = sprintf('<a href="%s" title="Custom Link" style="font-weight:700;">%s</a>', 'https://test.test', __('Custom Link', 'sportshub-games') );

        return $links;
    }

    public function register_sportshub_games_menu() {
        acf_add_options_page([
            'page_title'    => __('Sports Hub Games Settings'),
            'menu_title'    => __('Sports Hub Games'),
            'menu_slug'     => SHGAMES_NAME_SLUG,
            'capability'    => 'edit_posts',
            'icon_url'      => 'dashicons-games',
            'redirect'      => true,
        ]);

        acf_add_options_page([
            'page_title'  => __('Sports and Leagues'),
            'menu_title'  => __('Sports and Leagues'),
            'parent_slug' => SHGAMES_NAME_SLUG,
        ]);

        acf_add_options_page([
            'page_title'  => __('Sports Hub Games Contents'),
            'menu_title'  => __('Sports Contents'),
            'parent_slug' => SHGAMES_NAME_SLUG,
        ]);

        acf_add_options_page([
            'page_title'  => __('Sports Hub Games Settings'),
            'menu_title'  => __('Sports Settings'),
            'parent_slug' => SHGAMES_NAME_SLUG,
        ]);
    }

    public function add_custom_post_type() {
        $labels = [
            'name'               => __('Previews', 'sportshub-games'),
            'singular_name'      => __('Game', 'sportshub-games'),
            'menu_name'          => __('Games Previews', 'sportshub-games'),
            'all_items'          => __('All Games', 'sportshub-games'),
            'search_items'       => __('Search Games', 'sportshub-games'),
            'not_found_in_trash' => __('No games found in Trash.', 'sportshub-games'),
            'not_found'          => __('No games found.', 'sportshub-games'),
        ];

        $supports = [
            'title',
            'editor',
            'author',
            'thumbnail',
            'excerpt',
            'custom-fields',
            'revisions',
            'page-attributes'
        ];
         
        $args = [
            'labels'            => $labels,
            'supports'          => $supports,
            'public'            => true,
            'has_archive'       => true,
            'show_in_rest'      => true,
            'hierarchical'      => true,
            'capability_type'   => 'post',
        ];

        register_post_type(get_option('shg_slug'), $args);
    }

    public function activation_hook_callback() {
        if (!wp_next_scheduled('sportshub_cron_event')) {
            wp_schedule_event(time(), 'twicedaily', 'sportshub_cron_event');
            update_option('sportshub_cron_time', 'twicedaily');
        }
    }

    public function deactivation_hook_callback() {
        wp_clear_scheduled_hook('sportshub_cron_event');
    }

    public function generate_form_sportshub_games() {
        $helpers = New Sportshub_Games_Helpers;
        $helpers->setRedisGroup(REDIS_GROUP_SHGAMES);
        $sports = $helpers->getSporstLeaguesOptions(true);
        $sports_masterfeeds = $helpers->getSporstLeaguesMasterfeedsOptions(true);

        if (is_array($sports)) {
            $sports = ['0' => 'All Sports'] + $sports;
        }

        if (is_array($sports_masterfeeds)) {
            $sports_masterfeeds = ['0' => 'All Sports'] + $sports_masterfeeds;
        }

        if(function_exists('acf_add_local_field_group')):
            acf_add_local_field_group(array(
                'key' => 'group_6g76687g682',
                'title' => 'Sports Hub Games Settings',
                'fields' => array(
                    array(
                        'key' => 'field_64d565794ac4f',
                        'label' => 'Status',
                        'name' => 'shg_status',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'inactive' => 'Inactive',
                            'active' => 'Active',
                        ),
                        'default_value' => 'inactive',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'field_64d5657911111',
                        'label' => 'Data Source',
                        'name' => 'shg_data_source',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'members' => 'Members',
                            'masterfeed' => 'Masterfeed',
                        ),
                        'default_value' => 'members',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'field_sdww56565f43eq23swrt4',
                        'label' => 'Debug Blocks Games Active?',
                        'name' => 'shg_debug_block_games',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'none' => 'None',
                            'active' => 'Active',
                        ),
                        'default_value' => 'none',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'shg_field_646e9a1f84fe1',
                        'label' => 'Cronjob run time',
                        'name' => 'shg_cron_time',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '15',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'hourly' => 'Once Hourly',
                            'twicedaily' => 'Twice Daily',
                            'daily' => 'Once Daily',
                            'weekly' => 'Once Weekly',
                        ),
                        'default_value' => 'twicedaily',
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'field_64c95bea5809b',
                        'label' => 'Last Sports Pull Datetime',
                        'name' => 'shg_last_run_message',
                        'aria-label' => '',
                        'type' => 'message',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '15',
                            'class' => '',
                            'id' => '',
                        ),
                        'message' => get_option('sportshub_last_run', 'never'),
                        'new_lines' => 'wpautop',
                        'esc_html' => 0,
                    ),
                    array(
                        'key' => 'field_6e4tg34t4th34tf34t43',
                        'label' => 'URL Teams',
                        'name' => 'shg_teams_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_6e5456y43tdrr23r5r34gr34',
                        'label' => 'URL Games',
                        'name' => 'shg_games_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_64c3e7608071c',
                        'label' => 'URL Members',
                        'name' => 'sportshub_api_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'members',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_64c3e7748071d',
                        'label' => 'Token Members',
                        'name' => 'sportshub_api_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'members',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_64c3ed32424324324',
                        'label' => 'URL Masterfeed',
                        'name' => 'sportshub_api_masterfeed_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'masterfeed',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_64c2dryrsdrrsdr',
                        'label' => 'Token Masterfeed',
                        'name' => 'sportshub_api_masterfeed_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'masterfeed',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_646cfeef60fcf',
                        'label' => 'Sports/Leagues Members',
                        'name' => 'shg_sports_leagues',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'members',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_646cff2460fd1',
                                'label' => 'League',
                                'name' => 'shg_sport_league',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '100',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $helpers->getSporstLeaguesOptions(true),
                                'default_value' => 1,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_646cfeef60fcf',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'shg_field_646cfew5f3r34dd3r34r',
                        'label' => 'Sports/Leagues Masterfeed',
                        'name' => 'shg_sports_leagues_masterfeed',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'masterfeed',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_646wf3rr3f2rrffd1',
                                'label' => 'League',
                                'name' => 'shg_sport_league_masterfeed',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '100',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $helpers->getSporstLeaguesMasterfeedsOptions(true),
                                'default_value' => 1,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_646cfew5f3r34dd3r34r',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'shg_field_64c158c571784',
                        'label' => 'Sports / Authors',
                        'name' => 'shg_authors',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'members',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_64c95a091857e',
                                'label' => 'Sport',
                                'name' => 'shg_sport',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_64c158c571784',
                            ),
                            array(
                                'key' => 'shg_field_64c15afbee6ca',
                                'label' => 'Author',
                                'name' => 'shg_sport_user',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $this->get_authors(),
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_64c158c571784',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'shg_field_6445cdffggfsdfdfhdfhdg',
                        'label' => 'Sports / Authors',
                        'name' => 'shg_authors_masterfeed',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'masterfeed',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_fhjfhgweghbdbdf',
                                'label' => 'Sport',
                                'name' => 'shg_sport_masterfeed',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports_masterfeeds,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_6445cdffggfsdfdfhdfhdg',
                            ),
                            array(
                                'key' => 'shg_field_64fhjfghrgrggdg',
                                'label' => 'Author',
                                'name' => 'shg_sport_user_masterfeed',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $this->get_authors(),
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_6445cdffggfsdfdfhdfhdg',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'field_64c95c1fafc97',
                        'label' => 'Sports / Main Games',
                        'name' => 'shg_sports_main_games',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'members',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'field_64c95c1fafc98',
                                'label' => 'Sport',
                                'name' => 'shg_sport',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'field_64c95c1fafc97',
                            ),
                            array(
                                'key' => 'field_64c95c1fafc99',
                                'label' => 'Game',
                                'name' => 'shg_game',
                                'aria-label' => '',
                                'type' => 'post_object',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'post_type' => array(
                                    0 => get_option('shg_slug'),
                                ),
                                'post_status' => '',
                                'taxonomy' => '',
                                'return_format' => 'id',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 1,
                                'parent_repeater' => 'field_64c95c1fafc97',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'field_64c95e46g75h54drr32',
                        'label' => 'Sports/Main Games',
                        'name' => 'shg_sports_main_games_masterfeed',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d5657911111',
                                    'operator' => '==',
                                    'value' => 'masterfeed',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'field_er5byn56u4tc3rrbnfs',
                                'label' => 'Sport',
                                'name' => 'shg_sport_masterfeed',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports_masterfeeds,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'field_64c95e46g75h54drr32',
                            ),
                            array(
                                'key' => 'field_64erb44cr43rrrcr',
                                'label' => 'Game',
                                'name' => 'shg_game_masterfeed',
                                'aria-label' => '',
                                'type' => 'post_object',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'post_type' => array(
                                    0 => get_option('shg_slug'),
                                ),
                                'post_status' => '',
                                'taxonomy' => '',
                                'return_format' => 'id',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 1,
                                'parent_repeater' => 'field_64c95e46g75h54drr32',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'shg_field_646ec0e1bb72a',
                        'label' => 'Reset Last Run',
                        'name' => 'shg_reset_last_run',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'default' => 'Default',
                            'reset' => 'Reset',
                        ),
                        'default_value' => '',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-sports-settings',
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            endif;    
    }

    public function generate_form_sportshub_sports_leagues() {
        if(function_exists('acf_add_local_field_group') ):
            acf_add_local_field_group(array(
                'key' => 'group_63245d37g78678',
                'title' => 'Sports and Leagues Members',
                'fields' => array(
                    array(
                        'key' => 'field_64b0165634r3r34r3422',
                        'label' => '',
                        'name' => 'shgame_sports_and_leagues',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'field_64b0169de0de2uyiu',
                                'label' => 'Sport ID',
                                'name' => 'shg_sport_id',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_64b0165634r3r34r3422',
                            ),
                            array(
                                'key' => 'field_64b01799e0de3kuyy',
                                'label' => 'Sport Title',
                                'name' => 'shg_sport_title',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_64b0165634r3r34r3422',
                            ),
                            array(
                                'key' => 'field_64b01799e0de3111',
                                'label' => 'Game Part',
                                'name' => 'shg_game_parts',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_64b0165634r3r34r3422',
                            ),
                            array(
                                'key' => 'field_64b017c4e0de422222',
                                'label' => 'Leagues',
                                'name' => 'shg_leagues',
                                'aria-label' => '',
                                'type' => 'repeater',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'layout' => 'table',
                                'min' => 0,
                                'max' => 0,
                                'collapsed' => '',
                                'button_label' => 'Add Row',
                                'rows_per_page' => 20,
                                'sub_fields' => array(
                                    array(
                                        'key' => 'field_64b017d3e0de56566',
                                        'label' => 'League ID',
                                        'name' => 'shg_league_id',
                                        'aria-label' => '',
                                        'type' => 'text',
                                        'instructions' => '',
                                        'required' => 0,
                                        'conditional_logic' => 0,
                                        'wrapper' => array(
                                            'width' => '30',
                                            'class' => '',
                                            'id' => '',
                                        ),
                                        'default_value' => '',
                                        'maxlength' => '',
                                        'placeholder' => '',
                                        'prepend' => '',
                                        'append' => '',
                                        'parent_repeater' => 'field_64b017c4e0de422222',
                                    ),
                                    array(
                                        'key' => 'field_64b017e1e0de656551',
                                        'label' => 'League Title',
                                        'name' => 'shg_league_title',
                                        'aria-label' => '',
                                        'type' => 'text',
                                        'instructions' => '',
                                        'required' => 0,
                                        'conditional_logic' => 0,
                                        'wrapper' => array(
                                            'width' => '80',
                                            'class' => '',
                                            'id' => '',
                                        ),
                                        'default_value' => '',
                                        'maxlength' => '',
                                        'placeholder' => '',
                                        'prepend' => '',
                                        'append' => '',
                                        'parent_repeater' => 'field_64b017c4e0de422222',
                                    ),
                                ),
                                'parent_repeater' => 'field_64b0165634r3r34r3422',
                            ),
                        ),
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-sports-and-leagues',
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            acf_add_local_field_group(array(
                'key' => 'group_6334f45g643h523f44d24355',
                'title' => 'Sports and Leagues Masterfeeds',
                'fields' => array(
                    array(
                        'key' => 'field_64b05433f5435d334535345',
                        'label' => '',
                        'name' => 'shgame_sports_and_leagues_masterfeed',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'field_64b016sfsfwfrfwefwefiu',
                                'label' => 'Sport ID',
                                'name' => 'shg_sport_id_masterfeed',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_64b05433f5435d334535345',
                            ),
                            array(
                                'key' => 'field_64wrf23rr2fd23r',
                                'label' => 'Sport Title',
                                'name' => 'shg_sport_title_masterfeed',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_64b05433f5435d334535345',
                            ),
                            array(
                                'key' => 'field_6t34t43t3t3tft34t3dt',
                                'label' => 'Game Part',
                                'name' => 'shg_game_parts_masterfeed',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_64b05433f5435d334535345',
                            ),
                            array(
                                'key' => 'field_64tt3v4tt3t3t3t3t4',
                                'label' => 'Leagues',
                                'name' => 'shg_leagues_masterfeed',
                                'aria-label' => '',
                                'type' => 'repeater',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'layout' => 'table',
                                'min' => 0,
                                'max' => 0,
                                'collapsed' => '',
                                'button_label' => 'Add Row',
                                'rows_per_page' => 20,
                                'sub_fields' => array(
                                    array(
                                        'key' => 'field_64by3b43t43c3t43t',
                                        'label' => 'League ID',
                                        'name' => 'shg_league_id_masterfeed',
                                        'aria-label' => '',
                                        'type' => 'text',
                                        'instructions' => '',
                                        'required' => 0,
                                        'conditional_logic' => 0,
                                        'wrapper' => array(
                                            'width' => '30',
                                            'class' => '',
                                            'id' => '',
                                        ),
                                        'default_value' => '',
                                        'maxlength' => '',
                                        'placeholder' => '',
                                        'prepend' => '',
                                        'append' => '',
                                        'parent_repeater' => 'field_64tt3v4tt3t3t3t3t4',
                                    ),
                                    array(
                                        'key' => 'field_6g34t4b4t4b546u43ct',
                                        'label' => 'League Title',
                                        'name' => 'shg_league_title_masterfeed',
                                        'aria-label' => '',
                                        'type' => 'text',
                                        'instructions' => '',
                                        'required' => 0,
                                        'conditional_logic' => 0,
                                        'wrapper' => array(
                                            'width' => '80',
                                            'class' => '',
                                            'id' => '',
                                        ),
                                        'default_value' => '',
                                        'maxlength' => '',
                                        'placeholder' => '',
                                        'prepend' => '',
                                        'append' => '',
                                        'parent_repeater' => 'field_64tt3v4tt3t3t3t3t4',
                                    ),
                                ),
                                'parent_repeater' => 'field_64b05433f5435d334535345',
                            ),
                        ),
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-sports-and-leagues',
                        ),
                    ),
                ),
                'menu_order' => 1,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            endif;    
    }

    public function save_options_sports_leagues() {
        try {
            $option = (isset($_GET['page'])) ? $_GET['page'] : null;

            if (!empty($option) && $option == 'acf-options-sports-and-leagues' && isset($_POST['acf']) && is_array($_POST['acf']) && count($_POST['acf']) > 0) {
                $helpers = New Sportshub_Games_Helpers;
                $helpers->setRedisGroup(REDIS_GROUP_SHGAMES);

                if ($helpers->initRedis()) {
                    try {
                        $key_sports = $helpers->getRedisKeySports();
                        $key_sports_masterfeeds = $helpers->getRedisKeySportsMasterfeeds();
                        $key_sports_leagues = $helpers->getRedisKeySportsLeagues();
                        $key_sports_leagues_masterfeeds = $helpers->getRedisKeySportsLeaguesMasterfeeds();
                        
                        $helpers->setRedis($key_sports, serialize([]), SHGAMES_REDIS_TIMEOUT);
                        $helpers->setRedis($key_sports_masterfeeds, serialize([]), SHGAMES_REDIS_TIMEOUT);
                        $helpers->setRedis($key_sports_leagues, serialize([]), SHGAMES_REDIS_TIMEOUT);
                        $helpers->setRedis($key_sports_leagues_masterfeeds, serialize([]), SHGAMES_REDIS_TIMEOUT);
                    } catch (\Exception $exception) {
                        error_log(print_r([
                            'message' => $exception->getMessage(),
                            'keys' => [$key_sports, $key_sports_masterfeeds, $key_sports_leagues],
                        ], true));
                    }
                } else {
                    error_log(print_r([
                        'message' =>  'Redis activation is necessary to store the sports and leagues cache.',
                    ], true));
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
                'option' => $option,
            ], true));
        }
    }

    public function get_authors() {
        try {
            $users_list = [];
            $roles = ['contributor', 'author', 'editor', 'administrator'];
            $users = get_users(['role__in' => $roles]);

            if (is_array($users) && count($users) > 0) {
                foreach ($users as $user) {
                    if (isset($user->data->user_login) && !empty($user->data->user_login)) {
                        $users_list[$user->data->ID] = $user->data->user_login;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
                'roles' => $roles,
            ], true));
        }

        return $users_list;
    }

    public function save_options_sportshub_games() {
        try {
            $option = (isset($_GET['page'])) ? $_GET['page'] : null;

            if (!empty($option) && $option == 'acf-options-sports-settings' && isset($_POST['acf']) && is_array($_POST['acf']) && count($_POST['acf']) > 0) {
                global $shg;

                $cpt_status = current($_POST['acf']);
                $cpt_source = next($_POST['acf']);
                $debug_game = next($_POST['acf']);
                $cron = next($_POST['acf']);
                $url_teams = next($_POST['acf']);
                $url_games = next($_POST['acf']);
                $url = next($_POST['acf']);
                $token = next($_POST['acf']);
                $sports_leagues = next($_POST['acf']);
                $sports_authors = next($_POST['acf']);
                $sports_main_games = next($_POST['acf']);
                $reset = next($_POST['acf']);

                $set_option_status = (!empty($cpt_status) && $cpt_status == 'active') ? true : false;
                $set_option_source = (!empty($cpt_source)) ? $cpt_source : 'members';

                update_option('shg_status', $set_option_status, true);
                update_option('shg_source', $set_option_source, 'members');
                update_option('shg_slug', $shg->get_slug(false));

                $original_cron = get_option('sportshub_cron_time');

                if ((!empty($cron)) && ($original_cron != $cron)) {
                    wp_clear_scheduled_hook('sportshub_cron_event');
                    wp_schedule_event(time(), $cron, 'sportshub_cron_event');
                }

                update_option('sportshub_cron_time', $cron);

                if (!empty($reset) && $reset == 'reset') {
                    delete_option('sportshub_last_run');
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
                'option' => $option,
            ], true));
        }
    }

    public function generate_form_cpt_game() {
        if( function_exists('acf_add_local_field_group') ):
            acf_add_local_field_group(array(
                'key' => 'group_6406f02fb6ca4123',
                'title' => 'SBG Automatic Games',
                'fields' => array(
                    array(
                        'key' => 'field_64ca95f99bde7',
                        'label' => 'Game External ID',
                        'name' => 'shg_external_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f99f79a',
                        'label' => 'Game Date',
                        'name' => 'shg_game_date',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9a31c8',
                        'label' => 'Sport ID',
                        'name' => 'shg_sport_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9a6ca9',
                        'label' => 'Category',
                        'name' => 'shg_category',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9aa865',
                        'label' => 'League',
                        'name' => 'shg_league',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9ae30c',
                        'label' => 'Game Stat',
                        'name' => 'shg_game_stat',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9b1c9b',
                        'label' => 'Picks Count',
                        'name' => 'shg_game_picks_count',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caab7wrwrwrerr',
                        'label' => 'ID Home Team',
                        'name' => 'shg_id_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caab7637201',
                        'label' => 'Home Team',
                        'name' => 'shg_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caab9037202',
                        'label' => 'Home Score',
                        'name' => 'shg_home_score',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caabaa37203',
                        'label' => 'Home API Image',
                        'name' => 'shg_home_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64cd0bdfc6401',
                        'label' => 'Home Local Image',
                        'name' => 'shg_home_local_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caaaffsfsfssf5a',
                        'label' => 'ID Visitor Team',
                        'name' => 'shg_id_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caaa201035a',
                        'label' => 'Visitor Team',
                        'name' => 'shg_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caaa6b1035b',
                        'label' => 'Visitor Score',
                        'name' => 'shg_visitor_score',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caaa871035c',
                        'label' => 'Visitor API Image',
                        'name' => 'shg_visitor_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64cd0c16c6402',
                        'label' => 'Visitor Local Image',
                        'name' => 'shg_visitor_local_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9b5b77',
                        'label' => 'Game URL',
                        'name' => 'shg_game_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9b96a0',
                        'label' => 'Current Game Odds',
                        'name' => '',
                        'aria-label' => '',
                        'type' => 'accordion',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'open' => 1,
                        'multi_expand' => 1,
                        'endpoint' => 0,
                    ),
                    array(
                        'key' => 'field_64ca95f9bd20f',
                        'label' => 'Money Line Home Odds',
                        'name' => 'shg_current_money_line_home_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9c0cf3',
                        'label' => 'Money Line Visitor Odds',
                        'name' => 'shg_current_money_line_visitor_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9c4764',
                        'label' => 'Total Over Points',
                        'name' => 'shg_current_total_over_points',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9c821e',
                        'label' => 'Total Over Odds',
                        'name' => 'shg_current_total_over_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9cbdca',
                        'label' => 'Total Under Points',
                        'name' => 'shg_current_total_under_points',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9cf7e9',
                        'label' => 'Total Under Odds',
                        'name' => 'shg_current_total_under_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9d3431',
                        'label' => 'Spread Home Points',
                        'name' => 'shg_current_spread_home_points',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9d7134',
                        'label' => 'Spread Home Odds',
                        'name' => 'shg_current_spread_home_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9daac6',
                        'label' => 'Spread Visitor Points',
                        'name' => 'shg_current_spread_visitor_points',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9de694',
                        'label' => 'Spread Visitor Odds',
                        'name' => 'shg_current_spread_visitor_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9b9123',
                        'label' => 'Opener Game Odds',
                        'name' => '',
                        'aria-label' => '',
                        'type' => 'accordion',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'open' => 1,
                        'multi_expand' => 1,
                        'endpoint' => 0,
                    ),
                    array(
                        'key' => 'field_64ca95f9bd20f123',
                        'label' => 'Money Line Home Odds',
                        'name' => 'shg_opener_money_line_home_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9c0cf3123',
                        'label' => 'Money Line Visitor Odds',
                        'name' => 'shg_opener_money_line_visitor_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9c4764123',
                        'label' => 'Total Over Points',
                        'name' => 'shg_opener_total_over_points',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9c821e123',
                        'label' => 'Total Over Odds',
                        'name' => 'shg_opener_total_over_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9cbdca123',
                        'label' => 'Total Under Points',
                        'name' => 'shg_opener_total_under_points',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9cf7e9123',
                        'label' => 'Total Under Odds',
                        'name' => 'shg_opener_total_under_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9d3431123',
                        'label' => 'Spread Home Points',
                        'name' => 'shg_opener_spread_home_points',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9d7134123',
                        'label' => 'Spread Home Odds',
                        'name' => 'shg_opener_spread_home_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9daac6123',
                        'label' => 'Spread Visitor Points',
                        'name' => 'shg_opener_spread_visitor_points',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9de694123',
                        'label' => 'Spread Visitor Odds',
                        'name' => 'shg_opener_spread_visitor_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'post_type',
                            'operator' => '==',
                            'value' => get_option('shg_slug'),
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));
            endif;        
    }

    public function generate_form_cpt_game_simple() {
        if( function_exists('acf_add_local_field_group') ):
            acf_add_local_field_group(array(
                'key' => 'group_6406f02fbe5g643f54433',
                'title' => 'SBG Automatic Games',
                'fields' => array(
                    array(
                        'key' => 'field_64ca353f45f5345',
                        'label' => 'Game External ID',
                        'name' => 'shg_external_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_6tr43t3d343dtr3',
                        'label' => 'Sport ID',
                        'name' => 'shg_sport_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_e5tg34t43tg43t43ft3t',
                        'label' => 'Category',
                        'name' => 'shg_category',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_6erwrwwrwefr',
                        'label' => 'Game Date',
                        'name' => 'shg_game_date',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_ryryy4y4t4t34t',
                        'label' => 'ID Home Team',
                        'name' => 'shg_id_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_435345435r3345r',
                        'label' => 'Home Team',
                        'name' => 'shg_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_643f3f453535g345345',
                        'label' => 'Home City',
                        'name' => 'shg_home_city',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_sedffuhjyu67wessdf',
                        'label' => 'ID Visitor Team',
                        'name' => 'shg_id_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_ghjkui873weterrtyh',
                        'label' => 'Visitor Team',
                        'name' => 'shg_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_ertrb67u887645vsdf',
                        'label' => 'Visitor City',
                        'name' => 'shg_visitor_city',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_fthyr64fctft34t',
                        'label' => 'Game URL',
                        'name' => 'shg_game_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_awq2ceeruty6u45t3r32d',
                        'label' => 'Game API Image',
                        'name' => 'shg_game_api_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_ftuyy7945certgte43tr',
                        'label' => 'Game Local Image',
                        'name' => 'shg_game_local_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'post_type',
                            'operator' => '==',
                            'value' => get_option('shg_slug'),
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));
            endif;        
    }

    public function generate_form_sportshub_games_contents() {
        if(function_exists('acf_add_local_field_group')):
            acf_add_local_field_group(array(
                'key' => 'group_6234f432234',
                'title' => 'Sports Hub Games Contents',
                'fields' => array(
                    array(
                        'key' => 'field_642344324324324',
                        'label' => 'Status',
                        'name' => 'shg_content_status',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'inactive' => 'Inactive',
                            'active' => 'Active',
                        ),
                        'default_value' => 'inactive',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'shg_field_6445f3d47678sac',
                        'label' => 'URL',
                        'name' => 'shg_content_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_642344324324324',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_6ghf23q4bgfh9hgfq23gfd',
                        'label' => 'Token',
                        'name' => 'shg_content_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_642344324324324',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_646cseffw45frdfcf',
                        'label' => 'Content Types',
                        'name' => 'shg_content_types',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_642344324324324',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_6ghfw4f534523gfd',
                                'label' => 'Type Slug',
                                'name' => 'shg_content_type_slug',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'shg_field_646cseffw45frdfcf',
                            ),
                            array(
                                'key' => 'shg_field_6ge34b45y54yjrddfgdfgh',
                                'label' => 'Type Title',
                                'name' => 'shg_content_type_title',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'shg_field_646cseffw45frdfcf',
                            ),
                        ),
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-sports-contents',
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            endif;    
    }

    public function save_options_contents_types() {
        try {
            $option = (isset($_GET['page'])) ? $_GET['page'] : null;

            if (!empty($option) && $option == 'acf-options-sports-contents' && isset($_POST['acf']) && is_array($_POST['acf']) && count($_POST['acf']) > 0) {
                $helpers = New Sportshub_Games_Helpers;
                $helpers->setRedisGroup(REDIS_GROUP_SHGAMES);

                if ($helpers->initRedis()) {
                    try {
                        $key_sports_contents = $helpers->getRedisKeySportsContents();
                        $helpers->setRedis($key_sports_contents, serialize([]), SHGAMES_REDIS_TIMEOUT);
                    } catch (\Exception $exception) {
                        error_log(print_r([
                            'message' => $exception->getMessage(),
                            'keys' => [$key_sports_contents],
                        ], true));
                    }
                } else {
                    error_log(print_r([
                        'message' =>  'Redis activation is necessary to store the sports contents cache.',
                    ], true));
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
                'option' => $option,
            ], true));
        }
    }
}
