<?php

// Exit if accessed directly.
if ( ! defined('ABSPATH') ) exit;

/**
 * Class Sportshub_Games_Run
 * Thats where we bring the plugin to life
 * @package     SHGAMES
 * @subpackage  Classes/Sportshub_Games_Run
 * @author      IT
 * @since       1.0.0
 */

class Sportshub_Games_Run {
    /**
     * Our Sportshub_Games_Run constructor 
     * to run the plugin logic.
     * @since 1.0.0
     */

    function __construct() {
        $this->add_hooks();
    }

    /**
     * ######################
     * ###
     * #### WORDPRESS HOOKS
     * ###
     * ######################
     */

    /**
     * Registers all WordPress and plugin related hooks
     * @access  private
     * @since   1.0.0
     * @return  void
     */

    private function add_hooks() {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts_and_styles'], 20);
        add_action('admin_menu', [$this, 'register_sportshub_games_menu'], 20);
        add_action('acf/init', [$this, 'generate_form_sportshub_games'], 20);
        add_action('acf/init', [$this, 'generate_form_sportshub_games_contents'], 20);
        add_action('acf/save_post', [$this, 'save_options_sportshub_games']);
        add_action('acf/save_post', [$this, 'save_options_contents_types']);
        add_action('save_post', [$this, 'auto_lock_manually_edited_posts'], 10, 3);
        add_filter('cron_schedules', [$this,'sportshub_add_cron_interval']);
        add_action('sportshub_cron_event', [$this, 'sportshub_cron_function']);
        register_activation_hook(SHGAMES_PLUGIN_FILE, [$this, 'activation_hook_callback']);
        register_deactivation_hook(SHGAMES_PLUGIN_FILE, [$this, 'deactivation_hook_callback']);

        if (empty(get_option('shg_source')) || get_option('shg_source') == 'members') {
            add_action('acf/init', [$this, 'generate_form_cpt_game']);
        } else {
            add_action('acf/init', [$this, 'generate_form_cpt_game_simple']);
        }

        add_filter('acf/load_value/name=shg_reset_last_run', [$this, 'load_field_shg_reset_last_run'], 10, 3);
        
        if (get_option('shg_status') == 1) {
            add_action('init', [$this, 'add_custom_post_type'], 10);
        }

        add_filter('mime_types', [$this, 'shg_add_jfif_files'], 10, 3);
        
        //SH-custom DEV-691
        add_action('plugins_loaded', function() {
            if (!defined('RANK_MATH_VERSION') || !function_exists('rank_math')) {
                return;
            }
            if (!has_action('rank_math/vars/register_extra_replacements', [$this, 'shg_register_rankmath_vars'])) {
                add_action('rank_math/vars/register_extra_replacements', [$this, 'shg_register_rankmath_vars'], 9);
            }
        }, 9);
        //SH-custom DEV-691
    }

    //SH-custom DEV-691
    public function shg_register_rankmath_vars() {
            if (!function_exists('rank_math')) return;
            $rank_math_variables = [
                [
                    //custom_ es necesario para evitar los warnings
                    'key' => 'custom_home_team_name',
                    'config' => [
                        'name' => esc_html__('Home Team', 'rank-math'),
                        'description' => esc_html__('Home team name', 'rank-math'),
                        'variable' => 'home_team_name',
                        'example' => 'Bulls',
                    ],
                    'callback' => [$this, 'shg_home_team_name_callback']
                ],
                [
                    'key' => 'custom_visitor_team_name',
                    'config' => [
                        'name' => esc_html__('Visitor Team', 'rank-math'),
                        'description' => esc_html__('Visitor team name', 'rank-math'),
                        'variable' => 'visitor_team_name',
                        'example' => 'Lakers',
                    ],
                    'callback' => [$this, 'shg_visitor_team_name_callback']
                ],
                [
                    'key' => 'custom_sport_team',
                    'config' => [
                        'name' => esc_html__('Sport team', 'rank-math'),
                        'description' => esc_html__('Sport team', 'rank-math'),
                        'variable' => 'sport_team',
                        'example' => 'Basketball',
                    ],
                    'callback' => [$this, 'shg_sport_team_callback']
                ],
                [
                    'key' => 'custom_league_team',
                    'config' => [
                        'name' => esc_html__('League team', 'rank-math'),
                        'description' => esc_html__('League', 'rank-math'),
                        'variable' => 'league_team',
                        'example' => 'NBA',
                    ],
                    'callback' => [$this, 'shg_league_team_callback']
                ],
                [
                    'key' => 'custom_date_team',
                    'config' => [
                        'name' => esc_html__('date team', 'rank-math'),
                        'description' => esc_html__('Date', 'rank-math'),
                        'variable' => 'date_team',
                        'example' => 'March 18, 2024',
                    ],
                    'callback' => [$this, 'shg_date_callback']
                ],
                [
                    'key' => 'custom_league_or_sport_team',
                    'config' => [
                        'name' => esc_html__('league or sport', 'rank-math'),
                        'description' => esc_html__('Sport or League', 'rank-math'),
                        'variable' => 'league_or_sport_team',
                        'example' => 'Basketball|NBA',
                    ],
                    'callback' => [$this, 'shg_league_or_sport_team_callback']
                ],


            ];

            if (!empty($rank_math_variables)) {
                foreach ($rank_math_variables as $variable) {
                    if (!empty($variable['key']) && !empty($variable['config']) && !empty($variable['callback'])) {
                        rank_math_register_var_replacement(
                            $variable['key'],
                            $variable['config'],
                            $variable['callback']
                        );
                    }
                }
            }
        }
    //SH-custom DEV-691

    public function shg_league_or_sport_team_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shg_league_name'][0]) && !empty($meta['shg_league_name'][0])) {
                return $meta['shg_league_name'][0];
            }

            if (isset($meta['shg_sport_name'][0]) && !empty($meta['shg_sport_name'][0])) {
                return $meta['shg_sport_name'][0];
            }
        }

        return '';
    }

    public function shg_date_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shg_game_date'][0]) && !empty($meta['shg_game_date'][0])) {
                try {
                    $d = DateTime::createFromFormat('Y-m-d H:i:s', $meta['shg_game_date'][0]);
                    $date_status = $d && $d->format('Y-m-d H:i:s') == $meta['shg_game_date'][0];

                    if ($date_status) {
                        return $d->format('F j, Y');
                    }
                } catch (\Exception $exception) {
                    //
                }
            }
        }

        return '';
    }

    public function shg_league_team_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shg_league_name'][0]) && !empty($meta['shg_league_name'][0])) {
                return $meta['shg_league_name'][0];
            }
        }

        return '';
    }

    public function shg_sport_team_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shg_sport_name'][0]) && !empty($meta['shg_sport_name'][0])) {
                return $meta['shg_sport_name'][0];
            }
        }

        return '';
    }

    public function shg_visitor_team_name_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shg_visitor_team'][0]) && !empty($meta['shg_visitor_team'][0])) {
                return $meta['shg_visitor_team'][0];
            }
        }

        return '';
    }

    public function shg_home_team_name_callback()
    {
        $post = get_post();
        if (isset($post->ID)) {
            $meta = get_post_meta($post->ID);

            if (isset($meta['shg_home_team'][0]) && !empty($meta['shg_home_team'][0])) {
                return $meta['shg_home_team'][0];
            }
        }

        return '';
    }

    public function shg_add_jfif_files($mimes) {
        $mimes['jfif'] = "image/jpeg";

        return $mimes;
    }

    public function enqueue_scripts_and_styles() {
        $shg = Sportshub_Games::instance();
        $shgames_styles = $shg->helpers->webpack('shgames_styles.css');

        wp_register_script('shgames_scripts', SHGAMES_PLUGIN_URL.'core/includes/assets/js/shgames-scripts.js', [], SHGAMES_VERSION, true);
        wp_register_style('shgames_styles', $shgames_styles, [], SHGAMES_VERSION);
    }

    public function load_field_shg_reset_last_run($value) {
        if ($value == 'reset') {
            update_field('shg_reset_last_run', 'default');
        }

        return 'default';
    }

    public function sportshub_add_cron_interval($schedules) {
        if(!array_key_exists("hourly", $schedules)) {
            $schedules['hourly'] = array(
                'interval' => 60 * 60,
                'display'  => __('Every Hour'),
            );
        }

        if(!array_key_exists("every_2_hours", $schedules)) {
            $schedules['every_2_hours'] = array(
                'interval' => (60 * 60) * 2,
                'display'  => __('Every 2 Hours'),
            );
        }

        if(!array_key_exists("every_3_hours", $schedules)) {
            $schedules['every_3_hours'] = array(
                'interval' => (60 * 60) * 3,
                'display'  => __('Every 3 Hours'),
            );
        }

        if(!array_key_exists("twicedaily", $schedules)) {
            $schedules['twicedaily'] = array(
                'interval' => 60 * 60 * 12,
                'display'  => __('Every 12 hours'),
            );
        }

        if(!array_key_exists("daily", $schedules)) {
            $schedules['daily'] = array(
                'interval' => 60 * 60 * 24,
                'display'  => __('Every Day'),
            );
        }

        if(!array_key_exists("weekly", $schedules)) {
            $schedules['weekly'] = array(
                'interval' => 60 * 60 * 24 * 7,
                'display'  => __('Once Weekly'),
            );
        }

        return $schedules;
    }

    public function sportshub_cron_function() {
        // Call the WP-CLI command
        $command = 'wp sportshub_games pull';
        $output = shell_exec($command);
    }

    public function add_plugin_action_link($links) {
        $links['our_shop'] = sprintf('<a href="%s" title="Custom Link" style="font-weight:700;">%s</a>', 'https://test.test', __('Custom Link', 'sportshub-games') );

        return $links;
    }

    public function register_sportshub_games_menu() {
        acf_add_options_page([
            'page_title'    => __('Sports Hub Games Settings'),
            'menu_title'    => __('Sports Hub Games'),
            'menu_slug'     => SHGAMES_NAME_SLUG,
            'capability'    => 'edit_posts',
            'icon_url'      => 'dashicons-games',
            'redirect'      => true,
        ]);

        acf_add_options_page([
            'page_title'  => __('Sports Hub Games Contents'),
            'menu_title'  => __('Sports Contents'),
            'parent_slug' => SHGAMES_NAME_SLUG,
        ]);

        acf_add_options_page([
            'page_title'  => __('Sports Hub Games Settings'),
            'menu_title'  => __('Sports Settings'),
            'parent_slug' => SHGAMES_NAME_SLUG,
        ]);
    }

    public function add_custom_post_type() {
        global $shg;
        $shg = (!empty($shg)) ? $shg : Sportshub_Games::instance();

        $labels = [
            'name'               => __($shg->slug['label'], 'sportshub-games'),
            'singular_name'      => __('Game', 'sportshub-games'),
            'menu_name'          => __('Games '.$shg->slug['label'], 'sportshub-games'),
            'all_items'          => __('All Games', 'sportshub-games'),
            'search_items'       => __('Search Games', 'sportshub-games'),
            'not_found_in_trash' => __('No games found in Trash.', 'sportshub-games'),
            'not_found'          => __('No games found.', 'sportshub-games'),
        ];

        $supports = [
            'title',
            'editor',
            'author',
            'thumbnail',
            'excerpt',
            'custom-fields',
            'revisions',
            'page-attributes'
        ];

        $taxonomyCategory = get_option('shg_slug') . '_categories';
        $taxonomyTag = get_option('shg_slug') .'_tags';

        $args = [
            'labels'            => $labels,
            'supports'          => $supports,
            'public'            => true,
            'has_archive'       => true,
            'show_in_rest'      => true,
            'hierarchical'      => true,
            'capability_type'   => 'post',
            'taxonomies'        => array($taxonomyCategory, $taxonomyTag),
        ];

        add_rewrite_rule('^'.get_option('shg_slug').'/?$', 'index.php?pagename='.get_option('shg_slug'), 'top');
        add_rewrite_rule('^'.get_option('shg_slug').'/page/([0-9]+)/?$', 'index.php?pagename='.get_option('shg_slug').'&paged=$matches[1]', 'top');

        register_post_type(get_option('shg_slug'), $args);

        register_taxonomy(
            $taxonomyCategory,
            get_option('shg_slug'),
            array(
                'label'         => __('Categories'),
                'singular_name' => __('Category'),
                'public'        => true,
                'hierarchical'  => true,
                'show_in_rest'  => true,
                'rewrite'       => true,
                'query_var'     => true,
            )
        );

        register_taxonomy(
            $taxonomyTag,
            get_option('shg_slug'),
            array(
                'label'         => __('Tags'),
                'singular_name' => __('Tag'),
                'public'        => true,
                'hierarchical'  => false,
                'show_in_rest'  => true,
                'rewrite'       => true,
                'query_var'     => true,
            )
        );
    }

    public function activation_hook_callback() {
        if (!wp_next_scheduled('sportshub_cron_event')) {
            wp_schedule_event(time(), 'twicedaily', 'sportshub_cron_event');
            update_option('sportshub_cron_time', 'twicedaily');
        }
    }

    public function deactivation_hook_callback() {
        wp_clear_scheduled_hook('sportshub_cron_event');
    }

    public function generate_form_sportshub_games() {
        $helpers = New Sportshub_Games_Helpers;
        $helpers->setRedisGroup(REDIS_GROUP_SHGAMES);
        $sports = (function_exists('shMappingSportsLeaguesOption')) ? shMappingSportsLeaguesOption() : [];
        $sports_clear = $sports;

        if (is_array($sports)) {
            $sports = ['0' => 'All Sports'] + $sports;
        }

        if(function_exists('acf_add_local_field_group')):
            acf_add_local_field_group(array(
                'key' => 'group_6g76687g682',
                'title' => 'Sports Hub Games Settings',
                'fields' => array(
                    array(
                        'key' => 'shg_field_shg_status',
                        'label' => 'Status',
                        'name' => 'shg_status',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '14',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'inactive' => 'Inactive',
                            'active' => 'Active',
                        ),
                        'default_value' => 'inactive',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'shg_field_shg_data_source',
                        'label' => 'Data Source',
                        'name' => 'shg_data_source',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '17',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'members' => 'Members',
                            'masterfeed' => 'Masterfeed',
                        ),
                        'default_value' => 'members',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'shg_field_debug_block_games',
                        'label' => 'Debug Blocks Games Active?',
                        'name' => 'shg_debug_block_games',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '15',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'none' => 'None',
                            'active' => 'Active',
                        ),
                        'default_value' => 'none',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'shg_field_status_games',
                        'label' => 'Game Status',
                        'name' => 'shg_status_games',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '14',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'draft' => 'Draft',
                            'publish' => 'Publish',
                        ),
                        'default_value' => 'draft',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'shg_field_cron_time',
                        'label' => 'Cronjob run time',
                        'name' => 'shg_cron_time',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '13',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'hourly' => 'Once Hourly',
                            'every_2_hours' => 'Once Every 2 Hours',
                            'every_3_hours' => 'Once Every 3 Hours',
                            'twicedaily' => 'Twice Daily',
                            'daily' => 'Once Daily',
                            'weekly' => 'Once Weekly',
                        ),
                        'default_value' => 'twicedaily',
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'shg_field_date_range',
                        'label' => 'Days range',
                        'name' => 'shg_range_days',
                        'type' => 'number',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '8',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => 7,
                        'min' => 1,
                    ),
                    array(
                        'key' => 'shg_field_last_run_message',
                        'label' => 'Last Sports Pull Datetime',
                        'name' => 'shg_last_run_message',
                        'aria-label' => '',
                        'type' => 'message',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '14',
                            'class' => '',
                            'id' => '',
                        ),
                        'message' => get_option('sportshub_last_run', 'never'),
                        'new_lines' => 'wpautop',
                        'esc_html' => 0,
                    ),
                    array(
                        'key' => 'shg_field_teams_url',
                        'label' => 'URL Teams',
                        'name' => 'shg_teams_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_games_url',
                        'label' => 'URL Games',
                        'name' => 'shg_games_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_sportshub_api_url',
                        'label' => 'URL Members',
                        'name' => 'sportshub_api_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'shg_field_shg_data_source',
                                    'operator' => '==',
                                    'value' => 'members',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_sportshub_api_token',
                        'label' => 'Token Members',
                        'name' => 'sportshub_api_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'shg_field_shg_data_source',
                                    'operator' => '==',
                                    'value' => 'members',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_sportshub_api_masterfeed_url',
                        'label' => 'URL Masterfeed',
                        'name' => 'sportshub_api_masterfeed_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'shg_field_shg_data_source',
                                    'operator' => '==',
                                    'value' => 'masterfeed',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_sportshub_api_masterfeed_token',
                        'label' => 'Token Masterfeed',
                        'name' => 'sportshub_api_masterfeed_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'shg_field_shg_data_source',
                                    'operator' => '==',
                                    'value' => 'masterfeed',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_sports_leagues',
                        'label' => 'Sports/Leagues',
                        'name' => 'shg_sports_leagues',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'field_shg_sport_league',
                                'label' => 'League',
                                'name' => 'shg_sport_league',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '100',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports_clear,
                                'default_value' => 1,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_sports_leagues',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'shg_field_authors',
                        'label' => 'Sports / Authors',
                        'name' => 'shg_authors',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_sport_authors',
                                'label' => 'Sport',
                                'name' => 'shg_sport',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_authors',
                            ),
                            array(
                                'key' => 'shg_field_sport_user',
                                'label' => 'Author',
                                'name' => 'shg_sport_user',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $this->get_authors(),
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_authors',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'shg_field_sports_main_games',
                        'label' => 'Sports / Main Games',
                        'name' => 'shg_sports_main_games',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_sport_games',
                                'label' => 'Sport',
                                'name' => 'shg_sport',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_sports_main_games',
                            ),
                            array(
                                'key' => 'shg_field_game',
                                'label' => 'Game',
                                'name' => 'shg_game',
                                'aria-label' => '',
                                'type' => 'post_object',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'post_type' => array(
                                    0 => get_option('shg_slug'),
                                ),
                                'post_status' => '',
                                'taxonomy' => '',
                                'return_format' => 'id',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 1,
                                'parent_repeater' => 'shg_field_sports_main_games',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'shg_field_sports_featured_images',
                        'label' => 'Sports / Featured Images',
                        'name' => 'shg_sports_featured_images',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_sport_image',
                                'label' => 'Sport',
                                'name' => 'shg_sport',
                                'aria-label' => '',
                                'type' => 'select',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'choices' => $sports,
                                'default_value' => false,
                                'return_format' => 'value',
                                'multiple' => 0,
                                'allow_null' => 0,
                                'ui' => 0,
                                'ajax' => 0,
                                'placeholder' => '',
                                'parent_repeater' => 'shg_field_sports_featured_images',
                            ),
                            array(
                                'key' => 'shg_field_image',
                                'label' => 'Featued Image',
                                'name' => 'shg_image',
                                'aria-label' => '',
                                'type' => 'image',
                                'instructions' => '',
                                'required' => 0,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'acfe_save_meta' => 0,
                                'uploader' => '',
                                'return_format' => 'id',
                                'upload_folder' => 'sh-games',
                                'acfe_thumbnail' => 0,
                                'min_width' => '',
                                'min_height' => '',
                                'min_size' => '',
                                'max_width' => '',
                                'max_height' => '',
                                'max_size' => '',
                                'mime_types' => '',
                                'preview_size' => 'thumbnail',
                                'library' => 'all',
                                'parent_repeater' => 'shg_field_sports_featured_images',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'shg_field_reset_last_run',
                        'label' => 'Reset Last Run',
                        'name' => 'shg_reset_last_run',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'default' => 'Default',
                            'reset' => 'Reset',
                        ),
                        'default_value' => '',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-sports-settings',
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            endif;    
    }

    public function get_authors() {
        try {
            $users_list = [];
            $roles = ['contributor', 'author', 'editor', 'administrator'];
            $users = get_users(['role__in' => $roles]);

            if (is_array($users) && count($users) > 0) {
                foreach ($users as $user) {
                    if (isset($user->data->user_login) && !empty($user->data->user_login)) {
                        $users_list[$user->data->ID] = $user->data->user_login;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
                'roles' => $roles,
            ], true));
        }

        return $users_list;
    }

    public function save_options_sportshub_games() {
        try {
            $option = (isset($_GET['page'])) ? $_GET['page'] : null;

            if (!empty($option) && $option == 'acf-options-sports-settings' && isset($_POST['acf']) && is_array($_POST['acf']) && count($_POST['acf']) > 0) {
                global $shg;

                $fields = $_POST['acf'];
                $cpt_status = $fields['shg_field_shg_status'];
                $cpt_range_days = $fields['shg_field_date_range'];
                $cpt_source = $fields['shg_field_shg_data_source'];
                $cron = $fields['shg_field_cron_time'];
                $reset = $fields['shg_field_reset_last_run'];

                $set_option_status = (!empty($cpt_status) && $cpt_status == 'active') ? true : false;
                $set_option_source = (!empty($cpt_source)) ? $cpt_source : 'members';

                update_option('shg_status', $set_option_status, true);
                update_option('shg_source', $set_option_source, 'members');
                update_option('shg_slug', $shg->get_slug(false));

                if (!empty($cpt_range_days) && $cpt_range_days >= 1) {
                    update_option('sportshub_range_days', absint($cpt_range_days));
                }

                $original_cron = get_option('sportshub_cron_time');

                if ((!empty($cron)) && ($original_cron != $cron)) {
                    wp_clear_scheduled_hook('sportshub_cron_event');
                    wp_schedule_event(time(), $cron, 'sportshub_cron_event');
                }

                update_option('sportshub_cron_time', $cron);

                if (!empty($reset) && $reset == 'reset') {
                    delete_option('sportshub_last_run');
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
                'option' => $option,
            ], true));
        }
    }

    /**
     * Auto-lock published posts when manually edited to prevent CLI overwrites
     * This method sets a meta field when a published/pending post is manually edited
     * The WP-CLI pull command will skip posts with this flag set
     * 
     * @param int $post_id The post ID
     * @param WP_Post $post The post object
     * @param bool $update Whether this is an existing post being updated
     * @since 1.1.38
     */
    public function auto_lock_manually_edited_posts($post_id, $post, $update) {
        // 1. Ignore autosaves and revisions
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        if (wp_is_post_revision($post_id)) {
            return;
        }
        
        // 2. Only apply to the game post type
        $cpt_slug = get_option('shg_slug');
        if (empty($cpt_slug) || get_post_type($post_id) !== $cpt_slug) {
            return;
        }
        
        // 3. Skip if being updated via WP-CLI (our pull command)
        if (defined('WP_CLI') && WP_CLI) {
            return;
        }
        
        // 4. Skip if the lock is already set (no need to keep updating)
        if (get_post_meta($post_id, 'shg_manually_edited', true) === '1') {
            return;
        }
        
        // 5. Only set lock if post is published or pending and being updated (not new)
        if ($update && in_array($post->post_status, ['publish', 'pending'])) {
            update_post_meta($post_id, 'shg_manually_edited', '1');
            
            // Log the action for debugging
            error_log(sprintf(
                'SH Games: Post ID %d has been locked from automatic CLI updates (manually edited)',
                $post_id
            ));
        }
    }

    public function generate_form_cpt_game() {
        if( function_exists('acf_add_local_field_group') ):
            acf_add_local_field_group(array(
                'key' => 'group_6406f02fb6ca4123',
                'title' => 'SBG Automatic Games',
                'fields' => array(
                    array(
                        'key' => 'field_64ca95f99bde7',
                        'label' => 'Game External ID',
                        'name' => 'shg_external_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_manually_edited_lock',
                        'label' => 'Lock from Auto-Updates',
                        'name' => 'shg_manually_edited',
                        'aria-label' => '',
                        'type' => 'true_false',
                        'instructions' => 'When enabled, this post will not be updated by the WP-CLI pull command. Automatically enabled when you edit a published/pending post.',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'message' => '',
                        'default_value' => 0,
                        'ui' => 1,
                        'ui_on_text' => 'Locked',
                        'ui_off_text' => 'Unlocked',
                    ),
                    array(
                        'key' => 'field_64ca95f99f79a',
                        'label' => 'Game Date',
                        'name' => 'shg_game_date',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_qwddqwdwqd2d23r3r32455r',
                        'label' => 'Links Block',
                        'name' => 'shg_game_links_block',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'acfe_save_meta' => 0,
                        'choices' => array(
                            'show' => 'Show',
                            'hide' => 'Hide',
                        ),
                        'default_value' => 'hide',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'field_64ca95f9a31c8',
                        'label' => 'Sport ID',
                        'name' => 'shg_sport_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64cqwe234f234f423',
                        'label' => 'Sport Name',
                        'name' => 'shg_sport_name',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9aa865',
                        'label' => 'League ID',
                        'name' => 'shg_league',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64rte3f44353d435',
                        'label' => 'League Name',
                        'name' => 'shg_league_name',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '15',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9b1c9b',
                        'label' => 'Picks Count',
                        'name' => 'shg_game_picks_count',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caab7wrwrwrerr',
                        'label' => 'ID Home Team',
                        'name' => 'shg_id_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caab7637201',
                        'label' => 'Home Team',
                        'name' => 'shg_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caab9037202',
                        'label' => 'Home Score',
                        'name' => 'shg_home_score',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caabaa37203',
                        'label' => 'Home API Image',
                        'name' => 'shg_home_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64cd0bdfc6401',
                        'label' => 'Home Local Image',
                        'name' => 'shg_home_local_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caaaffsfsfssf5a',
                        'label' => 'ID Visitor Team',
                        'name' => 'shg_id_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caaa201035a',
                        'label' => 'Visitor Team',
                        'name' => 'shg_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caaa6b1035b',
                        'label' => 'Visitor Score',
                        'name' => 'shg_visitor_score',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64caaa871035c',
                        'label' => 'Visitor API Image',
                        'name' => 'shg_visitor_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64cd0c16c6402',
                        'label' => 'Visitor Local Image',
                        'name' => 'shg_visitor_local_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_64ca95f9b5b77',
                        'label' => 'Game URL',
                        'name' => 'shg_game_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_62re23rr23r23r2',
                        'label' => 'Top Handicapper',
                        'name' => '',
                        'aria-label' => '',
                        'type' => 'accordion',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'open' => 1,
                        'multi_expand' => 1,
                        'endpoint' => 0,
                    ),
                    array(
                        'key' => 'field_top_handicapper_001',
                        'label' => 'ID',
                        'name' => 'shg_game_top_handicapper_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '10',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_top_handicapper_002',
                        'label' => 'Name',
                        'name' => 'shg_game_top_handicapper_name',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_top_handicapper_003',
                        'label' => 'API Image',
                        'name' => 'shg_game_top_handicapper_api_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '35',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_top_handicapper_004',
                        'label' => 'Local Image',
                        'name' => 'shg_game_top_handicapper_local_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '35',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_top_handicapper_005',
                        'label' => 'Description',
                        'name' => 'shg_game_top_handicapper_description',
                        'aria-label' => '',
                        'type' => 'textarea',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'acfe_textarea_code' => 0,
                        'maxlength' => '',
                        'rows' => 4,
                        'placeholder' => '',
                        'new_lines' => '',
                    ),
                    array(
                        'key' => 'field_gregergegeggeger',
                        'label' => 'Results',
                        'name' => '',
                        'aria-label' => '',
                        'type' => 'accordion',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'open' => 1,
                        'multi_expand' => 1,
                        'endpoint' => 0,
                    ),
                    array(
                        'key' => 'field_results_001',
                        'label' => 'Rank',
                        'name' => 'shg_game_results_rank',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_results_004',
                        'label' => 'Yesterday',
                        'name' => 'shg_game_results_yesterday',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_results_005',
                        'label' => 'Last Three Days',
                        'name' => 'shg_game_results_lastthreedays',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_results_006',
                        'label' => 'Last Seven Days',
                        'name' => 'shg_game_results_lastsevendays',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_results_007',
                        'label' => 'Last Thirty Days',
                        'name' => 'shg_game_results_lastthirtydays',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_top_handicapper_picks',
                        'label' => 'Picks Handicapper',
                        'name' => 'shg_picks',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'field_top_handicapper_pick_sport',
                                'label' => 'Sport',
                                'name' => 'shg_pick_sport',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                            array(
                                'key' => 'field_top_handicapper_pick_date',
                                'label' => 'Date',
                                'name' => 'shg_pick_date',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '40',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                            array(
                                'key' => 'field_top_handicapper_pick_status',
                                'label' => 'Status',
                                'name' => 'shg_pick_status',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                            array(
                                'key' => 'field_top_handicapper_pick_home_name',
                                'label' => 'Home Name',
                                'name' => 'shg_pick_home_name',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '40',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                            array(
                                'key' => 'field_top_handicapper_pick_home_image_api',
                                'label' => 'Home Image Api',
                                'name' => 'shg_pick_home_image_api',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                            array(
                                'key' => 'field_top_handicapper_pick_home_image_local',
                                'label' => 'Home Image Local',
                                'name' => 'shg_pick_home_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                            array(
                                'key' => 'field_top_handicapper_pick_visitor_name',
                                'label' => 'Visitor Name',
                                'name' => 'shg_pick_visitor_name',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '40',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                            array(
                                'key' => 'field_top_handicapper_pick_visitor_image_api',
                                'label' => 'Visitor Image Api',
                                'name' => 'shg_pick_visitor_image_api',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                            array(
                                'key' => 'field_top_handicapper_pick_visitor_image_local',
                                'label' => 'Visitor Image Local',
                                'name' => 'shg_pick_visitor_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                            array(
                                'key' => 'field_top_handicapper_pick_game_url',
                                'label' => 'Game Url',
                                'name' => 'shg_pick_game_url',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_handicapper_picks',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'field_64ca95f9b96a0',
                        'label' => 'Game Odds',
                        'name' => '',
                        'aria-label' => '',
                        'type' => 'accordion',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'open' => 1,
                        'multi_expand' => 1,
                        'endpoint' => 0,
                    ),
                    array(
                        'key' => 'field_shg_game_id_odds',
                        'label' => 'Game ID',
                        'name' => 'shg_game_id_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_external_id_odds',
                        'label' => 'External ID',
                        'name' => 'shg_external_id_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_sportsbook_odds',
                        'label' => 'Sportsbook',
                        'name' => 'shg_sportsbook_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_date_odds',
                        'label' => 'Date',
                        'name' => 'shg_date_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_home_handicap_spread_odds',
                        'label' => 'Spread Home Handicap',
                        'name' => 'shg_home_handicap_spread_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_home_spread_odds',
                        'label' => 'Spread Home Odds',
                        'name' => 'shg_home_spread_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_home_money_odds',
                        'label' => 'Money Line Home Odds',
                        'name' => 'shg_home_money_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_home_handicap_overunder_odds',
                        'label' => 'Over/Under Line Home Handicap',
                        'name' => 'shg_home_handicap_overunder_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_home_overunder_odds',
                        'label' => 'Over/Under Line Home Odds',
                        'name' => 'shg_home_overunder_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_away_handicap_spread_odds',
                        'label' => 'Spread Visitor Handicap',
                        'name' => 'shg_away_handicap_spread_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_away_spread_odds',
                        'label' => 'Spread Visitor Odds',
                        'name' => 'shg_away_spread_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_away_money_odds',
                        'label' => 'Money Line Visitor Odds',
                        'name' => 'shg_away_money_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_away_handicap_overunder_odds',
                        'label' => 'Over/Under Line Visitor Handicap',
                        'name' => 'shg_away_handicap_overunder_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_away_overunder_odds',
                        'label' => 'Over/Under Line Visitor Odds',
                        'name' => 'shg_away_overunder_odds',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_top_picks_game',
                        'label' => 'Picks Game',
                        'name' => 'shg_picks_game',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'key_shg_pick_game_handicapper_name',
                                'label' => 'Handicapper Name',
                                'name' => 'shg_pick_game_handicapper_name',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_handicapper_package',
                                'label' => 'Handicapper Package',
                                'name' => 'shg_pick_game_handicapper_package',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_handicapper_image',
                                'label' => 'Handicapper Image',
                                'name' => 'shg_pick_game_handicapper_image',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_handicapper_image_local',
                                'label' => 'Handicapper Image Local',
                                'name' => 'shg_pick_game_handicapper_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '30',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_sport_name',
                                'label' => 'Sport Name',
                                'name' => 'shg_pick_game_sport_name',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'field_pick_game_sport_acronym',
                                'label' => 'Acronym Sport',
                                'name' => 'shg_pick_game_sport_acronym',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '10',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'field_pick_game_bet',
                                'label' => 'Bet',
                                'name' => 'shg_pick_game_bet',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '10',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_handicapper_win',
                                'label' => 'Win',
                                'name' => 'shg_pick_game_handicapper_win',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '10',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_handicapper_lose',
                                'label' => 'Lose',
                                'name' => 'shg_pick_game_handicapper_lose',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '10',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'field_pick_game_teaser',
                                'label' => 'Teaser',
                                'name' => 'shg_pick_game_teaser',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'field_top_pick_date_game',
                                'label' => 'Date',
                                'name' => 'shg_pick_game_date',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_home_image_api',
                                'label' => 'Home Image Api',
                                'name' => 'shg_pick_game_home_image_api',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_home_image_local',
                                'label' => 'Home Image Local',
                                'name' => 'shg_pick_game_home_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_visitor_image_api',
                                'label' => 'Visitor Image Api',
                                'name' => 'shg_pick_game_visitor_image_api',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'key_shg_pick_game_visitor_image_local',
                                'label' => 'Visitor Image Local',
                                'name' => 'shg_pick_game_visitor_image_local',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                            array(
                                'key' => 'field_pick_game_url_profile',
                                'label' => 'Profile Url',
                                'name' => 'shg_pick_game_url_profile',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'required_message' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'field_top_picks_game',
                            ),
                        ),
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'post_type',
                            'operator' => '==',
                            'value' => get_option('shg_slug'),
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));
            endif;        
    }

    public function generate_form_cpt_game_simple() {
        if( function_exists('acf_add_local_field_group') ):
            acf_add_local_field_group(array(
                'key' => 'group_6406f02fbe5g643f54433',
                'title' => 'SBG Automatic Games',
                'fields' => array(
                    array(
                        'key' => 'field_64ca353f45f5345',
                        'label' => 'Game External ID',
                        'name' => 'shg_external_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_shg_manually_edited_lock_simple',
                        'label' => 'Lock from Auto-Updates',
                        'name' => 'shg_manually_edited',
                        'aria-label' => '',
                        'type' => 'true_false',
                        'instructions' => 'When enabled, this post will not be updated by the WP-CLI pull command. Automatically enabled when you edit a published/pending post.',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'message' => '',
                        'default_value' => 0,
                        'ui' => 1,
                        'ui_on_text' => 'Locked',
                        'ui_off_text' => 'Unlocked',
                    ),
                    array(
                        'key' => 'field_6tr43t3d343dtr3',
                        'label' => 'Sport ID',
                        'name' => 'shg_sport_id',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_6erwrwwrwefr',
                        'label' => 'Game Date',
                        'name' => 'shg_game_date',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_ryryy4y4t4t34t',
                        'label' => 'ID Home Team',
                        'name' => 'shg_id_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_435345435r3345r',
                        'label' => 'Home Team',
                        'name' => 'shg_home_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_sedffuhjyu67wessdf',
                        'label' => 'ID Visitor Team',
                        'name' => 'shg_id_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_ghjkui873weterrtyh',
                        'label' => 'Visitor Team',
                        'name' => 'shg_visitor_team',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '40',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_fthyr64fctft34t',
                        'label' => 'Game URL',
                        'name' => 'shg_game_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_awq2ceeruty6u45t3r32d',
                        'label' => 'Game API Image',
                        'name' => 'shg_game_api_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_ftuyy7945certgte43tr',
                        'label' => 'Game Local Image',
                        'name' => 'shg_game_local_image',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '25',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'post_type',
                            'operator' => '==',
                            'value' => get_option('shg_slug'),
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));
            endif;        
    }

    public function generate_form_sportshub_games_contents() {
        if(function_exists('acf_add_local_field_group')):
            acf_add_local_field_group(array(
                'key' => 'group_6234f432234',
                'title' => 'Sports Hub Games Contents',
                'fields' => array(
                    array(
                        'key' => 'field_642344324324324',
                        'label' => 'Status',
                        'name' => 'shg_content_status',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '20',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'inactive' => 'Inactive',
                            'active' => 'Active',
                        ),
                        'default_value' => 'inactive',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_shg_content_validate_content',
                        'label' => 'Check if post is duplicated by text search on title and content (heavy impact on DB)',
                        'name' => 'shg_content_validate_content',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '80',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            '0' => 'No',
                            '1' => 'Yes',
                        ),
                        'default_value' => '0',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'shg_field_6445f3d47678sac',
                        'label' => 'URL',
                        'name' => 'shg_content_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_642344324324324',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_6ghf23q4bgfh9hgfq23gfd',
                        'label' => 'Token',
                        'name' => 'shg_content_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_642344324324324',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_646cseffw45frxxxx',
                        'label' => 'Content Types Members',
                        'name' => 'shg_content_types_members',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_642344324324324',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_6ghfw4f5334ff34f3r34r43',
                                'label' => 'Type Slug',
                                'name' => 'shg_content_type_slug_members',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'shg_field_646cseffw45frxxxx',
                            ),
                            array(
                                'key' => 'shg_field_6ghw4535fd32244s4324',
                                'label' => 'Block Game Links Title',
                                'name' => 'shg_content_type_block_title_members',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'shg_field_646cseffw45frxxxx',
                            ),
                            array(
                                'key' => 'shg_field_6fjy4t3fg54r34fg4t3',
                                'label' => 'Icon Block Game Links',
                                'name' => 'shg_content_type_block_icon_members',
                                'aria-label' => '',
                                'type' => 'image',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '20',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'acfe_save_meta' => 0,
                                'uploader' => '',
                                'return_format' => 'array',
                                'upload_folder' => '',
                                'acfe_thumbnail' => 0,
                                'min_width' => '',
                                'min_height' => '',
                                'min_size' => '',
                                'max_width' => '',
                                'max_height' => '',
                                'max_size' => '',
                                'mime_types' => '',
                                'preview_size' => 'medium',
                                'library' => 'all',
                            ),
                            array(
                                'key' => 'shg_field_6ge34b34f3fgdfgh',
                                'label' => 'Content Heading Title',
                                'name' => 'shg_content_type_title_members',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '40',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'shg_field_646cseffw45frxxxx',
                            ),
                        ),
                    ),
                    array(
                        'key' => 'shg_field_646cseffw45frdfcf',
                        'label' => 'Content Types Masterfeed',
                        'name' => 'shg_content_types',
                        'aria-label' => '',
                        'type' => 'repeater',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_642344324324324',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '',
                            'class' => '',
                            'id' => '',
                        ),
                        'layout' => 'block',
                        'pagination' => 0,
                        'min' => 0,
                        'max' => 0,
                        'collapsed' => '',
                        'button_label' => 'Add Row',
                        'rows_per_page' => 20,
                        'sub_fields' => array(
                            array(
                                'key' => 'shg_field_6ghfw4f534523gfd',
                                'label' => 'Type Slug',
                                'name' => 'shg_content_type_slug',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'shg_field_646cseffw45frdfcf',
                            ),
                            array(
                                'key' => 'shg_field_6ge34b45y54yjrddfgdfgh',
                                'label' => 'Content Heading Title',
                                'name' => 'shg_content_type_title',
                                'aria-label' => '',
                                'type' => 'text',
                                'instructions' => '',
                                'required' => 1,
                                'conditional_logic' => 0,
                                'wrapper' => array(
                                    'width' => '50',
                                    'class' => '',
                                    'id' => '',
                                ),
                                'default_value' => '',
                                'maxlength' => '',
                                'placeholder' => '',
                                'prepend' => '',
                                'append' => '',
                                'parent_repeater' => 'shg_field_646cseffw45frdfcf',
                            ),
                        ),
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-sports-contents',
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            endif;    
    }

    public function save_options_contents_types() {
        try {
            $option = (isset($_GET['page'])) ? $_GET['page'] : null;

            if (!empty($option) && $option == 'acf-options-sports-contents' && isset($_POST['acf']) && is_array($_POST['acf']) && count($_POST['acf']) > 0) {
                $helpers = New Sportshub_Games_Helpers;
                $helpers->setRedisGroup(REDIS_GROUP_SHGAMES);

                if ($helpers->initRedis()) {
                    try {
                        $key_sports_contents = $helpers->getRedisKeySportsContents();
                        $key_sports_contents_members = $helpers->getRedisKeySportsContentsMembers();
                        $helpers->setRedis($key_sports_contents, serialize([]), SHGAMES_REDIS_TIMEOUT);
                        $helpers->setRedis($key_sports_contents_members, serialize([]), SHGAMES_REDIS_TIMEOUT);
                    } catch (\Exception $exception) {
                        error_log(print_r([
                            'message' => $exception->getMessage(),
                            'keys' => [$key_sports_contents, $key_sports_contents_members],
                        ], true));
                    }
                } else {
                    error_log(print_r([
                        'message' =>  'Redis activation is necessary to store the sports contents cache.',
                    ], true));
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
                'option' => $option,
            ], true));
        }
    }
}
