<?php

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * HELPER COMMENT START
 * 
 * This class contains all of the available CLI commands for your plugin. 
 * Down below, we added a command that allows you to display the current 
 * details about your plugin within the console. 
 * 
 * To test the command, please head over to your WordPress instance and type in the following
 * 
 * wp sh_handicappers details
 * 
 * HELPER COMMENT END
 */

WP_CLI::add_command('sh_handicappers', 'Sh_Handicappers_CLI');

/**
 * Class Sh_Handicappers_CLI
 *
 * This class contains all WP CLI related features.
 *
 * @package		SHHANDICAP
 * @subpackage	Classes/Sh_Handicappers_CLI
 * @author		IT
 * @since		1.0.0
 */
class Sh_Handicappers_CLI extends WP_CLI_Command {
    private $helpers;
    /**
     * Our Sh_Handicappers_CLI constructor 
     * to run the plugin CLI logic.
     *
     * @since 1.0.0
     */
    function __construct(){
        $this->helpers = New Sh_Handicappers_Helpers;
    }

    /**
     * Get plugin details
     *
     * ## OPTIONS
     *
     * None. Returns basic info regarding the plugin instance.
     *
     * ## EXAMPLES
     *
     * To access this feature, type the following into your console:
     * wp sh_handicappers details
     *
     * @param		array $args
     * @param		array $assoc_args
     * @return		void
     */
    public function details($args, $assoc_args) {
        WP_CLI::line(sprintf(__('Plugin name: %s', 'SH Handicappers'), SHHANDICAP_NAME));
        WP_CLI::line(sprintf(__('Plugin version: %s', 'SH Handicappers'), SHHANDICAP_VERSION));
    }

    public function pull($args, $assoc_args) {
        $this->removeLogsFiles();

        if (get_option('shh_status') != 1) {
            error_log(sprintf(__('Sportshub Handicappers: Plugin disabled.', 'SportsHub Handicappers')));
            WP_CLI::line(sprintf(__('Sportshub Handicappers: Plugin disabled.', 'SportsHub Handicappers')));

            return;
        }

        $hancicappers_selected = get_field('sh_handicappers', 'options');
        $ids = array_column($hancicappers_selected, 'shh_handicapper');
        $filename =  'SHH_PULL_'. date('Y-m-d-H-i-s') . '.json';
        $file_path = SHHANDICAP_PLUGIN_LOG_DIR . '/' . $filename;
        $this->helpers->getHadicappersOptions($file_path);
        $all_hancicappers = $this->helpers->parseFeedFile($file_path);
        $handicappers_data = [];
        $new_post = 0;
        $existing_post = 0;

        foreach ($all_hancicappers as $item) {
            if (in_array($item['id'], $ids)) {
                $summary = $this->getSummaryData($item['id']);

                if (count($summary) == 0) {
                    continue;
                }
                
                $sports_ranks = $this->getSportRanks($summary['sports_ranks']);

                $today_picks_data =  $this->getTodayPicksData($item['id']);
                $number_of_picks = count($today_picks_data);               
                $today_picks_images = ($number_of_picks > 0) ? $this->getTodayPicksImages($summary['today_picks']) : [];
                $today_picks = ($number_of_picks > 0) ?  $this->getTodayPicks($today_picks_data['handicapper_picks']) : [];

                $handicappers_data = [
                    'id' => $item['id'],
                    'name' => $item['name'],
                    'bio' => $item['bio'],
                    'image' => $item['image'],
                    'day_results' => serialize($summary['day_results']),
                    'sports_ranks'=> serialize($sports_ranks),
                    'today_picks'=> serialize($today_picks),
                    'today_picks_images'=> serialize($today_picks_images)
                ];

                $response = $this->insertPostContent($handicappers_data);
                if (is_array($response) && isset($response['type'])) {
                    if ($response['type'] == 'new') {
                        $new_post++;
                    } elseif ($response['type'] == 'duplicate') {
                        $existing_post++;
                    } else {
                        WP_CLI::Line($response['message']);
                    }
                }
            }
        }

        error_log(sprintf(__('Sportshub Games: %s Succesfully inserted posts.', 'SportsHub Handicappers'), $new_post));
        WP_CLI::line(sprintf(__('%s Succesfully inserted posts.', 'SportsHub Handicappers'), $new_post));

        if ($existing_post > 0) {
            error_log(sprintf(__('SportsHub Handicappers: %s Succesfully updated posts.', 'SportsHub Handicappers'), $existing_post));
            WP_CLI::line(sprintf(__('%s Succesfully updated posts.', 'SportsHub Handicappers'), $existing_post));
        }

        update_option('shh_last_run', date('Y-m-d H:i:s'));
    }

    public function getTitleHandicapper($capper) {
        return str_replace('[HANDICAPPERNAME]', $capper, get_option('shh_post_placeholder'));
    }

    public function insertPostContent($data)
    {
        wp_reset_postdata();

        $args_search = [
            'post_type' => get_option('shh_slug'),
            'meta_query' => [
                [
                    'key' => 'handicapper_id',
                    'value' => $data['id'],
                    'compare' => '=',
                ],
            ],
        ];

        $post_id = null;
        $query = new WP_Query($args_search);

        if (!$query->have_posts()) {
            $post_id = wp_insert_post([
                'post_type' => get_option('shh_slug'),
                'post_name' => $data['name'],
                'post_title' => $this->getTitleHandicapper($data['name']),
                'post_content' => $this->getContentFormat($data['bio']),
                'post_status' => 'publish'
            ]);

            if (!is_wp_error($post_id)) {
                add_post_meta($post_id, 'handicapper_id', $data['id'], true);
                add_post_meta($post_id, 'image_url', $data['image'], true);
                add_post_meta($post_id, 'day_results', $data['day_results']);
                add_post_meta($post_id, 'sports_ranks', $data['sports_ranks']);
                add_post_meta($post_id, 'today_picks', $data['today_picks']);
                add_post_meta($post_id, 'today_picks_images', $data['today_picks_images']);

                # Get local capper image
                $capper_image_url = $this->setLocalImage($data['image']);

                if (!empty($capper_image_url)) {
                    update_post_meta($post_id, 'image_url', $capper_image_url);
                }

                $picks_images = unserialize($data['today_picks_images']);
                if (isset($picks_images) && is_array($picks_images) && count($picks_images) > 0) {
                    $picks_images_new = [];

                    foreach ($picks_images as $index => $images) {
                        foreach ($images as $key => $image) {
                            $team_image = $this->setLocalImage($image);

                            if (!empty($team_image)) {
                                $picks_images_new[$index][$key] = $team_image;
                            }
                        }
                    }
                    
                    if (is_array($picks_images_new) && count($picks_images_new) > 0) {
                        update_post_meta($post_id, 'today_picks_images', serialize($picks_images_new));
                    }
                }
            }

            if (!is_wp_error($post_id)) {
                WP_CLI::line(__('New Post, ID: '. $post_id, 'SportsHub Handicappers'));

                return ['type' => 'new', 'post_id' => $post_id, 'message' => ''];
            } else {
                return ['type' => 'error', 'post_id' => null, 'message' => $post_id->get_error_message()];
            }
        } else {
            while ($query->have_posts()) {
                $query->the_post();
                $post_id = get_the_ID();

                wp_update_post([
                    'ID' => $post_id,
                    'post_title' => $this->getTitleHandicapper($data['name']),
                    'post_content' => $this->getContentFormat($data['bio']),
                    'meta_input' => [
                        //'image_url' => $data['image'],
                        'day_results' => $data['day_results'],
                        'sports_ranks' => $data['sports_ranks'],
                        'today_picks' => $data['today_picks'],
                        //'today_picks_images' => $data['today_picks_images'],
                    ]
                ]);

                # Get local capper image
                $capper_image_url = $this->setLocalImage($data['image']);

                if (!empty($capper_image_url)) {
                    update_post_meta($post_id, 'image_url', $capper_image_url);
                }

                $picks_images = unserialize($data['today_picks_images']);
                if (isset($picks_images) && is_array($picks_images) && count($picks_images) > 0) {
                    $picks_images_new = [];

                    foreach ($picks_images as $index => $images) {
                        foreach ($images as $key => $image) {
                            $team_image = $this->setLocalImage($image);

                            if (!empty($team_image)) {
                                $picks_images_new[$index][$key] = $team_image;
                            }
                        }
                    }
                    
                    if (is_array($picks_images_new) && count($picks_images_new) > 0) {
                        update_post_meta($post_id, 'today_picks_images', serialize($picks_images_new));
                    }
                }
            }

            WP_CLI::line(__('Update Post, ID: '. $post_id, 'SportsHub Handicappers'));
            return ['type' => 'duplicate', 'post_id' => $post_id, 'message' => ''];
        }
    }

    public function setLocalImage($image_url) {
        $helpers = New Sh_Handicappers_Helpers;
        $image_object = $helpers->getNameFile($image_url, false);

        $image_object_url = null;
        if (is_array($image_object) && isset($image_object['is_image']) && $image_object['is_image'] == true) {
            $attachment_capper_check = $this->getAttachment('post_title', $image_object['name']);

            if (empty($attachment_capper_check)) {
                # Download the image and upload it to the media library
                $home_attachment_id = media_sideload_image($image_object['image_url'], 0, $image_object['name'], 'id');

                # Update image attachment with alt text
                update_post_meta($home_attachment_id, '_wp_attachment_image_alt', $image_object['alt']);

                # Get url of already stored image
                $attachment_capper_check = $this->getAttachment('ID', $home_attachment_id);
                $image_object_url = $attachment_capper_check[0]['guid'];
            } else {
                $image_object_url = $attachment_capper_check[0]['guid'];
            }
        }

        return $image_object_url;
    }

    public function getAttachment($field, $value) {
        global $wpdb;
        $value = esc_sql($value);
        $attachment = $wpdb->get_results( "SELECT * FROM $wpdb->posts WHERE $field = '$value' AND post_type = 'attachment'", ARRAY_A);

        if (is_array($attachment)) {
            return $attachment;
        }

        return null;
    }

    public function getContentFormat($content) {
        $block_init = '<!-- wp:paragraph --><p class="shh-paragraph">';
        $block_end = '</p><!-- /wp:paragraph -->';

        return $block_init.strip_tags($content).$block_end;
    }

    public function getSummaryData($handicapperId) {
        $params = $this->helpers->getQueryParams(['handicapper' => $handicapperId]);
        $url = get_field('shh_summary_url', 'options');
        $summaryUrl = $this->helpers->getApiUrl($url, '&'.$params);       
        $summaryRequest = $this->helpers->getApiRequest($summaryUrl);

        if ($summaryRequest['http_code'] == 200) {
            $response_json = $this->helpers->isJson($summaryRequest['data']);            

            if ($response_json != null) {
                return $response_json;
            }
        }
    }

    public function getTodayPicksData($handicapperId) {
        $date = date('Y-m-d');
        $params = $this->helpers->getQueryParams([
            'id_handicapper' => $handicapperId,
            'principalDate' => $date,
            'secondaryDate' => $date,
        ]);
        $url = get_field('shh_picks_url', 'options');
        $picksUrl = $this->helpers->getApiUrl($url, '&'.$params);
        $picksRequest = $this->helpers->getApiRequest($picksUrl);

        if ($picksRequest['http_code'] == 200) {
            $response_json = $this->helpers->isJson($picksRequest['data']);            

            if ($response_json != null) {                
                return $response_json;
            }
        }

        return [];
    }

    public function getSportNameById($sportsArray, $id) {
        foreach ($sportsArray as $sport) {
            if ($sport['id'] === $id) {
                return $sport['sport_name'];
            }
        }
        
        return null;
    }

    public function getSports() {
        $url = get_field('shh_sports_url', 'options');
        $sportUrl = $this->helpers->getApiUrl($url);
        $sportsRequest = $this->helpers->getApiRequest($sportUrl);

        if ($sportsRequest['http_code'] == 200) {
            $response_json = $this->helpers->isJson($sportsRequest['data']);            

            if ($response_json != null) {
                $data = json_decode($sportsRequest['data'], true);

                $sports = [];
                foreach ($data as $item) {
                    $sports[$item['id']] = $item['sport_name'];
                }

                return $sports;
            }
        }

        return [];
    }

    public function getSportRanks($sportsRanks) {
        foreach ($sportsRanks as $key => $row) {
            $sportsRanksSort[] = $row['rank'];
        }

        array_multisort($sportsRanksSort, SORT_ASC, $sportsRanks);

        $sportsRankOrder = array_map(function ($sportRank) {
            return [
                'sportId' => $sportRank['id_sport'],
                'rank' => $sportRank['rank'],
            ];
        }, $sportsRanks);

        return $sportsRankOrder;
    }

    public function getTodayPicks($todayPicks) {
        return array_map(function ($todayPick) {

            $formattedDate = $this->formatDate($todayPick['game_date']);

            return [
                'description' => $todayPick['picks'][0]['pick_details'][0]['description'],
                'bet_type' => $todayPick['picks'][0]['pick_details'][0]['bet_type'],
                'game_date' => $formattedDate,
                'sport_name' => $todayPick['sport'],
                'visitor_name' => $todayPick['visitor_name'],
                'home_name' => $todayPick['home_name'],
                'visitor_id' => $todayPick['id_visitor_team'],
                'home_id' => $todayPick['id_home_team']
            ];
        }, $todayPicks);
    }

    public function getTodayPicksImages($todayPicks) {
        return array_map(function ($todayPick) {

            return [
                $todayPick['pick_details'][0]['game']['id_visitor_team']['id'] => $todayPick['pick_details'][0]['game']['id_visitor_team']['image'],
                $todayPick['pick_details'][0]['game']['id_home_team']['id'] => $todayPick['pick_details'][0]['game']['id_home_team']['image']
            ];
        }, $todayPicks);
    }

    function removeLogsFiles() {
        $log_lifetime_in_days = get_option('options_shh_logs_lifetime', 0);
        $log_lifetime_in_seconds = $log_lifetime_in_days * 24 * 60 * 60;
        $log_directory = SHHANDICAP_PLUGIN_LOG_DIR;

        if (!file_exists($log_directory)) {
            wp_mkdir_p($log_directory);
        }
        
        $message = false;
        $log_files = glob($log_directory . '/*');
        
        foreach ($log_files as $log_file) {
            if (filemtime($log_file) < time() - $log_lifetime_in_seconds) {
                unlink($log_file);
                $message = true;
            }
        }

        if ($message) {
            WP_CLI::line(sprintf(__('Sportshub Handicappers: Remove old log files.', 'SportsHub Handicappers')));
        }

        return true;
    }

    public function formatDate($date) {
        $date = new DateTime($date);
        $date_formatted = $date->format('d/m');
        $hora_formatted = $date->format('H:i');
        return $date_formatted . ' | ' . $hora_formatted;
    }
}
