<?php

// Exit if accessed directly.
if (!defined('ABSPATH')) exit;

/**
 * HELPER COMMENT START
 * 
 * This is the main class that is responsible for registering
 * the core functions, including the files and setting up all features. 
 * 
 * To add a new class, here's what you need to do: 
 * 1. Add your new class within the following folder: core/includes/classes
 * 2. Create a new variable you want to assign the class to (as e.g. public $helpers)
 * 3. Assign the class within the instance() function ( as e.g. self::$instance->helpers = new Sh_Handicappers_Helpers();)
 * 4. Register the class you added to core/includes/classes within the includes() function
 * 
 * HELPER COMMENT END
 */

if (!class_exists('Sh_Handicappers')) :

    /**
     * Main Sh_Handicappers Class.
     *
     * @package		SHHANDICAP
     * @subpackage	Classes/Sh_Handicappers
     * @since		1.0.0
     * @author		IT
     */
    final class Sh_Handicappers {

        /**
         * The real instance
         *
         * @access	private
         * @since	1.0.0
         * @var		object|Sh_Handicappers
         */
        private static $instance;

        /**
         * SHHANDICAP helpers object.
         *
         * @access	public
         * @since	1.0.0
         * @var		object|Sh_Handicappers_Helpers
         */
        public $helpers;

        /**
         * SHHANDICAP settings object.
         *
         * @access	public
         * @since	1.0.0
         * @var		object|Sh_Handicappers_Settings
         */
        public $settings;
        public $slug;

        /**
         * Throw error on object clone.
         *
         * Cloning instances of the class is forbidden.
         *
         * @access	public
         * @since	1.0.0
         * @return	void
         */
        public function __clone() {
            _doing_it_wrong(__FUNCTION__, __('You are not allowed to clone this class.', 'sh-handicappers'), '1.0.0');
        }

        /**
         * Disable unserializing of the class.
         *
         * @access	public
         * @since	1.0.0
         * @return	void
         */
        public function __wakeup() {
            _doing_it_wrong(__FUNCTION__, __('You are not allowed to unserialize this class.', 'sh-handicappers'), '1.0.0');
        }

        /**
         * Main Sh_Handicappers Instance.
         *
         * Insures that only one instance of Sh_Handicappers exists in memory at any one
         * time. Also prevents needing to define globals all over the place.
         *
         * @access		public
         * @since		1.0.0
         * @static
         * @return		object|Sh_Handicappers	The one true Sh_Handicappers
         */
        public static function instance() {
            if (!isset(self::$instance) && ! (self::$instance instanceof Sh_Handicappers)) {
                self::$instance					= new Sh_Handicappers;
                self::$instance->base_hooks();
                self::$instance->includes();
                self::$instance->helpers		= new Sh_Handicappers_Helpers();
                self::$instance->settings		= new Sh_Handicappers_Settings();

                if (defined('SHH_SLUG')) {
                    $slug = json_decode(SHH_SLUG, true);
                } else {
                    error_log(print_r([
                        'message' => 'The constant SHH_SLUG is not defined.',
                    ], true));

                    $slug = [
                        'slug' => 'handicapper',
                        'label' => 'Handicapper',
                    ];
                }

                self::$instance->slug = $slug;

                //Fire the plugin logic
                new Sh_Handicappers_Run();

                /**
                 * Fire a custom action to allow dependencies
                 * after the successful plugin setup
                 */
                do_action('SHHANDICAP/plugin_loaded');
            }

            return self::$instance;
        }

        /**
         * Include required files.
         *
         * @access  private
         * @since   1.0.0
         * @return  void
         */
        private function includes() {
            require_once SHHANDICAP_PLUGIN_DIR . 'core/includes/classes/class-sh-handicappers-helpers.php';
            require_once SHHANDICAP_PLUGIN_DIR . 'core/includes/classes/class-sh-handicappers-settings.php';
            if (defined('WP_CLI') && WP_CLI){
                require_once SHHANDICAP_PLUGIN_DIR . 'core/includes/classes/class-sh-handicappers-cli.php';
            }

            require_once SHHANDICAP_PLUGIN_DIR . 'core/includes/classes/class-sh-handicappers-run.php';
        }

        /**
         * Add base hooks for the core functionality
         *
         * @access  private
         * @since   1.0.0
         * @return  void
         */
        private function base_hooks() {
            add_action('plugins_loaded', array(self::$instance, 'load_textdomain'));
        }

        /**
         * Loads the plugin language files.
         *
         * @access  public
         * @since   1.0.0
         * @return  void
         */
        public function load_textdomain() {
            load_plugin_textdomain('sh-handicappers', FALSE, dirname(plugin_basename(SHHANDICAP_PLUGIN_FILE)) . '/languages/');
        }

        public function get_slug($array = true) {
            if ($array) {
                return $this->slug;
            } else {
                return $this->slug['slug'];
            }
        }
    }

endif; // End if class_exists check.