<?php

// Exit if accessed directly.
if (!defined('ABSPATH')) exit;

/**
 * Class Sh_Mapping_Run
 *
 * Thats where we bring the plugin to life
 *
 * @package        SHMAPPING
 * @subpackage    Classes/Sh_Mapping_Run
 * @author        IT
 * @since        1.0.0
 */

class Sh_Mapping_Run {

    /**
     * Our Sh_Mapping_Run constructor 
     * to run the plugin logic.
     *
     * @since 1.0.0
     */
    function __construct() {
        $this->add_hooks();
        add_action('admin_notices', [$this, 'show_custom_messages']);
    }

    /**
     * ######################
     * ###
     * #### WORDPRESS HOOKS
     * ###
     * ######################
     */

    /**
     * Registers all WordPress and plugin related hooks
     *
     * @access    private
     * @since    1.0.0
     * @return    void
     */
    private function add_hooks() {
        add_action('acf/init', [$this, 'generate_form_sh_mapping'], 20);
        add_action('acf/save_post', [$this, 'save_settings_sh_mapping'], 20);
        add_filter('acf/load_value/name=sh_mapping_reset_last_run', [$this, 'load_field_sh_mapping_reset_last_run'], 10, 3);
        add_filter('acf/load_value/name=sh_mapping_load_player', [$this, 'load_field_sh_mapping_load_player'], 10, 3);
        add_action('sh_mapping_cron_event', [$this, 'sh_mapping_cron_function']);
        add_filter('cron_schedules', [$this,'sh_mapping_add_cron_interval']);
        
        add_action('admin_enqueue_scripts', [$this, 'enqueue_backend_scripts_and_styles'], 20);
        add_action('admin_menu', [$this, 'register_custom_admin_menu_pages'], 20);
        register_activation_hook(SHMAPPING_PLUGIN_FILE, [$this, 'activation_hook_callback']);
        register_deactivation_hook(SHMAPPING_PLUGIN_FILE, [$this, 'deactivation_hook_callback']);
    }

    public function load_field_sh_mapping_load_player($value) {
        return 'default';
    }

    public function load_field_sh_mapping_reset_last_run($value) {
        if ($value == 'reset') {
            update_field('sh_mapping_reset_last_run', 'default');
        }

        return 'default';
    }

    public function sh_mapping_cron_function() {
        // Call the WP-CLI command
        $command = 'wp sh_mapping pull';
        $output = shell_exec($command);
    }

    public function activation_hook_callback() {
        if (!wp_next_scheduled('sh_mapping_cron_event')) {
            wp_schedule_event(time(), 'daily', 'sh_mapping_cron_event');
            update_option('sh_mapping_cron_time', 'daily');
        }
    }

    public function deactivation_hook_callback() {
        wp_clear_scheduled_hook('sh_mapping_cron_event');
    }

    public function sh_mapping_add_cron_interval($schedules) {
        if (!array_key_exists("hourly", $schedules)) {
            $schedules['hourly'] = array(
                'interval' => 60 * 60,
                'display'  => __('Every Hour'),
            );
        }

        if (!array_key_exists("twicedaily", $schedules)) {
            $schedules['twicedaily'] = array(
                'interval' => 60 * 60 * 12,
                'display'  => __('Every 12 hours'),
            );
        }

        if (!array_key_exists("daily", $schedules)) {
            $schedules['daily'] = array(
                'interval' => 60 * 60 * 24,
                'display'  => __('Every Day'),
            );
        }

        if (!array_key_exists("weekly", $schedules)) {
            $schedules['weekly'] = array(
                'interval' => 60 * 60 * 24 * 7,
                'display'  => __('Once Weekly'),
            );
        }

        return $schedules;
    }

    public function enqueue_backend_scripts_and_styles() {
        wp_enqueue_style('shmapping-backend-styles', SHMAPPING_PLUGIN_URL . 'core/includes/assets/css/backend-styles.css', array(), SHMAPPING_VERSION, 'all');
    }

    public function register_custom_admin_menu_pages() {
        acf_add_options_page([
            'page_title'    => __('SH Mapping'),
            'menu_title'    => __('SH Mapping'),
            'menu_slug'     => SHMAPPING_NAME_SLUG,
            'capability'    => 'edit_posts',
            'icon_url'      => 'dashicons-networking',
            'redirect'      => true,
        ]);

        acf_add_options_page([
            'page_title'  => __('Settings'),
            'menu_title'  => __('SH Mapping Settings'),
            'parent_slug' => SHMAPPING_NAME_SLUG,
        ]);
    }

    public function generate_form_sh_mapping() {
        if(function_exists('acf_add_local_field_group')):
            acf_add_local_field_group(array(
                'key' => 'group_6g76687gwwerewr682_mapping',
                'title' => 'Sports Hub Mapping Settings',
                'fields' => array(
                    array(
                        'key' => 'field_64d565794ac4f_mapping',
                        'label' => 'Status',
                        'name' => 'sh_mapping_status',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '35',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'inactive' => 'Inactive',
                            'active' => 'Active',
                        ),
                        'default_value' => 'inactive',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'shm_field_646e9a1f84fe1_mapping',
                        'label' => 'Cronjob run time',
                        'name' => 'sh_mapping_cron_time',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '35',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'hourly' => 'Once Hourly',
                            'twicedaily' => 'Twice Daily',
                            'daily' => 'Once Daily',
                            'weekly' => 'Once Weekly',
                        ),
                        'default_value' => 'daily',
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'field_64c95bea5809b_mapping',
                        'label' => 'Last Mapping Pull Datetime',
                        'name' => 'sh_mapping_last_run_message',
                        'aria-label' => '',
                        'type' => 'message',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '30',
                            'class' => '',
                            'id' => '',
                        ),
                        'message' => get_option('sh_mapping_last_run', 'never'),
                        'new_lines' => 'wpautop',
                        'esc_html' => 0,
                    ),
                    array(
                        'key' => 'field_6e4tg34t4th34tf34t43_mapping',
                        'label' => 'URL Data',
                        'name' => 'sh_mapping_url',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_6e4sfdssrwrdrwr_mapping',
                        'label' => 'URL Seasons',
                        'name' => 'sh_mapping_url_seasons',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_6drter6534dr34r32dr_mapping',
                        'label' => 'URL Sports',
                        'name' => 'sh_mapping_url_sports',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'field_6dqwd34tk986wr3eqsgrfdr_mapping',
                        'label' => 'URL Leagues',
                        'name' => 'sh_mapping_url_leagues',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_mapping_url_players',
                        'label' => 'URL Players',
                        'name' => 'sh_mapping_url_players',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shm_field_64c3e7748071d_mapping',
                        'label' => 'Token',
                        'name' => 'sh_mapping_token',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_sh_mapping_url_hierarchy',
                        'label' => 'URL Hierarchy',
                        'name' => 'sh_mapping_url_hierarchy',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'key_mapping_token_hierarchy',
                        'label' => 'Token',
                        'name' => 'sh_mapping_token_hierarchy',
                        'aria-label' => '',
                        'type' => 'text',
                        'instructions' => '',
                        'required' => 1,
                        'conditional_logic' => array(
                            array(
                                array(
                                    'field' => 'field_64d565794ac4f_mapping',
                                    'operator' => '==',
                                    'value' => 'active',
                                ),
                            ),
                        ),
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'default_value' => '',
                        'maxlength' => '',
                        'placeholder' => '',
                        'prepend' => '',
                        'append' => '',
                    ),
                    array(
                        'key' => 'shg_field_646ec0e1bb72a_mapping',
                        'label' => 'Reset Last Run',
                        'name' => 'sh_mapping_reset_last_run',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'default' => 'Default',
                            'reset' => 'Reset',
                        ),
                        'default_value' => '',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                    array(
                        'key' => 'key_mapping_load_player',
                        'label' => 'Load Players',
                        'name' => 'sh_mapping_load_player',
                        'aria-label' => '',
                        'type' => 'radio',
                        'instructions' => '',
                        'required' => 0,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '50',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => array(
                            'default' => 'Default',
                            'load' => 'Load Players',
                        ),
                        'default_value' => '',
                        'return_format' => 'value',
                        'allow_null' => 0,
                        'other_choice' => 0,
                        'layout' => 'horizontal',
                        'save_other_choice' => 0,
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-sh-mapping-settings',
                        ),
                    ),
                ),
                'menu_order' => 0,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            $players_example = [];
            $hierarchy_league_example = [];
            $players = function_exists('shMappingTeamPlayers') ? shMappingTeamPlayers() : [];
            $sports_only = function_exists('shMappingSportsOnly') ? shMappingSportsOnly() : [];
            $sports_leagues_options = function_exists('shMappingSportsLeaguesOption') ? shMappingSportsLeaguesOption() : [];
            $sports_leagues_titles = function_exists('shMappingSportsLeaguesTitles') ? shMappingSportsLeaguesTitles() : [];
            $sports_direct = function_exists('shMappingSportsDirect') ? shMappingSportsDirect() : [];
            $leagues_sport_direct = function_exists('shMappingLeaguesSportsDirect') ? shMappingLeaguesSportsDirect('/sport/basketball') : [];
            $seasons_direct = function_exists('shMappingSeasonsDirect') ? shMappingSeasonsDirect('/sport/basketball/league:403878') : [];
            $team_leagues_direct = function_exists('shMappingTeamsLeagueDirect') ? shMappingTeamsLeagueDirect('/sport/basketball/league:403878') : [];
            $hierarchy_sports = function_exists('shMappingHierarchySports') ? shMappingHierarchySports() : [];
            $hierarchy_family = function_exists('shMappingHierarchyFamily') ? shMappingHierarchyFamily('7-68') : [];
            $hierarchy_leagues = function_exists('shMappingHierarchyLeagues') ? shMappingHierarchyLeagues() : [];

            if (is_array($players) && count($players) > 0) {
                $players_selected = array_slice($players, 0, 10);
                foreach ($players_selected as $id => $player) {
                    $players_example[$id] = $player['first_name'].' '.$player['last_name'];
                }
            }

            if (is_array($hierarchy_leagues) && count($hierarchy_leagues) > 0) {
                $hierarchy_leagues_selected = array_slice($hierarchy_leagues, 0, 20);
                foreach ($hierarchy_leagues_selected as $id => $name) {
                    $hierarchy_league_example[$id] = $name;
                }
            }

            acf_add_local_field_group(array(
                'key' => 'group_6g7fewffwefwf_mapping',
                'title' => 'Sports Hub Mapping Cache',
                'fields' => array(
                    array(
                        'key' => 'shg_field_rtgferttetetsdfvc',
                        'label' => 'Sports',
                        'name' => 'sh_mapping_sports',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Sports function shMappingSportsOnly()',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $sports_only,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'shg_field_64wefwfwfef1',
                        'label' => 'Sports / Leagues',
                        'name' => 'sh_mapping_sports_leagues',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Sports Leagues function shMappingSportsLeaguesOption()',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $sports_leagues_options,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'shg_field_64wrtyhyuy5445ft345t',
                        'label' => 'Sports / Leagues Titles',
                        'name' => 'sh_mapping_sports_leagues_titles',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Sports Leagues Titles function shMappingSportsLeaguesTitles()',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $sports_leagues_titles,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'sh_mapping_sports_direct',
                        'label' => 'Sports Direct',
                        'name' => 'sh_mapping_sports_direct',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Sports Direct function shMappingSportsDirect()',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $sports_direct,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'shg_field_ghfe45gd5yth4ef5tujessdrffgds',
                        'label' => 'Leagues Direct',
                        'name' => 'sh_mapping_leagues_direct',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Leagues Direct function shMappingLeaguesSportsDirect(/sport/basketball)',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $leagues_sport_direct,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'shg_field_gsf352hgf3r4gfd56e4gr4r3w',
                        'label' => 'Seasons Direct',
                        'name' => 'sh_mapping_seasons_direct',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Seasons Direct function shMappingSeasonsDirect(/sport/basketball/league:403878)',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $seasons_direct,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'shg_field_ghje4fdgfhrty5wresrdg',
                        'label' => 'Teams Direct',
                        'name' => 'sh_mapping_teams_direct',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Teams Direct function shMappingTeamsLeagueDirect(/sport/basketball/league:403878)',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $team_leagues_direct,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'key_sh_mapping_players',
                        'label' => 'Players',
                        'name' => 'sh_mapping_players',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Teams Direct function shMappingTeamPlayers()',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $players_example,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'key_sh_mapping_hierarchy_sports',
                        'label' => 'Hierarchy Sports',
                        'name' => 'sh_mapping_hierarchy_sports',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Hierarchy Sports function shMappingHierarchySports()',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $hierarchy_sports,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'key_sh_mapping_hierarchy_family',
                        'label' => 'Hierarchy Family',
                        'name' => 'sh_mapping_hierarchy_family',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => "Hierarchy Sports Family function shMappingHierarchyFamily(7-68)",
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $hierarchy_family,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                    array(
                        'key' => 'key_sh_mapping_hierarchy_leagues',
                        'label' => 'Hierarchy Leagues',
                        'name' => 'sh_mapping_hierarchy_leagues',
                        'aria-label' => '',
                        'type' => 'select',
                        'instructions' => 'Hierarchy Leagues function shMappingHierarchyLeagues()',
                        'required' => 1,
                        'conditional_logic' => 0,
                        'wrapper' => array(
                            'width' => '100',
                            'class' => '',
                            'id' => '',
                        ),
                        'choices' => '',
                        'choices' => $hierarchy_league_example,
                        'default_value' => 1,
                        'return_format' => 'value',
                        'multiple' => 0,
                        'allow_null' => 0,
                        'ui' => 0,
                        'ajax' => 0,
                        'placeholder' => '',
                    ),
                ),
                'location' => array(
                    array(
                        array(
                            'param' => 'options_page',
                            'operator' => '==',
                            'value' => 'acf-options-sh-mapping-settings',
                        ),
                    ),
                ),
                'menu_order' => 1,
                'position' => 'normal',
                'style' => 'default',
                'label_placement' => 'top',
                'instruction_placement' => 'label',
                'hide_on_screen' => '',
                'active' => true,
                'description' => '',
                'show_in_rest' => 0,
            ));

            endif;    
    }

    public function save_settings_sh_mapping() {
        try {
            $option = (isset($_GET['page'])) ? $_GET['page'] : null;

            if (!empty($option) && $option == 'acf-options-sh-mapping-settings' && isset($_POST['acf']) && is_array($_POST['acf']) && count($_POST['acf']) > 0) {
                try {
                    $status = current($_POST['acf']);
                    $cron = next($_POST['acf']);
                    $url = next($_POST['acf']);
                    $url_season = next($_POST['acf']);
                    $url_sports = next($_POST['acf']);
                    $url_leagues = next($_POST['acf']);
                    $url_players = next($_POST['acf']);
                    $token = next($_POST['acf']);
                    $url_hierarchy = next($_POST['acf']);
                    $token_hierarchy = next($_POST['acf']);
                    $reset = next($_POST['acf']);
                    $load_players = next($_POST['acf']);

                    $set_option_status = (!empty($status) && $status == 'active') ? true : false;
                    update_option('sh_mapping_status', $set_option_status, true);
                    $original_cron = get_option('sh_mapping_cron_time');
                    $helpers = New Sh_Mapping_Helpers;

                    if ((!empty($cron)) && ($original_cron != $cron)) {
                        wp_clear_scheduled_hook('sh_mapping_cron_event');
                        wp_schedule_event(time(), $cron, 'sh_mapping_cron_event');
                        update_option('sh_mapping_cron_time', $cron);
                    }

                    if (!empty($reset) && $reset == 'reset') {
                        delete_option('sh_mapping_last_run');

                        $helpers->delMappingData();
                        $sports_mapped = $helpers->getMappingData();

                        if (!is_array($sports_mapped) || count($sports_mapped) == 0) {
                            error_log(print_r([
                                'message' => sprintf(__('SH Mapping: Error while trying to recover the Sports Mapping.', 'sh-mapping')),
                                'sports_mapped' => $sports_mapped
                            ], true));

                            set_transient('sh_mapping_error_message', __('SH Mapping: Error while trying to recover the Sports Mapping.', 'sh-mapping'), 30);

                            return;
                        }

                        $sports_leagues_options = $helpers->getSports($sports_mapped);
                        $sports_leagues_titles = $helpers->getSportsTitles($sports_mapped);
                        $sports_teams_direct = $helpers->getSportsTeamsDirect($sports_mapped);
                        $sports_seasons = $helpers->getMappingDataSeasons($sports_mapped);
                        $sports_direct = $helpers->getMappingDataSports();
                        $leagues_direct = $helpers->getMappingDataLeagues();
                        $sports_only = $helpers->getSportsOnly($sports_mapped);

                        $mapped_hierarchy = $helpers->getMappingDataHierarchy();

                        if (!is_array($mapped_hierarchy) || count($mapped_hierarchy) == 0) {
                            error_log(print_r([
                                'message' => sprintf(__('SH Mapping: Error while trying to recover the Sports Mapping Hierarchy.', 'sh-mapping')),
                                'sports_mapped' => $mapped_hierarchy
                            ], true));

                            set_transient('sh_mapping_error_message', __('SH Mapping: Error while trying to recover Mapping Hierarchy data.', 'sh-mapping'), 30);

                            return;
                        }

                        $hierarchy_leagues = $helpers->getMappingHierarchyLeagues($mapped_hierarchy);
                    }

                    if (!empty($load_players) && $load_players == 'load') {
                        $cache_players_direct = SHMAPPING_PLUGIN_LOG_DIR.'/'.$helpers->getFileNameDataPlayers();
                        if (file_exists($cache_players_direct)) {
                            unlink($cache_players_direct);
                        }

                        $players = $helpers->loadMappingDataPlayers();
                    }

                    update_option('sh_mapping_last_run', date('Y-m-d H:i:s'));
                } catch (\Exception $exception) {
                    error_log(print_r([
                        'message' => $exception->getMessage(),
                    ], true));
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
                'option' => $option,
            ], true));
        }
    }

    function show_custom_messages() {
        if ($mensaje = get_transient('sh_mapping_error_message')) {
            ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php echo esc_html($mensaje); ?></p>
                </div>
            <?php
            delete_transient('sh_mapping_error_message');
        }
    }
}
