<?php

// Exit if accessed directly.

use function WP_CLI\Utils\is_json;

if (!defined('ABSPATH')) exit;

/**
 * Class Sh_Mapping_Helpers
 *
 * This class contains repetitive functions that
 * are used globally within the plugin.
 *
 * @package        SHMAPPING
 * @subpackage    Classes/Sh_Mapping_Helpers
 * @author        IT
 * @since        1.0.0
 */

class Sh_Mapping_Helpers {
    private $redis_cache;
    private $redis_cache_group;

    function __construct() {
        $this->redis_cache = $this->initRedis();
        $this->setRedisGroup();
    }

    public function initRedis() {
        global $wp_object_cache;

        if (isset($wp_object_cache->redis) && $wp_object_cache->redis instanceof \Redis) {
            return true;
        }

        return false;
    }

    public function setRedisGroup($group = '') {
        $this->redis_cache_group = $group;
    }

    public function getRedis($key) {
        return wp_cache_get($key, $this->redis_cache_group);
    }

    public function setRedis($key, $data, $expire = 0) {
        wp_cache_set($key, $data, $this->redis_cache_group, $expire);

        return true;
    }

    public function getRedisKeyMapping() {
        return REDIS_KEY_SHMAPPING;
    }

    public function getRedisKeySportsLeaguesOptions() {
        return 'sh_mapping_sports_leagues_options';
    }

    public function getRedisKeySportsOnly() {
        return 'sh_mapping_sports_only';
    }

    public function getRedisKeySportsLeaguesTitles() {
        return 'sh_mapping_sports_leagues_titles';
    }

    public function getRedisKeySportsLeaguesAllData() {
        return 'sh_mapping_sports_leagues_all_data';
    }

    public function getRedisKeyLeaguesDirect() {
        return 'sh_mapping_leagues_direct';
    }

    public function getRedisKeyTeamsDirect() {
        return 'sh_mapping_teams_direct';
    }

    public function getRedisKeySeasonsDirect() {
        return 'sh_mapping_sasons_direct';
    }

    public function getRedisKeySportsDirect() {
        return 'sh_mapping_sports_direct';
    }

    public function getRedisKeyLeaguesSportDirect() {
        return 'sh_mapping_leagues_sport_direct';
    }

    public function getRedisKeyHierarchySports() {
        return 'sh_mapping_hierarchy_sports';
    }

    public function getRedisKeyHandicappers() {
        return 'sh_mapping_handicappers';
    }

    public function getRedisKeyMemberships() {
        return 'sh_mapping_memberships';
    }

    public function getRedisKeyMembershipsLogic() {
        return 'sh_mapping_memberships_logic';
    }

    public function getRedisKeyLeaderboardData() {
        return 'sh_mapping_leaderboard_data';
    }

    public function getRedisKeyLeaderboardSports() {
        return 'sh_mapping_leaderboard_sports';
    }

    public function getRedisKeyLeaderboardCategories() {
        return 'sh_mapping_leaderboard_categories';
    }

    public function getRedisKeyHierarchySportsLeagues() {
        return 'sh_mapping_hierarchy_sports_leagues';
    }

    public function getMappingData()
    {
        $sports_mapped = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);

        if ($this->initRedis()) {
            $key = $this->getRedisKeyMapping();
            $mapping = $this->getRedis($key);
    
            if (!empty($mapping)) {
                $unserialize_data = unserialize($mapping);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameData();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && is_serialized($data)) {
                    $data_unserialize = unserialize($data);
                    if (is_array($data_unserialize)) {
                        return unserialize($data);
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        $mapping_url = $this->getRequestMapping();
        $execute_request_mapping = $this->exeRequest($mapping_url, $file_path_mapped);
        
        if ($execute_request_mapping) {
            $parse_results_mapping = $this->parseFeedFile($file_path_mapped);
            
            if (!empty($parse_results_mapping) && isset($parse_results_mapping['data-content']) && is_array($parse_results_mapping['data-content']) && isset($parse_results_mapping['data-content']['sports']) && isset($parse_results_mapping['data-content']['leagues']) && isset($parse_results_mapping['data-content']['teams'])) {
                $sports = $parse_results_mapping['data-content']['sports'];
                $leagues = $parse_results_mapping['data-content']['leagues'];
                $teams = $parse_results_mapping['data-content']['teams'];

                if (is_array($sports) && count($sports) > 0) {
                    foreach ($sports as $sport) {
                        $sports_mapped[$sport['id']] = [
                            'id' => $sport['id'],
                            'sports_direct' => $sport['sports_direct'],
                            'name' => $sport['name'],
                            'logo' => $sport['logo'],
                        ];

                        $sports_mapped[$sport['id']]['sports'] = [
                            'masterfeed' => $sport['field_level_media'],
                            'members' => $sport['handicappers'],
                        ];
    
                        if (is_array($leagues) && count($leagues) > 0) {
                            foreach ($leagues as $league) {
                                if (!isset($league['mapped_sport_id']) || (isset($league['mapped_sport_id']) && $league['mapped_sport_id'] != $sport['id'])) continue;
            
                                $sports_mapped[$sport['id']]['leagues'][$league['id']] = [
                                    'id' => $league['id'],
                                    'sports_direct' => $league['sports_direct'],
                                    'name' => $league['name'],
                                    'logo' => $league['logo'],
                                    'masterfeed' => $league['field_level_media'],
                                    'members' => $league['handicappers'],
                                ];

                                if (is_array($teams) && count($teams) > 0) {
                                    foreach ($teams as $team) {
                                        if (!isset($team['mapped_league_id']) || (isset($team['mapped_league_id']) && $team['mapped_league_id'] != $league['id'])) continue;
                    
                                        $sports_mapped[$sport['id']]['leagues'][$league['id']]['teams'][$team['id']] = [
                                            'id' => $team['id'],
                                            'sports_direct' => $team['sports_direct'],
                                            'name' => $team['name'],
                                            'logo' => $team['logo'],
                                            'masterfeed' => $team['field_level_media'],
                                            'members' => $team['handicappers'],
                                        ];
                                    }
                                }
                            }
                        }
                    }
                }

                if ($this->initRedis() && is_array($sports_mapped) && count($sports_mapped) > 0) {
                    $this->setRedis($key, serialize($sports_mapped), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        return $sports_mapped;
    }

    public function getMappingLeaderboardData()
    {
        $sports_mapped = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyLeaderboardData();

        if ($this->initRedis()) {
            $leaderboard_data = $this->getRedis($key);
    
            if (!empty($leaderboard_data) && $this->isJson($leaderboard_data)) {
                $unserialize_data = json_decode($leaderboard_data, true);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameLeaderboard();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && $this->isJson($data)) {
                    $data_unserialize = json_decode($data, true);
                    if (is_array($data_unserialize)) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        $sports_url = $this->getRequestLeaderboardSports();
        $categories_url = $this->getRequestLeaderboardCategories();
        $execute_request = $this->exeRequest($sports_url, $file_path_mapped);
        
        if ($execute_request) {
            $leaderboard_sports = $this->parseFeedFile($file_path_mapped);

            if (is_array($leaderboard_sports) && count($leaderboard_sports) > 0) {
                foreach ($leaderboard_sports as $sport) {
                    $sports_mapped[$sport['id']] = [
                        'id' => $sport['id'],
                        'name' => $sport['sport_name'],
                        'categories' => []
                    ];

                    $category_url = $categories_url.'&sport='.$sport['id'];
                    $category_data = $this->exeRequestSimple($category_url);

                    if (is_array($category_data) && count($category_data) > 0) {
                        foreach ($category_data as $category) {
                            $sports_mapped[$sport['id']]['categories'][$category['id']] = [
                                'id' => $category['id'],
                                'name' => $category['description'],
                            ];
                        }
                    }
                }

                if (file_exists($file_path_mapped)) unlink($file_path_mapped);
                file_put_contents($file_path_mapped, json_encode($sports_mapped));

                if ($this->initRedis() && is_array($sports_mapped) && count($sports_mapped) > 0) {
                    $this->setRedis($key, json_encode($sports_mapped), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        return $sports_mapped;
    }

    public function getMappingDataHandicappers()
    {
        $handicappers_mapped = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyHandicappers();

        if ($this->initRedis()) {
            $handicappers = $this->getRedis($key);
    
            if (!empty($handicappers)) {
                $unserialize_data = unserialize($handicappers);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameHandicappers();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);
                

                if (!empty($data) && $this->isJson($data)) {
                    $data_unserialize = json_decode($data, true);

                    if (is_array($data_unserialize)) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        $mapping_url = $this->getRequestMappingHandicappers();
        $response_handicappers = $this->exeRequest($mapping_url, $file_path_mapped);
        
        if ($response_handicappers) {
            $handicappers = $this->parseFeedFile($file_path_mapped);
            
            if (is_array($handicappers) && count($handicappers) > 0) {
                foreach ($handicappers as $handicapper) {
                    $handicappers_mapped[$handicapper['id']] = $handicapper['name'];
                }

                asort($handicappers_mapped);

                if (file_exists($file_path_mapped)) unlink($file_path_mapped);
                file_put_contents($file_path_mapped, json_encode($handicappers_mapped));

                if ($this->initRedis() && is_array($handicappers_mapped) && count($handicappers_mapped) > 0) {
                    $this->setRedis($key, serialize($handicappers_mapped), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        return $handicappers_mapped;
    }

    public function getMappingDataHierarchy()
    {
        $clear_hierarchies = [];

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameDataHierarchy();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && $this->isJson($data)) {
                    $data_unserialize = json_decode($data, true);
                    if (is_array($data_unserialize)) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        $mapping_url = $this->getRequestMappinghierarchy();
        $hierarchy_request = $this->exeRequestSimple($mapping_url);
        
        if (!empty($hierarchy_request) && is_array($hierarchy_request) && count($hierarchy_request) > 0) {
            $order_hierarchies = $this->orderHierarchies($hierarchy_request);
            $clear_hierarchies = $this->clearHierarchy($order_hierarchies);

            file_put_contents($file_path_mapped, json_encode($clear_hierarchies));
        }

        return $clear_hierarchies;
    }

    public function getMappingMembershipsData()
    {
        $memberships_mapped = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyMemberships();

        if ($this->initRedis()) {
            $memberships = $this->getRedis($key);
    
            if (!empty($memberships)) {
                $unserialize_data = unserialize($memberships);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameDataMemberships();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data)) {
                    $data_decode = json_decode($data, true);
                    
                    return $data_decode;
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        $memberships_url = $this->getRequestMembership();
        $execute_request_mapping = $this->exeRequest($memberships_url, $file_path_mapped);

        if ($execute_request_mapping) {
            $memberships = $this->parseFeedFile($file_path_mapped);
            
            if (is_array($memberships) && count($memberships) > 0) {
                foreach ($memberships as $membership) {
                    $memberships_mapped[$membership['id']] = $membership;
                }

                if ($this->initRedis() && is_array($memberships_mapped) && count($memberships_mapped) > 0) {
                    $this->setRedis($key, serialize($memberships_mapped), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        return $memberships_mapped;
    }

    public function orderHierarchies(&$items, $parentId = null) {
        $result = [];

        foreach ($items as &$item) {
            if ($item['id_parent_group_external'] === $parentId) {
                $children = $this->orderHierarchies($items, $item['id_group_external']);

                if ($children) {
                    $item['groups'] = $children;
                } else {
                    $item['groups'] = [];
                }

                $result[] = $item;
            }
        }

        return $result;
    }

    public function clearHierarchy($data) {
        $result = [];

        foreach ($data as $item) {
            $key = $item['id'];
            $group_name = trim($item['group_name']);
            $group = ['group_name' => $group_name, 'items' => []];

            if (isset($item['groups'])) {
                $group['items'] = $this->clearHierarchy($item['groups']);
            }

            if (isset($item['leagues'])) {
                foreach ($item['leagues'] as $league) {
                    $league_key = $league['id_league']['id'];
                    $group['leagues']['L'.$league_key] = $league['id_league']['description'];
                }
            }

            $result['G'.$key] = $group;
        }

        return $result;
    }

    public function getMappingHierarchySports($hierarchy_data)
    {
        $hierarchy_sports = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyHierarchySports();

        if (!is_array($hierarchy_data) || count($hierarchy_data) == 0) return [];
        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);

        foreach ($hierarchy_data as $hierarchy_key => $hierarchy) {
            $hierarchy_sports[$hierarchy_key] = $hierarchy['group_name'];
        }

        asort($hierarchy_sports);

        if ($this->initRedis() && is_array($hierarchy_sports) && count($hierarchy_sports) > 0) {
            $this->setRedis($key, json_encode($hierarchy_sports), SHMAPPING_REDIS_TIMEOUT);
        }

        return $hierarchy_sports;
    }

    public function getLeagues($array, &$hierarchy_leagues) {
        if (isset($array['leagues']) && is_array($array['leagues']) && count($array['leagues']) > 0) {
            foreach ($array['leagues'] as $key => $league) {
                $hierarchy_leagues[$key] = $league;
            }
        }
        if (isset($array['items'])) {
            foreach ($array['items'] as $group) {
                $this->getLeagues($group, $hierarchy_leagues);
            }
        }
    }

    public function getMappingHierarchyFilter($hierarchy_data, $id)
    {
        if (!is_array($hierarchy_data) || count($hierarchy_data) == 0 || empty($id)) return [];

        $hierarchies = [];
        $indexes = explode('-', $id);
        $end = $indexes[count($indexes)-1];

        $this->navigateArrayRecursively($hierarchy_data, $indexes, $end, $hierarchies);

        return $hierarchies;
    }

    public function navigateArrayRecursively($array, $indexes, $end, &$hierarchies)
    {
        $currentIndex = array_shift($indexes);
        
        if (isset($array[$currentIndex])) {
            if ($currentIndex != $end) {
                return $this->navigateArrayRecursively($array[$currentIndex]['items'], $indexes, $end, $hierarchies);
            }

            $items = [];
            $leagues = [];

            if (isset($array[$currentIndex]['items']) && 
                is_array($array[$currentIndex]['items']) && 
                count($array[$currentIndex]['items']) > 0) {
                foreach ($array[$currentIndex]['items'] as $key => $item) {
                    $items[$key.'-child'] = $item['group_name'];
                }
            }

            if (isset($array[$currentIndex]['leagues']) && 
                is_array($array[$currentIndex]['leagues']) && 
                count($array[$currentIndex]['leagues']) > 0) {
                $leagues = $array[$currentIndex]['leagues'];
            }

            $hierarchies = $items + $leagues;
        }

        return [];
    }

    public function getMappingDataPlayers()
    {
        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameDataPlayers();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data)) {
                    $data_unserialize = json_decode($data, true);
                    
                    return $data_unserialize;
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return [];
    }

    public function loadMappingDataPlayers()
    {
        $players_mapped = [];

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameDataPlayers();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        $mapping_url = $this->getRequestMappingPlayers();
        $request_mapping = $this->exeRequestSimple($mapping_url);

        if (is_array($request_mapping) && count($request_mapping) > 0) {
            if (is_array($request_mapping[0]['data-content'][0][0]['players']) && count($request_mapping[0]['data-content'][0][0]['players']) > 0) {
                $players = $request_mapping[0]['data-content'][0][0]['players'];

                foreach ($players as $player) {
                    $players_mapped[$player['id']] = [
                        'id' => $player['id'],
                        'first_name' => $player['first_name'],
                        'last_name' => $player['last_name'],
                    ];
                }

                if (is_array($players_mapped) && count($players_mapped) > 0) {
                    file_put_contents($file_path_mapped, json_encode($players_mapped));
                    error_log(sprintf(__('JSON response saved to: %s', 'sh-mapping'), $file_path_mapped));
                }
            }
        }

        return $players_mapped;
    }

    public function getMappingDataSeasons($data)
    {
        $seasons_array = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);

        if ($this->initRedis()) {
            $key = $this->getRedisKeySeasonsDirect();
            $seasons_data = $this->getRedis($key);
    
            if (!empty($seasons_data)) {
                $unserialize_data = unserialize($seasons_data);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameSeasonsDirect();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && is_serialized($data)) {
                    $data_unserialize = unserialize($data);
                    if (is_array($data_unserialize)) {
                        return unserialize($data);
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        $mapping_url = $this->getRequestMappingSeasons();
        $execute_request_mapping_seasons = $this->exeRequest($mapping_url, $file_path_mapped);
        
        if ($execute_request_mapping_seasons) {
            $parse_results_mapping = $this->parseFeedFile($file_path_mapped);

            if (!empty($parse_results_mapping) && isset($parse_results_mapping[0]['data-content'][0][0]['seasons']) && is_array($parse_results_mapping[0]['data-content'][0][0]['seasons'])) {
                $seasons = $parse_results_mapping[0]['data-content'][0][0]['seasons'];

                foreach ($seasons as $season) {
                    $seasons_array[$season['league_id']][$season['id']] = $season['name'];
                }

                if ($this->initRedis() && is_array($seasons_array) && count($seasons_array) > 0) {
                    $this->setRedis($key, serialize($seasons_array), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        try {
            file_put_contents($file_path_mapped, serialize($seasons_array));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $seasons_array;
    }

    public function getMappingDataSports()
    {
        $sports_data = [];
        $key = $this->getRedisKeySportsDirect();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);
    
            if (!empty($data_sports)) {
                $unserialize_data = unserialize($data_sports);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameSportsDirect();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && is_serialized($data)) {
                    $data_unserialize = unserialize($data);
                    if (is_array($data_unserialize)) {
                        return unserialize($data);
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        $mapping_url = $this->getRequestMappingSports();
        $execute_request_mapping = $this->exeRequest($mapping_url, $file_path_mapped);
        
        if ($execute_request_mapping) {
            $parse_results_mapping = $this->parseFeedFile($file_path_mapped);

            if (!empty($parse_results_mapping) && isset($parse_results_mapping[0]['data-content'][0][0]['sports']) && is_array($parse_results_mapping[0]['data-content'][0][0]['sports']) && count($parse_results_mapping[0]['data-content'][0][0]['sports']) > 0) {
                $sports = $parse_results_mapping[0]['data-content'][0][0]['sports'];

                foreach ($sports as $sport) {
                    $sports_data[$sport['id']] = $this->getCapitalName($sport['name']);
                }

                asort($sports_data);

                if ($this->initRedis() && is_array($sports_data) && count($sports_data) > 0) {
                    $this->setRedis($key, serialize($sports_data), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        try {
            file_put_contents($file_path_mapped, serialize($sports_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $sports_data;
    }

    public function getMappingDataLeagues()
    {
        $leagues_data = [];
        $key = $this->getRedisKeyLeaguesSportDirect();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);
    
            if (!empty($data_sports)) {
                $unserialize_data = unserialize($data_sports);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameLeaguesSportDirect();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && is_serialized($data)) {
                    $data_unserialize = unserialize($data);
                    if (is_array($data_unserialize)) {
                        return unserialize($data);
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        $mapping_url = $this->getRequestMappingLeagues();
        $execute_request_mapping = $this->exeRequest($mapping_url, $file_path_mapped);
        
        if ($execute_request_mapping) {
            $parse_results_mapping = $this->parseFeedFile($file_path_mapped);

            if (!empty($parse_results_mapping) && isset($parse_results_mapping[0]['data-content'][0][0]['leagues']) && is_array($parse_results_mapping[0]['data-content'][0][0]['leagues']) && count($parse_results_mapping[0]['data-content'][0][0]['leagues']) > 0) {
                $leagues = $parse_results_mapping[0]['data-content'][0][0]['leagues'];

                foreach ($leagues as $league) {
                    $leagues_data[$league['sport_id']][] = [
                        'id' => $league['id'],
                        'name' => $league['name'],
                        'nickname' => $league['nickname'],
                    ];
                }

                asort($leagues_data);

                if ($this->initRedis() && is_array($leagues_data) && count($leagues_data) > 0) {
                    $this->setRedis($key, serialize($leagues_data), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        try {
            file_put_contents($file_path_mapped, serialize($leagues_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $leagues_data;
    }

    private function getCapitalName($name) {
        if ($name == 'horseracing') {
            $name = 'horse racing';
        }

        return ucwords($name);
    }

    public function delMappingData()
    {
        if ($this->initRedis()) {
            try {
                $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
                $key_sports = $this->getRedisKeyMapping();
                $key_sports_options = $this->getRedisKeySportsLeaguesOptions();
                $key_sports_titles = $this->getRedisKeySportsLeaguesTitles();
                $key_sports_direct = $this->getRedisKeySportsDirect();
                $key_leagues_sport_direct = $this->getRedisKeyLeaguesSportDirect();
                $key_leagues_direct = $this->getRedisKeyLeaguesDirect();
                $key_teams_direct = $this->getRedisKeyTeamsDirect();
                $key_seasons_direct = $this->getRedisKeySeasonsDirect();
                $key_sports_only = $this->getRedisKeySportsOnly();
                $hierarchy_sports = $this->getRedisKeyHierarchySports();
                $key_handicappers = $this->getRedisKeyHandicappers();
                $key_memberships = $this->getRedisKeyMemberships();
                $key_memberships_logic = $this->getRedisKeyMembershipsLogic();
                $key_leaderboard = $this->getRedisKeyLeaderboardData();
                $key_leaderboard_sports = $this->getRedisKeyLeaderboardSports();
                $key_leaderboard_categories = $this->getRedisKeyLeaderboardCategories();
                $key_hierarchy_sports_leagues = $this->getRedisKeyHierarchySportsLeagues();

                $this->setRedis($key_sports, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_sports_options, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_sports_titles, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_sports_direct, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_leagues_direct, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_teams_direct, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_seasons_direct, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_leagues_sport_direct, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_sports_only, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($hierarchy_sports, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_handicappers, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_memberships, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_memberships_logic, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_leaderboard, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_leaderboard_sports, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_leaderboard_categories, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_hierarchy_sports_leagues, json_encode([]), SHMAPPING_REDIS_TIMEOUT);

                error_log('SH Mapping: Clear cache.');
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => $exception->getMessage(),
                    'keys' => [
                        $key_sports, $key_sports_options, $key_sports_titles, $key_teams_direct, $key_leagues_direct, 
                        $key_seasons_direct, $key_sports_direct, $key_leagues_sport_direct, $key_sports_only,
                        $hierarchy_sports, $key_handicappers, $key_memberships, $key_memberships_logic,
                        $key_leaderboard, $key_leaderboard_sports, $key_leaderboard_categories, $key_hierarchy_sports_leagues
                    ],
                ], true));
            }
        }

        try {
            $cache_data = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameData();
            if (file_exists($cache_data)) {
                unlink($cache_data);
            }

            $cache_sports_options = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameSportsOptions();
            if (file_exists($cache_sports_options)) {
                unlink($cache_sports_options);
            }

            $cache_sports_titles = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameTitles();
            if (file_exists($cache_sports_titles)) {
                unlink($cache_sports_titles);
            }

            $cache_sports_direct = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameSportsDirect();
            if (file_exists($cache_sports_direct)) {
                unlink($cache_sports_direct);
            }

            $cache_leagues_sport_direct = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameLeaguesSportDirect();
            if (file_exists($cache_leagues_sport_direct)) {
                unlink($cache_leagues_sport_direct);
            }

            $cache_leagues_direct = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameLeaguesDirect();
            if (file_exists($cache_leagues_direct)) {
                unlink($cache_leagues_direct);
            }

            $cache_teams_direct = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameTeamsDirect();
            if (file_exists($cache_teams_direct)) {
                unlink($cache_teams_direct);
            }

            $cache_seasons_direct = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameSeasonsDirect();
            if (file_exists($cache_seasons_direct)) {
                unlink($cache_seasons_direct);
            }

            $cache_sports_only = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameSportsOnly();
            if (file_exists($cache_sports_only)) {
                unlink($cache_sports_only);
            }

            $cache_hierarchy = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameDataHierarchy();
            if (file_exists($cache_hierarchy)) {
                unlink($cache_hierarchy);
            }

            $cache_handicappers = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameHandicappers();
            if (file_exists($cache_handicappers)) {
                unlink($cache_handicappers);
            }

            $cache_memberships = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameDataMemberships();
            if (file_exists($cache_memberships)) {
                unlink($cache_memberships);
            }

            $cache_memberships_logic = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileMembersLogic();
            if (file_exists($cache_memberships_logic)) {
                unlink($cache_memberships_logic);
            }

            $cache_leaderboard = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameLeaderboard();
            if (file_exists($cache_leaderboard)) {
                unlink($cache_leaderboard);
            }

            $cache_leaderboard_sports = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameLeaderboardSports();
            if (file_exists($cache_leaderboard_sports)) {
                unlink($cache_leaderboard_sports);
            }

            $cache_leaderboard_categories = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameLeaderboardCategories();
            if (file_exists($cache_leaderboard_categories)) {
                unlink($cache_leaderboard_categories);
            }

            $cache_hierarchy_sports_leagues = SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameHierarchySportsLeagues();
            if (file_exists($cache_hierarchy_sports_leagues)) {
                unlink($cache_hierarchy_sports_leagues);
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
            ], true));
        }
    }

    private function getRequestMapping()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_url', 'options')))
            ? get_field('sh_mapping_url', 'options') 
            : 'http://masterfeed.shnadmin.com/feed/mapped/get_data';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');

        
        $api_request = $api_base.'?key='.$api_token;

        return $api_request;
    }

    private function getRequestMappinghierarchy()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_url_hierarchy', 'options')))
            ? get_field('sh_mapping_url_hierarchy', 'options') 
            : 'https://members.sportshub.com/api/get-hierarchy';
        $api_token = get_field('sh_mapping_members_token', 'option');

        
        $api_request = $api_base.'?api_token='.$api_token;

        return $api_request;
    }

    private function getRequestMappingPlayers()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_url_players', 'options')))
            ? get_field('sh_mapping_url_players', 'options') 
            : 'http://masterfeed.shnadmin.com/feed/sportdirect/players';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');

        
        $api_request = $api_base.'?key='.$api_token;

        return $api_request;
    }

    private function getRequestMappingSports()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_url_sports', 'options')))
            ? get_field('sh_mapping_url_sports', 'options') 
            : 'http://masterfeed.shnadmin.com/feed/sportdirect/sports';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');

        
        $api_request = $api_base.'?key='.$api_token;

        return $api_request;
    }

    private function getRequestMappingLeagues()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_url_leagues', 'options')))
            ? get_field('sh_mapping_url_leagues', 'options') 
            : 'http://masterfeed.shnadmin.com/feed/sportdirect/leagues';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');

        
        $api_request = $api_base.'?key='.$api_token;

        return $api_request;
    }

    private function getRequestMappingSeasons()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_url_seasons', 'options')))
            ? get_field('sh_mapping_url_seasons', 'options') 
            : 'http://masterfeed.shnadmin.com/feed/sportdirect/seasons';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');
        
        $api_request = $api_base.'?key='.$api_token;

        return $api_request;
    }

    private function getRequestMappingHandicappers()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_url_handicappers', 'options')))
            ? get_field('sh_mapping_url_handicappers', 'options') 
            : 'https://members.sportshub.com/api/handicappers/data';
        $api_token = get_field('sh_mapping_members_token', 'option');

        
        $api_request = $api_base.'?api_token='.$api_token;

        return $api_request;
    }

    private function getRequestMembership()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_membership_url', 'options')))
            ? get_field('sh_mapping_membership_url', 'options') 
            : 'https://members.sportshub.com/api/membership-info';
        $api_token = get_field('sh_mapping_members_token', 'option');

        
        $api_request = $api_base.'?api_token='.$api_token;

        return $api_request;
    }

    private function getRequestLeaderboardSports()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_leaderboard_sports_url', 'options')))
            ? get_field('sh_mapping_leaderboard_sports_url', 'options') 
            : 'https://members.sportshub.com/api/leaderboardData/sports';
        $api_token = get_field('sh_mapping_members_token', 'option');
        
        $api_request = $api_base.'?api_token='.$api_token;

        return $api_request;
    }

    private function getRequestLeaderboardCategories()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_leaderboard_categories_url', 'options')))
            ? get_field('sh_mapping_leaderboard_categories_url', 'options') 
            : 'https://members.sportshub.com/api/leaderboardData/categories';
        $api_token = get_field('sh_mapping_members_token', 'option');
        
        $api_request = $api_base.'?api_token='.$api_token;

        return $api_request;
    }

    private function exeRequest($request, $file_path)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $request);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);

        if ($response === false) {
            curl_close($ch);
            error_log(sprintf(__('SH Mapping: ERROR: %s.', 'sh-mapping'), curl_error($ch)));
            WP_CLI::line(sprintf(__('ERROR: %s.', 'sh-mapping'), curl_error($ch)));
        } else {
            if ($this->isJson($response)) {
                $response_json = json_decode($response, true);

                if (isset($response_json['error'])) {
                    error_log(print_r([
                        'response_json' => $response_json,
                        'request' => $request,
                    ], true));

                    return false;
                }
            }

            file_put_contents($file_path, $response);
            curl_close($ch);

            error_log(sprintf(__('JSON response saved to: %s', 'sh-mapping'), $file_path));

            return true;
        }

        return false;
    }

    private function exeRequestSimple($request)
    {
        $response_json = [];
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $request);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);

        if ($response === false) {
            curl_close($ch);
            error_log(sprintf(__('SH Mapping: ERROR: %s.', 'sh-mapping'), curl_error($ch)));
            WP_CLI::line(sprintf(__('ERROR: %s.', 'sh-mapping'), curl_error($ch)));
        } else {
            if ($this->isJson($response)) {
                $response_json = json_decode($response, true);

                if (isset($response_json['error'])) {
                    error_log(print_r([
                        'response_json' => $response_json,
                    ], true));

                    return null;
                }
            }

            curl_close($ch);

            return $response_json;
        }

        return $response_json;
    }

    function isJson($string) {
        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }

    private function parseFeedFile($file_path)
    {
        $file_contents = file_get_contents($file_path);

        if ($file_contents === false) {
            error_log(__('SH Mapping Error: Failed to read the file.', 'sh-mapping'));

            return null;
        }

        $json_data = json_decode($file_contents, true);

        if ($json_data === null) {
            error_log(__('SH Mapping Error: Failed to decode the JSON.', 'sh-mapping'));

            return null;
        }

        return $json_data;
    }

    public function getSports($data)
    {
        $sports_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeySportsLeaguesOptions();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);
    
            if (!empty($data_sports)) {
                $unserialize_data = unserialize($data_sports);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                $leagues = [];
                $leagues[$sport['id']] = $sport['name'] .' - '. __('Any Category');

                foreach ($sport['leagues'] as $league) {
                    $leagues[$sport['id'].'-'.$league['id']] = $league['name'];
                }
            } else {
                $leagues = $sport['name'];
            }

            if (is_array($leagues)) {
                $sports_data[$sport['name']] = $leagues;
            } else {
                $sports_data[$sport['id']] = $leagues;
            }
        }

        if ($this->initRedis() && is_array($sports_data) && count($sports_data) > 0) {
            $this->setRedis($key, serialize($sports_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_sports =  $this->getFileNameSportsOptions();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;
    
            file_put_contents($file_path_sports, serialize($sports_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $sports_data;
    }

    public function getSportsOnly($data)
    {
        $sports_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeySportsOnly();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);
    
            if (!empty($data_sports)) {
                $unserialize_data = unserialize($data_sports);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            $sports_data[$sport['id']] = $sport['name'];
        }

        if ($this->initRedis() && is_array($sports_data) && count($sports_data) > 0) {
            $this->setRedis($key, serialize($sports_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_sports =  $this->getFileNameSportsOnly();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;
    
            file_put_contents($file_path_sports, serialize($sports_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $sports_data;
    }

    public function getSportsTitles($data)
    {
        $sport_titles = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeySportsLeaguesTitles();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);
    
            if (!empty($data_sports)) {
                $unserialize_data = unserialize($data_sports);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                $sport_titles[$sport['id']] = $sport['name'];

                foreach ($sport['leagues'] as $league) {
                    $sport_titles[$sport['id'].'-'.$league['id']] = $sport['name'].' | '.$league['name'];
                }
            } else {
                $sport_titles[$sport['id']] = $sport['name'];
            }
        }

        if ($this->initRedis() && is_array($sport_titles) && count($sport_titles) > 0) {
            $this->setRedis($key, serialize($sport_titles), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_titles =  $this->getFileNameTitles();
            $file_path_titles = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_titles;
    
            file_put_contents($file_path_titles, serialize($sport_titles));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $sport_titles;
    }

    public function getSportsLeaguesDirect($data)
    {
        $leagues_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyLeaguesDirect();

        if ($this->initRedis()) {
            $data_leagues = $this->getRedis($key);
    
            if (!empty($data_leagues)) {
                $unserialize_data = unserialize($data_leagues);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            $sport_name = $sport['name'];
            $leagues = [];

            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                foreach ($sport['leagues'] as $league_id => $league) {
                    if (isset($league['sports_direct'][0]) && !empty($league['sports_direct'][0])) {
                        $leagues[$league['sports_direct'][0]] = $league['name'];
                    }
                }
            }

            if (is_array($leagues) && count($leagues) > 0) {
                $leagues_data[$sport_name] = $leagues;
            }
        }

        if ($this->initRedis() && is_array($leagues_data) && count($leagues_data) > 0) {
            $this->setRedis($key, serialize($leagues_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_sports =  $this->getFileNameLeaguesDirect();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;
    
            file_put_contents($file_path_sports, serialize($leagues_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $leagues_data;
    }

    public function getSportsTeamsDirect($data)
    {
        $teams_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyTeamsDirect();

        if ($this->initRedis()) {
            $data_teams = $this->getRedis($key);
    
            if (!empty($data_teams)) {
                $unserialize_data = unserialize($data_teams);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                foreach ($sport['leagues'] as $league_id => $league) {
                    $teams = [];
                    $legue_id = (isset($league['sports_direct']) && is_array($league['sports_direct']) && count($league['sports_direct']) > 0)
                        ? $league['sports_direct'][0]
                        : $league['id'];

                    if (isset($league['teams']) && is_array($league['teams']) && count($league['teams']) > 0) {
                        foreach ($league['teams'] as $team_id => $team) {
                            if (isset($team['sports_direct'][0]) && !empty($team['sports_direct'][0])) {
                                $teams[$team['sports_direct'][0]] = $team['name'];
                            }
                        }
                    }

                    if (is_array($teams) && count($teams) > 0) {
                        $teams_data[$legue_id] = $teams;
                    }
                }
            }
        }

        if ($this->initRedis() && is_array($teams_data) && count($teams_data) > 0) {
            $this->setRedis($key, serialize($teams_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_sports =  $this->getFileNameTeamsDirect();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;
    
            file_put_contents($file_path_sports, serialize($teams_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $teams_data;
    }

    public function getLeaguesNames($data)
    {
        $leagues_array = [];

        foreach ($data as $league => $seasons) {
            foreach ($seasons as $id => $name) {
                $leagues_array[$id] = $name;
            }
        }

        return $leagues_array;
    }

    public function getLeaguesSport($leagues, $sport_id)
    {
        if (empty($sport_id) || !isset($leagues[$sport_id]) || !is_array($leagues[$sport_id]) || count($leagues[$sport_id]) == 0) return [];

        $leagues_array = [];
        foreach ($leagues[$sport_id] as $league) {
            $leagues_array[$league['id']] = $league['name'];
        }

        asort($leagues_array);

        return $leagues_array;
    }

    public function getTeamsLeague($teams, $league_id)
    {
        if (empty($league_id) || !isset($teams[$league_id]) || !is_array($teams[$league_id]) || count($teams[$league_id]) == 0) return [];

        $teams_array = (isset($teams[$league_id])) ? $teams[$league_id] : [];
        asort($teams_array);

        return $teams_array;
    }

    public function getSeasonsLeague($seasons, $league_id)
    {
        if (empty($league_id) || !isset($seasons[$league_id]) || !is_array($seasons[$league_id]) || count($seasons[$league_id]) == 0) return [];

        $seasons_array = (isset($seasons[$league_id])) ? $seasons[$league_id] : [];
        asort($seasons_array);

        return $seasons_array;
    }

    public function getMembersLogic($configs)
    {
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $memberships = [];
        $memberships_mapped = $this->getMappingMembershipsData();

        if (is_array($memberships_mapped) && count($memberships_mapped) > 0) {
            foreach ($memberships_mapped as $membership_key => $membership_data) {
                $memberships[$membership_data['id']] = [
                    'name' => (isset($membership_data['name']) && !empty($membership_data['name']))
                        ? $membership_data['name']
                        : $membership_data['name'],
                    'url' => $membership_data['url'],
                ];
            }
        }

        if (!is_array($memberships) || count($memberships) == 0) return [];

        $packages = [];
        foreach ($configs as $config_key => $config) {
            $key_new = $config['key'];
            if (empty($config['key'])) $key_new = 0;
            $data = $memberships[$config['value']];
            $packages[$key_new][] = $data;
        }

        if (!is_array($packages) || count($packages) == 0) return [];

        $key = $this->getRedisKeyMembershipsLogic();
        if ($this->initRedis()) {
            $this->setRedis($key, serialize($packages), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_members =  $this->getFileMembersLogic();
            $file_path_members = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_members;
    
            file_put_contents($file_path_members, serialize($packages));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $packages;
    }
    
    public function getLeaderboardSports($data)
    {
        if (!is_array($data) || count($data) == 0) return;
    
        $sports_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyLeaderboardSports();

        foreach ($data as $sport_id => $sport) {
            if (isset($sport['name']) && !empty($sport['name'])) {
                $sports_data[$sport['id']] = $sport['name'];
            }
        }

        if ($this->initRedis() && is_array($sports_data) && count($sports_data) > 0) {
            $this->setRedis($key, json_encode($sports_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_sports =  $this->getFileNameLeaderboardSports();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;
            file_put_contents($file_path_sports, json_encode($sports_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $sports_data;
    }

    public function getLeaderboardCategories($data)
    {
        if (!is_array($data) || count($data) == 0) return [];
    
        $categories_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyLeaderboardCategories();

        foreach ($data as $sport_id => $sport) {
            if (isset($sport['categories']) && is_array($sport['categories']) && count($sport['categories']) > 0) {
                $categories = [];
                $categories[$sport['id']] = $sport['name'] .' - '. __('Any Category');

                foreach ($sport['categories'] as $category) {
                    $categories[$sport['id'].'-'.$category['id']] = $category['name'];
                }
            } else {
                $categories = $sport['name'];
            }

            if (is_array($categories)) {
                $categories_data[$sport['name']] = $categories;
            } else {
                $categories_data[$sport['id']] = $categories;
            }
        }

        if ($this->initRedis() && is_array($categories_data) && count($categories_data) > 0) {
            $this->setRedis($key, json_encode($categories_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_categories =  $this->getFileNameLeaderboardCategories();
            $file_path_categories = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_categories;
            file_put_contents($file_path_categories, json_encode($categories_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $categories_data;
    }

    public function getFileNameData()
    {
        return 'SH_MAPPING.json';
    }

    public function getFileNameDataHierarchy()
    {
        return 'SH_MAPPING_HIERARCHY.json';
    }

    public function getFileNameDataPlayers()
    {
        return 'SH_MAPPING_PLAYERS.json';
    }

    public function getFileNameDataMemberships()
    {
        return 'SH_MAPPING_MEMBERSHIPS.json';
    }

    public function getFileNameSportsOptions()
    {
        return 'SH_MAPPING_SPORTS_OPTIONS.json';
    }

    public function getFileNameSportsOnly()
    {
        return 'SH_MAPPING_SPORTS_ONLY.json';
    }

    public function getFileNameTitles()
    {
        return 'SH_MAPPING_TITLES.json';
    }

    public function getFileNameLeaguesDirect()
    {
        return 'SH_MAPPING_LEAGUES_DIRECT.json';
    }

    public function getFileNameTeamsDirect()
    {
        return 'SH_MAPPING_TEAMS_DIRECT.json';
    }

    public function getFileNameSeasonsDirect()
    {
        return 'SH_MAPPING_SEASONS_DIRECT.json';
    }

    public function getFileNameSportsDirect()
    {
        return 'SH_MAPPING_SPORTS_DIRECT.json';
    }

    public function getFileNameLeaguesSportDirect()
    {
        return 'SH_MAPPING_LEAGUES_SPORT_DIRECT.json';
    }

    public function getFileNameHandicappers()
    {
        return 'SH_MAPPING_HANDICAPPERS.json';
    }

    public function getFileMembersLogic()
    {
        return 'SH_MAPPING_MEMBERS_LOGIC.json';
    }

    public function getFileNameLeaderboard()
    {
        return 'SH_MAPPING_LEADERBOARD.json';
    }

    public function getFileNameLeaderboardSports()
    {
        return 'SH_MAPPING_LEADERBOARD_SPORTS.json';
    }

    public function getFileNameLeaderboardCategories()
    {
        return 'SH_MAPPING_LEADERBOARD_CATEGORIES.json';
    }

    public function getFileNameHierarchySportsLeagues()
    {
        return 'SH_MAPPING_HIERARCHY_SPORTS_LEAGUES.json';
    }
}
