<?php
/**
 * SH Mapping 
 *
 * @package       SHMAPPING
 * @author        IT
 * @version       1.0.29
 *
 * @wordpress-plugin
 * Plugin Name:   SH Mapping 
 * Plugin URI:    https://www.sportshub.com
 * Description:   Sports Hub Mapping Plugin
 * Version:       1.0.29
 * Author:        IT
 * Author URI:    https://www.sportshub.com
 * Text Domain:   sh-mapping
 * Domain Path:   /languages
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) exit;

/**
 * HELPER COMMENT START
 * 
 * This file contains the main information about the plugin.
 * It is used to register all components necessary to run the plugin.
 * 
 * The comment above contains all information about the plugin 
 * that are used by WordPress to differenciate the plugin and register it properly.
 * It also contains further PHPDocs parameter for a better documentation
 * 
 * The function SHMAPPING() is the main function that you will be able to 
 * use throughout your plugin to extend the logic. Further information
 * about that is available within the sub classes.
 * 
 * HELPER COMMENT END
 */

// Plugin name
define('SHMAPPING_NAME', 'SH Mapping');
define('SHMAPPING_NAME_SLUG', 'sh-mapping');

// Plugin version
define('SHMAPPING_VERSION', '1.0.29');

// Plugin Root File
define('SHMAPPING_PLUGIN_FILE', __FILE__);

// Plugin base
define('SHMAPPING_PLUGIN_BASE', plugin_basename(SHMAPPING_PLUGIN_FILE));

// Plugin Folder Path
define('SHMAPPING_PLUGIN_DIR', plugin_dir_path(SHMAPPING_PLUGIN_FILE));

// Plugin Folder URL
define('SHMAPPING_PLUGIN_URL', plugin_dir_url(SHMAPPING_PLUGIN_FILE));

// UPLOAD DIR
$plugin_upload_dir = wp_upload_dir()['basedir'] . '/' . str_replace(' ', '_', SHMAPPING_NAME);
define('SHMAPPING_PLUGIN_LOG_DIR', $plugin_upload_dir);

// REDIS
define('REDIS_GROUP_SHMAPPING', 'sh-mapping');
define('REDIS_KEY_SHMAPPING', 'sh_mapping_data');

if (!defined('SHMAPPING_REDIS_TIMEOUT')) {
    define('SHMAPPING_REDIS_TIMEOUT', 3600);
}

/**
 * Load the main class for the core functionality
 */
require_once SHMAPPING_PLUGIN_DIR . 'core/class-sh-mapping.php';

/**
 * The main function to load the only instance
 * of our master class.
 *
 * @author  IT
 * @since   1.0.0
 * @return  object|Sh_Mapping
 */
function SHMAPPING() {
    return Sh_Mapping::instance();
}

$shmapping = SHMAPPING();

// Simple: If there are recent timeout errors, activate fallback mode
if (!get_transient('sh_mapping_simple_check_done')) {
    $debug_log_path = WP_CONTENT_DIR . '/debug.log';
    if (file_exists($debug_log_path)) {
        $recent_logs = file_get_contents($debug_log_path);
        if (strpos($recent_logs, 'Maximum execution time') !== false) {
            // Force fallback mode for 1 hour
            set_transient('sh_mapping_api_fails', 10, 3600); 
            error_log('SH Mapping: Simple fallback activated due to PHP timeouts');
        }
    }
    set_transient('sh_mapping_simple_check_done', true, 1800); // Check every 30 minutes
}

/**
 * Simple fallback check - just see if API is consistently failing
 */
function shMappingIsApiFailing() {
    $fail_count = get_transient('sh_mapping_api_fails') ?: 0;
    return $fail_count > 2; // If failed more than 2 times in the last 10 minutes
}

/**
 * Track API failures simply  
 */
function shMappingTrackApiFail() {
    $fail_count = get_transient('sh_mapping_api_fails') ?: 0;
    set_transient('sh_mapping_api_fails', $fail_count + 1, 600); // 10 minutes
    
    // CIRCUIT BREAKER: If we've failed 5+ times, activate emergency mode for 15 minutes
    if ($fail_count >= 5) {
        set_transient('sh_mapping_emergency_mode', true, 900); // 15 minutes emergency block
        error_log('SH Mapping: CIRCUIT BREAKER ACTIVATED - Too many API failures, blocking web API calls for 15 minutes');
    }
}

/**
 * Reset API failure counter (can be called after successful response)
 */
function shMappingResetApiFailures() {
    delete_transient('sh_mapping_api_fails');
}

/**
 * Check if API calls are allowed - Emergency mode blocks web access to APIs
 */
function shMappingAllowApiCalls() {
    // Check if emergency mode is manually activated
    if (get_transient('sh_mapping_emergency_mode')) {
        // EMERGENCY MODE: ONLY allow WP-CLI commands 
        if (defined('WP_CLI') && WP_CLI) {
            return true; // Allow ONLY for cron jobs
        }
        return false; // Block everything else
    }
    
    // Normal mode: Allow all API calls
    return true;
}

/**
 * Simple safe wrapper - just catch errors and return fallback
 */
function shMappingSafeExecute($callback, $fallback_return = [], $function_name = 'unknown') {
    // EMERGENCY BLOCKING: Only allow API calls for WP-CLI (cron)
    if (!shMappingAllowApiCalls()) {
        error_log("SH Mapping: EMERGENCY BLOCK - API call blocked for $function_name");
        return $fallback_return;
    }
    
    // If API is failing, use fallback directly (faster response)
    if (shMappingIsApiFailing()) {
        error_log("SH Mapping: API failing, using fallback for $function_name");
        return $fallback_return;
    }
    
    // Set time limit for this execution
    $start_time = microtime(true);
    
    try {
        $result = call_user_func($callback);
        
        // Track execution time
        $execution_time = microtime(true) - $start_time;
        if ($execution_time > 10) {
            error_log("SH Mapping: Slow execution detected in $function_name: {$execution_time}s");
            shMappingTrackApiFail(); // Count slow responses as failures
        }
        
        // If we got a valid result, reset failure counter
        if (is_array($result) && count($result) > 0) {
            shMappingResetApiFailures();
            return $result;
        }
        
        return $fallback_return;
    } catch (\Exception $e) {
        shMappingTrackApiFail();
        error_log("SH Mapping: Exception in $function_name: " . $e->getMessage());
        return $fallback_return;
    }
}

if (!function_exists('shMappingSportsLeaguesOption')) {
    function shMappingSportsLeaguesOption() {
        return shMappingSafeExecute(function() {
            global $shmapping;
            $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
            $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

            $unserialize_data = [];
            $cached_data = null; // Store cached data to fallback if API fails

            if ($shmapping->helpers->initRedis()) {
                $key = $shmapping->helpers->getRedisKeySportsLeaguesOptions();
                $data_sports = $shmapping->helpers->getRedis($key);

                if (!empty($data_sports)) {
                    $unserialize_data = unserialize($data_sports);

                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        $cached_data = $unserialize_data; // Store for potential fallback
                        return $unserialize_data;
                    }
                }
            }

            try {
                $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameSportsOptions();

                if (file_exists($file_path_sports)) {
                    $sports = file_get_contents($file_path_sports);

                    if (!empty($sports) && is_array(unserialize($sports))) {
                        $cached_data = unserialize($sports); // Store for potential fallback
                        return $cached_data;
                    }
                }
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => 'SH Mapping: '.$exception->getMessage(),
                ], true));
            }

            try {
                $helpers = New Sh_Mapping_Helpers;
                $sports_mapped = $helpers->getMappingData();
                $sports_leagues_options = $helpers->getSports($sports_mapped);

                // Only return fresh data if it's valid
                if (is_array($sports_leagues_options) && count($sports_leagues_options) > 0) {
                    return $sports_leagues_options;
                }
                
                // If API returned empty but we have cached data, use cached data
                if (!empty($cached_data)) {
                    error_log('SH Mapping: API returned empty data for sports leagues options, using cached data');
                    return $cached_data;
                }
                
                return $sports_leagues_options;
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => 'SH Mapping: '.$exception->getMessage(),
                    'sports_mapped' => isset($sports_mapped) ? $sports_mapped : 'undefined',
                ], true));
                
                // If API failed and we have cached data, return cached data
                if (!empty($cached_data)) {
                    error_log('SH Mapping: API request failed for sports leagues options, using cached data');
                    return $cached_data;
                }
            }

            return [];
        }, [], 'shMappingSportsLeaguesOption');
    }
}

if (!function_exists('shMappingSportsOnly')) {
    function shMappingSportsOnly() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $unserialize_data = [];
        $cached_data = null; // Store cached data to fallback if API fails

        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeySportsOnly();
            $data_sports = $shmapping->helpers->getRedis($key);

            if (!empty($data_sports)) {
                $unserialize_data = unserialize($data_sports);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    $cached_data = $unserialize_data; // Store for potential fallback
                    return $unserialize_data;
                }
            }
        }

        try {
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameSportsOnly();

            if (file_exists($file_path_sports)) {
                $sports = file_get_contents($file_path_sports);

                if (!empty($sports) && is_array(unserialize($sports))) {
                    $cached_data = unserialize($sports); // Store for potential fallback
                    return $cached_data;
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_mapped = $helpers->getMappingData();
            $sports_only = $helpers->getSportsOnly($sports_mapped);

            // Only return fresh data if it's valid
            if (is_array($sports_only) && count($sports_only) > 0) {
                return $sports_only;
            }
            
            // If API returned empty but we have cached data, use cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API returned empty data for sports only, using cached data');
                return $cached_data;
            }
            
            return $sports_only;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'sports_mapped' => $sports_mapped,
            ], true));
            
            // If API failed and we have cached data, return cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API request failed for sports only, using cached data');
                return $cached_data;
            }
        }

        return [];
    }
}

if (!function_exists('shMappingSportsLeaguesTitles')) {
    function shMappingSportsLeaguesTitles() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $unserialize_data = [];
        $cached_data = null; // Store cached data to fallback if API fails

        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeySportsLeaguesTitles();
            $data_sports = $shmapping->helpers->getRedis($key);

            if (!empty($data_sports)) {
                $unserialize_data = unserialize($data_sports);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    $cached_data = $unserialize_data; // Store for potential fallback
                    return $unserialize_data;
                }
            }
        }

        try {
            $file_path_titles = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameTitles();

            if (file_exists($file_path_titles)) {
                $titles = file_get_contents($file_path_titles);

                if (!empty($titles) && is_array(unserialize($titles))) {
                    $cached_data = unserialize($titles); // Store for potential fallback
                    return $cached_data;
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_mapped = $helpers->getMappingData();
            $sports_leagues_titles = $helpers->getSportsTitles($sports_mapped);

            // Only return fresh data if it's valid
            if (is_array($sports_leagues_titles) && count($sports_leagues_titles) > 0) {
                return $sports_leagues_titles;
            }
            
            // If API returned empty but we have cached data, use cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API returned empty data for sports leagues titles, using cached data');
                return $cached_data;
            }
            
            return $sports_leagues_titles;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'sports_mapped' => $sports_mapped,
            ], true));
            
            // If API failed and we have cached data, return cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API request failed for sports leagues titles, using cached data');
                return $cached_data;
            }
        }

        return [];
    }
}

if (!function_exists('shMappingSportsConverter')) {
    function shMappingSportsConverter($sports, $source, $type = 'string') {
        $sports_convert = [];
        $sports_mapped = [];

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_mapped = $helpers->getMappingData();

            if (is_array($sports) && count($sports) > 0) {
                foreach ($sports as $sport_id => $leagues) {
                    if (
                        isset($sports_mapped[$sport_id]) &&
                        is_array($sports_mapped[$sport_id]) &&
                        count($sports_mapped[$sport_id]) > 0 &&
                        isset($sports_mapped[$sport_id]['sports'][$source]) &&
                        is_array($sports_mapped[$sport_id]['sports'][$source]) &&
                        count($sports_mapped[$sport_id]['sports'][$source]) > 0
                    ) {
                        $sports_news = [];
                        $leagues_news = [];
                        $sports_news = $sports_mapped[$sport_id]['sports'][$source];

                        if (is_array($leagues) && count($leagues) > 0) {
                            foreach ($leagues as $league_id) {
                                if (
                                    isset($sports_mapped[$sport_id]['leagues'][$league_id][$source]) &&
                                    is_array($sports_mapped[$sport_id]['leagues'][$league_id][$source]) &&
                                    count($sports_mapped[$sport_id]['leagues'][$league_id][$source]) > 0
                                ) {
                                    $leagues_news = array_merge($leagues_news, $sports_mapped[$sport_id]['leagues'][$league_id][$source]);
                                }
                            }
                        }

                        foreach ($sports_news as $new_sport_id) {
                            if (isset($sports_convert[$new_sport_id])) {
                                $sports_convert[$new_sport_id] = array_merge($sports_convert[$new_sport_id], $leagues_news);
                            } else {
                                $sports_convert[$new_sport_id] = $leagues_news;
                            }
                        }
                    }
                }
            }

            if ($type == 'string') {
                $new_array_ids = [];
                foreach ($sports_convert as $sport_id => $league_ids) {
                    if (is_array($league_ids) && count($league_ids) == 0) {
                        array_push($new_array_ids, $sport_id);
                        continue;
                    }

                    foreach ($league_ids as $league_key => $league_id) {
                        array_push($new_array_ids, $league_id);
                    }
                }

                return implode(',', array_unique($new_array_ids));
            } else {
                return $sports_convert;
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'sports' => $sports,
                'type' => $type,
                'source' => $source,
                'sports_convert' => $sports_convert,
                'sports_mapped' => $sports_mapped,
            ], true));
        }

        return [];
    }
}

if (!function_exists('shMappingLeaguesDirect')) {
    function shMappingLeaguesDirect() {
        return shMappingSafeExecute(function() {
            global $shmapping;
            $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
            $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

            $unserialize_data = [];

            if ($shmapping->helpers->initRedis()) {
                $key = $shmapping->helpers->getRedisKeyLeaguesDirect();
                $data_leagues_direct = $shmapping->helpers->getRedis($key);

                if (!empty($data_leagues_direct)) {
                    $unserialize_data = unserialize($data_leagues_direct);

                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                }
            }

            try {
                $file_path_titles = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameLeaguesDirect();

                if (file_exists($file_path_titles)) {
                    $titles = file_get_contents($file_path_titles);

                    if (!empty($titles) && is_array(unserialize($titles))) {
                        return unserialize($titles);
                    }
                }
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => 'SH Mapping: '.$exception->getMessage(),
                ], true));
            }

            try {
                $helpers = New Sh_Mapping_Helpers;
                $sports_mapped = $helpers->getMappingData();
                $sports_teams_direct = $helpers->getSportsLeaguesDirect($sports_mapped);

                return $sports_teams_direct;
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => 'SH Mapping: '.$exception->getMessage(),
                    'sports_mapped' => isset($sports_mapped) ? $sports_mapped : 'undefined',
                ], true));
            }

            return [];
        }, [], 'shMappingLeaguesDirect');
    }
}

if (!function_exists('shMappingTeamsDirect')) {
    function shMappingTeamsDirect() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $unserialize_data = [];

        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeyTeamsDirect();
            $data_teams_direct = $shmapping->helpers->getRedis($key);

            if (!empty($data_teams_direct)) {
                $unserialize_data = unserialize($data_teams_direct);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        try {
            $file_path_titles = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameTeamsDirect();

            if (file_exists($file_path_titles)) {
                $titles = file_get_contents($file_path_titles);

                if (!empty($titles) && is_array(unserialize($titles))) {
                    return unserialize($titles);
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_mapped = $helpers->getMappingData();
            $sports_teams_direct = $helpers->getSportsTeamsDirect($sports_mapped);

            return $sports_teams_direct;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'sports_mapped' => $sports_mapped,
            ], true));
        }

        return [];
    }
}

if (!function_exists('shMappingSeasonsDirect')) {
    function shMappingSeasonsDirect($league_id = null) {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $unserialize_data = [];
        $cached_data = null; // Store cached data to fallback if API fails

        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeySeasonsDirect();
            $data_seasons_direct = $shmapping->helpers->getRedis($key);

            if (!empty($data_seasons_direct)) {
                $unserialize_data = unserialize($data_seasons_direct);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    $seasons = $shmapping->helpers->getSeasonsLeague($unserialize_data, $league_id);

                    if (is_array($seasons) && count($seasons) > 0) {
                        $cached_data = $seasons; // Store filtered data for potential fallback
                        return $seasons;
                    }
                }
            }
        }

        try {
            $file_path_titles = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameSeasonsDirect();

            if (file_exists($file_path_titles)) {
                $seasons_array = file_get_contents($file_path_titles);

                if (!empty($seasons_array) && is_array(unserialize($seasons_array))) {
                    $seasons = $shmapping->helpers->getSeasonsLeague(unserialize($seasons_array), $league_id);

                    if (is_array($seasons) && count($seasons) > 0) {
                        $cached_data = $seasons; // Store filtered data for potential fallback
                        return $seasons;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_mapped = $helpers->getMappingData();
            $sports_seasons_direct = $helpers->getMappingDataSeasons($sports_mapped);
            $seasons = $shmapping->helpers->getSeasonsLeague($sports_seasons_direct, $league_id);

            // Only return fresh data if it's valid
            if (is_array($seasons) && count($seasons) > 0) {
                return $seasons;
            }
            
            // If API returned empty but we have cached data, use cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API returned empty data for seasons direct, using cached data');
                return $cached_data;
            }

            return $seasons;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
            
            // If API failed and we have cached data, return cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API request failed for seasons direct, using cached data');
                return $cached_data;
            }
        }

        return [];
    }
}

if (!function_exists('shMappingSportsDirect')) {
    function shMappingSportsDirect() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $unserialize_data = [];
        $cached_data = null; // Store cached data to fallback if API fails

        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeySportsDirect();
            $data_seasons_direct = $shmapping->helpers->getRedis($key);

            if (!empty($data_seasons_direct)) {
                $unserialize_data = unserialize($data_seasons_direct);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    $cached_data = $unserialize_data; // Store for potential fallback
                    return $unserialize_data;
                }
            }
        }

        try {
            $file_path_titles = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameSportsDirect();

            if (file_exists($file_path_titles)) {
                $titles = file_get_contents($file_path_titles);

                if (!empty($titles) && is_array(unserialize($titles))) {
                    $cached_data = unserialize($titles); // Store for potential fallback
                    return $cached_data;
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_seasons_direct = $helpers->getMappingDataSports();

            // Only return fresh data if it's valid
            if (is_array($sports_seasons_direct) && count($sports_seasons_direct) > 0) {
                return $sports_seasons_direct;
            }
            
            // If API returned empty but we have cached data, use cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API returned empty data for sports direct, using cached data');
                return $cached_data;
            }
            
            return $sports_seasons_direct;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
            
            // If API failed and we have cached data, return cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API request failed for sports direct, using cached data');
                return $cached_data;
            }
        }

        return [];
    }
}

if (!function_exists('shMappingLeaguesSportsDirect')) {
    function shMappingLeaguesSportsDirect($sport_id = null) {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $unserialize_data = [];
        $cached_data = null; // Store cached data to fallback if API fails

        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeyLeaguesSportDirect();
            $data_seasons_direct = $shmapping->helpers->getRedis($key);

            if (!empty($data_seasons_direct)) {
                $unserialize_data = unserialize($data_seasons_direct);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    $leagues = $shmapping->helpers->getLeaguesSport($unserialize_data, $sport_id);

                    if (is_array($leagues) && count($leagues) > 0) {
                        $cached_data = $leagues; // Store filtered data for potential fallback
                        return $leagues;
                    }
                }
            }
        }

        try {
            $file_path_titles = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameLeaguesSportDirect();

            if (file_exists($file_path_titles)) {
                $leagues_sports = file_get_contents($file_path_titles);

                if (!empty($leagues_sports) && is_array(unserialize($leagues_sports))) {
                    $leagues = $shmapping->helpers->getLeaguesSport(unserialize($leagues_sports), $sport_id);

                    if (is_array($leagues) && count($leagues) > 0) {
                        $cached_data = $leagues; // Store filtered data for potential fallback
                        return $leagues;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        try {
            $helpers = New Sh_Mapping_Helpers;
            $leagues_sport_direct = $helpers->getMappingDataLeagues();
            $leagues = $shmapping->helpers->getLeaguesSport($leagues_sport_direct, $sport_id);

            // Only return fresh data if it's valid
            if (is_array($leagues) && count($leagues) > 0) {
                return $leagues;
            }
            
            // If API returned empty but we have cached data, use cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API returned empty data for leagues sports direct, using cached data');
                return $cached_data;
            }

            return $leagues;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
            
            // If API failed and we have cached data, return cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API request failed for leagues sports direct, using cached data');
                return $cached_data;
            }
        }

        return [];
    }
}

if (!function_exists('shMappingTeamsLeagueDirect')) {
    function shMappingTeamsLeagueDirect($league_id = null) {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $unserialize_data = [];
        $cached_data = null; // Store cached data to fallback if API fails

        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeyTeamsDirect();
            $data_teams_direct = $shmapping->helpers->getRedis($key);

            if (!empty($data_teams_direct)) {
                $unserialize_data = unserialize($data_teams_direct);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    $teams = $shmapping->helpers->getTeamsLeague($unserialize_data, $league_id);

                    if (is_array($teams) && count($teams) > 0) {
                        $cached_data = $teams; // Store filtered data for potential fallback
                        return $teams;
                    }
                }
            }
        }

        try {
            $file_path_teams = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameTeamsDirect();

            if (file_exists($file_path_teams)) {
                $teams_data = file_get_contents($file_path_teams);

                if (!empty($teams_data) && is_array(unserialize($teams_data))) {
                    $teams = $shmapping->helpers->getTeamsLeague(unserialize($teams_data), $league_id);

                    if (is_array($teams) && count($teams) > 0) {
                        $cached_data = $teams; // Store filtered data for potential fallback
                        return $teams;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_mapped = $helpers->getMappingData();
            $sports_teams_direct = $helpers->getSportsTeamsDirect($sports_mapped);
            $teams = $shmapping->helpers->getTeamsLeague($sports_teams_direct, $league_id);

            // Only return fresh data if it's valid
            if (is_array($teams) && count($teams) > 0) {
                return $teams;
            }
            
            // If API returned empty but we have cached data, use cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API returned empty data for teams league direct, using cached data');
                return $cached_data;
            }

            return $teams;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'sports_mapped' => $sports_mapped,
            ], true));
            
            // If API failed and we have cached data, return cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API request failed for teams league direct, using cached data');
                return $cached_data;
            }
        }

        return [];
    }
}

if (!function_exists('shMappingSportsLeaguesConverter')) {
    function shMappingSportsLeaguesConverter($sports, $source, $type = 'string') {
        $sports_convert = [];
        $leagues_convert = [];
        $sports_mapped = [];

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_mapped = $helpers->getMappingData();

            if (is_array($sports) && count($sports) > 0) {
                foreach ($sports as $sport_id => $leagues) {
                    if (isset($sports_mapped[$sport_id]) &&
                        is_array($sports_mapped[$sport_id]) &&
                        count($sports_mapped[$sport_id]) > 0 &&
                        isset($sports_mapped[$sport_id]['sports'][$source]) &&
                        is_array($sports_mapped[$sport_id]['sports'][$source]) &&
                        count($sports_mapped[$sport_id]['sports'][$source]) > 0
                    ) {
                        if (is_array($leagues) && count($leagues) > 0) {
                            foreach ($leagues as $league) {
                                if (isset($sports_mapped[$sport_id]['leagues'][$league][$source]) &&
                                    is_array($sports_mapped[$sport_id]['leagues'][$league][$source]) &&
                                    count($sports_mapped[$sport_id]['leagues'][$league][$source]) > 0
                                ) {
                                    $leagues_convert = array_merge($leagues_convert, $sports_mapped[$sport_id]['leagues'][$league][$source]);
                                }
                            }
                        } else {
                            $sports_convert = array_merge($sports_convert, $sports_mapped[$sport_id]['sports'][$source]);
                        }
                    }
                }

                if ($type == 'string') {
                    return [
                        'sports' => implode(',', $sports_convert),
                        'leagues' => implode(',', $leagues_convert),
                    ];
                } else {
                    return [
                        'sports' => $sports_convert, 
                        'leagues' => $leagues_convert
                    ];
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'sports' => $sports,
                'type' => $type,
                'source' => $source,
                'sports_convert' => $sports_convert,
                'leagues_convert' => $leagues_convert,
                'sports_mapped' => $sports_mapped,
            ], true));
        }

        return [];
    }
}

if (!function_exists('shMappingAuthorsMainsSportsLeaguesConverter')) {
    function shMappingAuthorsMainsSportsLeaguesConverter($data, $type = 'members') {
        $helpers = New Sh_Mapping_Helpers;
        $sports_mapped = $helpers->getMappingData();
        $sports_new = [];
        $leagues_new = [];

        if (is_array($data) && count($data) > 0) {
            foreach ($data as $sport_league => $author) {
                $sport_parts = explode('-', $sport_league);

                if (is_array($sport_parts) && count($sport_parts) == 2) {
                    $leagues = (isset($sports_mapped[$sport_parts[0]]['leagues'][$sport_parts[1]][$type]))
                        ? $sports_mapped[$sport_parts[0]]['leagues'][$sport_parts[1]][$type]
                        : null;
                    
                    if (!empty($leagues) && is_array($leagues) && count($leagues) > 0) {
                        foreach ($leagues as $league) {
                            $leagues_new[$league] = $data[$sport_parts[0].'-'.$sport_parts[1]];
                        }
                    }
                } else {
                    $sports = (isset($sports_mapped[$sport_parts[0]]['sports'][$type]))
                        ? $sports_mapped[$sport_parts[0]]['sports'][$type]
                        : null;

                    if (!empty($sports) && is_array($sports) && count($sports) > 0) {
                        foreach ($sports as $sport) {
                            $sports_new[$sport] = $data[$sport_parts[0]];
                        }
                    }
                }
            }
        }

        return [
            'sports' => $sports_new,
            'leagues' => $leagues_new,
        ];
    }
}

if (!function_exists('shMappingSportsLeaguesNamesConverter')) {
    function shMappingSportsLeaguesNamesConverter($type = 'members') {
        $helpers = New Sh_Mapping_Helpers;
        $sports_mapped = $helpers->getMappingData();
        $sports = [];
        $leagues = [];

        foreach ($sports_mapped as $sport_mapping_id => $sport_data) {
            $sport_name = $sport_data['name'];

            if (isset($sport_data['sports'][$type]) && is_array($sport_data['sports'][$type]) && count($sport_data['sports'][$type]) > 0) {
                foreach ($sport_data['sports'][$type] as $id => $sport_id) {
                    $sports[$sport_id] = $sport_name;
                }
            }

            if (isset($sport_data['leagues']) && is_array($sport_data['leagues']) && count($sport_data['leagues']) > 0) {
                foreach ($sport_data['leagues'] as $id => $league) {
                    $league_name = $league['name'];

                    foreach ($league[$type] as $id => $league_id) {
                        $leagues[$league_id] = $league_name;
                    }
                }
            }
        }

        return [
            'sports' => $sports,
            'leagues' => $leagues,
        ];
    }
}

if (!function_exists('shMappingTeamPlayers')) {
    function shMappingTeamPlayers() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();

        try {
            $players_mapped = $shmapping->helpers->getMappingDataPlayers();

            if (is_array($players_mapped) && count($players_mapped) > 0) {
                return $players_mapped;
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'players_mapped' => $players_mapped,
            ], true));
        }

        return [];
    }
}

if (!function_exists('shMappingHierarchySports')) {
    function shMappingHierarchySports() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $unserialize_data = [];
        $cached_data = null; // Store cached data to fallback if API fails
        $key = $shmapping->helpers->getRedisKeyHierarchySports();

        if ($shmapping->helpers->initRedis()) {
            $sports_data = $shmapping->helpers->getRedis($key);
    
            if (!empty($sports_data) && $shmapping->helpers->Isjson($sports_data)) {
                $unserialize_data = json_decode($sports_data, true);
                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    $cached_data = $unserialize_data; // Store for potential fallback
                    return $unserialize_data;
                }
            }
        }

        try {
            $mapped_hierarchy = $shmapping->helpers->getMappingDataHierarchy();
            $result = $shmapping->helpers->getMappingHierarchySports($mapped_hierarchy);
            
            // Only return fresh data if it's valid
            if (is_array($result) && count($result) > 0) {
                return $result;
            }
            
            // If API returned empty but we have cached data, use cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API returned empty data for hierarchy sports, using cached data');
                return $cached_data;
            }

            return $result;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
            
            // If API failed and we have cached data, return cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API request failed for hierarchy sports, using cached data');
                return $cached_data;
            }
        }

        return [];
    }
}

if (!function_exists('shMappingHierarchyFamily')) {
    function shMappingHierarchyFamily($item = null) {
        if (empty($item)) return [];

        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();

        try {
            $mapped_hierarchy = $shmapping->helpers->getMappingDataHierarchy();

            return $shmapping->helpers->getMappingHierarchyFilter($mapped_hierarchy, $item);
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return [];
    }
}

if (!function_exists('shMappingHierarchySportsLeagues')) {
    function shMappingHierarchySportsLeagues() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);
        
        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeyHierarchySportsLeagues();
            $sports_leagues = $shmapping->helpers->getRedis($key);
    
            if (!empty($sports_leagues) && $shmapping->helpers->Isjson($sports_leagues)) {
                $unserialize_data = json_decode($sports_leagues, true);
                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        try {
            $file_path = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameHierarchySportsLeagues();

            if (file_exists($file_path)) {
                $sports_leagues = file_get_contents($file_path);

                if (!empty($sports_leagues) && $shmapping->helpers->Isjson($sports_leagues)) {
                    $unserialize_data = json_decode($sports_leagues, true);
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        try {
            $sports_leagues = [];
            $mapped_hierarchy = $shmapping->helpers->getMappingDataHierarchy();

            // Sort sports and leagues
            uksort($mapped_hierarchy, function($aKey, $bKey) use ($mapped_hierarchy) {
                return strcmp($mapped_hierarchy[$aKey]['group_name'], $mapped_hierarchy[$bKey]['group_name']);
            });

            foreach ($mapped_hierarchy as &$sport) {
                if (isset($sport['items']) && is_array($sport['items'])) {
                    uksort($sport['items'], function($aKey, $bKey) use ($sport) {
                        return strcmp($sport['items'][$aKey]['group_name'], $sport['items'][$bKey]['group_name']);
                    });
                }
            }
            unset($sport);
            // Sort sports and leagues

            foreach ($mapped_hierarchy as $sport_id => $sport) {
                if (isset($sport['items']) && is_array($sport['items']) && count($sport['items']) > 0) {
                    $categories = [];
                    $categories[$sport_id] = $sport['group_name'] .' - '. __('Any League');

                    foreach ($sport['items'] as $league_id => $category) {
                        $categories[$sport_id.'-'.$league_id] = $category['group_name'];
                    }
                } else {
                    $categories = $sport['group_name'];
                }

                if (is_array($categories)) {
                    $sports_leagues[$sport['group_name']] = $categories;
                } else {
                    $sports_leagues[$sport_id] = $categories;
                }
            }

            if (file_exists($file_path)) unlink($file_path);
            file_put_contents($file_path, json_encode($sports_leagues));

            if ($shmapping->helpers->initRedis() && is_array($sports_leagues) && count($sports_leagues) > 0) {
                $shmapping->helpers->setRedis($key, json_encode($sports_leagues), SHMAPPING_REDIS_TIMEOUT);
            }

            return $sports_leagues;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return [];
    }
}

if (!function_exists('shMappingHandicappers')) {
    function shMappingHandicappers() {
        return shMappingSafeExecute(function() {
            global $shmapping;
            $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();

            $handicappers_mapped = [];
            $cached_data = null; // Store cached data to fallback if API fails

            try {
                // Note: getMappingDataHandicappers already handles Redis and file cache internally
                // We need to check if it returns valid data
                $handicappers_mapped = $shmapping->helpers->getMappingDataHandicappers();

                if (is_array($handicappers_mapped) && count($handicappers_mapped) > 0) {
                    return $handicappers_mapped;
                }
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => 'SH Mapping: '.$exception->getMessage(),
                    'handicappers_mapped' => $handicappers_mapped,
                ], true));
            }

            return [];
        }, [], 'shMappingHandicappers');
    }
}

if (!function_exists('shMappingMemberships')) {
    function shMappingMemberships($enabled = true) {
        return shMappingSafeExecute(function() use ($enabled) {
            global $shmapping;
            $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
            $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

            $cached_data = null; // Store cached data to fallback if API fails

            try {
                $memberships = [];
                $helpers = New Sh_Mapping_Helpers;
                // Note: getMappingMembershipsData already handles Redis and file cache internally
                $memberships_mapped = $helpers->getMappingMembershipsData();

                if (is_array($memberships_mapped) && count($memberships_mapped) > 0) {
                    foreach ($memberships_mapped as $key => $membership_data) {
                        if (($enabled && $membership_data['enable']) || !$enabled) {
                            $memberships[$membership_data['id']] = $membership_data['name'];
                        }
                    }
                    // Store processed data as cache
                    if (count($memberships) > 0) {
                        return $memberships;
                    }
                }

                return $memberships;
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => 'SH Mapping: '.$exception->getMessage(),
                ], true));
            }

            return [];
        }, [], 'shMappingMemberships');
    }
}

if (!function_exists('shMappingMembershipsHandicappers')) {
    function shMappingMembershipsHandicappers($attributes = []) {
        // Prevent infinite recursion - detect if we're already processing this function
        static $processing = false;
        if ($processing) {
            error_log('SH Mapping: Recursive call detected in shMappingMembershipsHandicappers, returning empty array');
            return [];
        }
        $processing = true;
        
        try {
            // Get mapping configuration from ACF
            $configs = get_field('sh_mapping_packages', 'option');
            
            // Initialize helper and get memberships data once
            $helpers = new Sh_Mapping_Helpers;
            $memberships_mapped = $helpers->getMappingMembershipsData();
            
            // Validate memberships data
            if (!is_array($memberships_mapped) || count($memberships_mapped) == 0) {
                error_log('SH Mapping: No memberships data available');
                return [];
            }
            
            // Create default 1:1 mapping only if no config exists
            if (empty($configs) || !is_array($configs)) {
                $configs = [];
                foreach ($memberships_mapped as $membership_id => $membership_data) {
                    if (isset($membership_data['id'])) {
                        $configs[] = [
                            'key' => $membership_data['id'],
                            'value' => $membership_data['id']
                        ];
                    }
                }
            }

            // Set default attributes with validation
            $separator = isset($attributes['separator']) ? $attributes['separator'] : ' + ';
            $type = isset($attributes['type']) ? $attributes['type'] : 'label';
            $target = isset($attributes['target']) ? $attributes['target'] : '_self';
            $field_name = (isset($attributes['short']) && !empty($attributes['short'])) ? 'short_name' : 'name';

            // Build memberships lookup array
            $memberships = [];
            foreach ($memberships_mapped as $key => $membership_data) {
                if (!isset($membership_data['id'])) continue;
                
                $memberships[$membership_data['id']] = [
                    'name' => (isset($membership_data[$field_name]) && !empty($membership_data[$field_name]))
                        ? $membership_data[$field_name]
                        : ($membership_data['name'] ?? 'Unknown'),
                    'url' => $membership_data['url'] ?? '',
                ];
            }

            // Process mapping configurations
            $packages = [];
            foreach ($configs as $config) {
                if (!isset($config['key']) || !isset($config['value'])) continue;
                
                $key_new = $config['key'];
                $membership_id = $config['value'];
                
                // Check if membership exists
                if (!isset($memberships[$membership_id])) {
                    error_log("SH Mapping Warning: Membership ID {$membership_id} not found in data");
                    continue;
                }
                
                $data = $memberships[$membership_id];

                // Generate package display (link or label)
                if ($type === 'link' && !empty($data['url'])) {
                    $package = sprintf(
                        '<a class="sh-handicappers-package" href="%s" target="%s">%s</a>', 
                        esc_url($data['url']), 
                        esc_attr($target), 
                        esc_html($data['name'])
                    );
                } else {
                    $package = esc_html($data['name']);
                }
                
                // Combine packages for same key
                if (isset($packages[$key_new])) {
                    $packages[$key_new] .= $separator . $package;
                } else {
                    $packages[$key_new] = $package;
                }
            }

            $processing = false; // Reset flag before returning
            return $packages;
            
        } catch (\Exception $exception) {
            error_log('SH Mapping Error: ' . $exception->getMessage());
            $processing = false; // Reset flag on exception
            return [];
        }
    }
}

if (!function_exists('shMappingMembershipsData')) {
    function shMappingMembershipsData() {
        return shMappingSafeExecute(function() {
            global $shmapping;
            $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
            $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

            try {
                $memberships = [];
                $helpers = New Sh_Mapping_Helpers;
                $memberships_mapped = $helpers->getMappingMembershipsData();

                $memberships_labels = shMappingMembershipsHandicappers(['type' => 'label', 'separator' => ' + ', 'target' => '_self', 'short' => true]);

                if (is_array($memberships_mapped) && count($memberships_mapped) > 0) {
                    foreach ($memberships_mapped as $key => $membership_data) {
                        if ($membership_data['enable'] && isset($membership_data['url']) && !empty($membership_data['url'])) {
                            $memberships[$membership_data['id']] = $membership_data;
                            $memberships[$membership_data['id']]['packages_label'] = (isset($memberships_labels[$membership_data['id']]))
                                ? $memberships_labels[$membership_data['id']]
                                : $membership_data['name'];
                        }
                    }
                }

                return $memberships;
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => 'SH Mapping: '.$exception->getMessage(),
                ], true));
            }

            return [];
        }, [], 'shMappingMembershipsData');
    }
}

if (!function_exists('shMappingGetMembershipsLogic')) {
    function shMappingGetMembershipsLogic() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeyMembershipsLogic();
            $data_members = $shmapping->helpers->getRedis($key);

            if (!empty($data_members)) {
                $unserialize_data = unserialize($data_members);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        try {
            $file_path_members = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileMembersLogic();

            if (file_exists($file_path_members)) {
                $members = file_get_contents($file_path_members);

                if (!empty($members) && is_array(unserialize($members))) {
                    return unserialize($members);
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }
            
        try {
            $configs = get_field('sh_mapping_packages', 'option');
            if (!is_array($configs) || count($configs) == 0) return [];

            $members = $shmapping->helpers->getMembersLogic($configs);

            return $members;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return [];
    }
}

if (!function_exists('shMappingLeaderboardSports')) {
    function shMappingLeaderboardSports() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $leaderboard_sports = [];
        $cached_data = null; // Store cached data to fallback if API fails

        // Check Redis cache first
        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeyLeaderboardSports();
            $data_sports = $shmapping->helpers->getRedis($key);

            if (!empty($data_sports) && $shmapping->helpers->isJson($data_sports)) {
                $unserialize_data = json_decode($data_sports, true);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    $cached_data = $unserialize_data; // Store for potential fallback
                    return $unserialize_data;
                }
            }
        }

        // Check file cache second
        try {
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameLeaderboardSports();
            if (file_exists($file_path_sports)) {
                $data = file_get_contents($file_path_sports);

                if (!empty($data) && $shmapping->helpers->isJson($data)) {
                    $data_unserialize = json_decode($data, true);
                    if (is_array($data_unserialize)) {
                        $cached_data = $data_unserialize; // Store for potential fallback
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        // Try to fetch fresh data from API
        try {
            $helpers = New Sh_Mapping_Helpers;
            $leaderboard_mapped = $helpers->getMappingLeaderboardData();
            $leaderboard_sports = $helpers->getLeaderboardSports($leaderboard_mapped);
            
            // Only return fresh data if it's valid
            if (is_array($leaderboard_sports) && count($leaderboard_sports) > 0) {
                return $leaderboard_sports;
            }
            
            // If API returned empty but we have cached data, use cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API returned empty data for leaderboard sports, using cached data');
                return $cached_data;
            }
            
            return $leaderboard_sports;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'leaderboard_mapped' => $leaderboard_mapped,
            ], true));
            
            // If API failed and we have cached data, return cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API request failed for leaderboard sports, using cached data');
                return $cached_data;
            }
        }

        return [];
    }
}

if (!function_exists('shMappingLeaderboardCategories')) {
    function shMappingLeaderboardCategories() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $leaderboard_categories = [];
        $cached_data = null; // Store cached data to fallback if API fails

        // Check Redis cache first
        if ($shmapping->helpers->initRedis()) {
            $key = $shmapping->helpers->getRedisKeyLeaderboardCategories();
            $data_sports = $shmapping->helpers->getRedis($key);

            if (!empty($data_sports) && $shmapping->helpers->isJson($data_sports)) {
                $unserialize_data = json_decode($data_sports, true);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    $cached_data = $unserialize_data; // Store for potential fallback
                    return $unserialize_data;
                }
            }
        }

        // Check file cache second
        try {
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR.'/'.$shmapping->helpers->getFileNameLeaderboardCategories();
            if (file_exists($file_path_sports)) {
                $data = file_get_contents($file_path_sports);

                if (!empty($data) && $shmapping->helpers->isJson($data)) {
                    $data_unserialize = json_decode($data, true);
                    if (is_array($data_unserialize) && count($data_unserialize) > 0) {
                        $cached_data = $data_unserialize; // Store for potential fallback
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        // Try to fetch fresh data from API
        try {
            $helpers = New Sh_Mapping_Helpers;
            $leaderboard_mapped = $helpers->getMappingLeaderboardData();
            $leaderboard_categories = $helpers->getLeaderboardCategories($leaderboard_mapped);

            // Only return fresh data if it's valid
            if (is_array($leaderboard_categories) && count($leaderboard_categories) > 0) {
                return $leaderboard_categories;
            }
            
            // If API returned empty but we have cached data, use cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API returned empty data for leaderboard categories, using cached data');
                return $cached_data;
            }
            
            return $leaderboard_categories;
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'leaderboard_mapped' => $leaderboard_mapped,
            ], true));
            
            // If API failed and we have cached data, return cached data
            if (!empty($cached_data)) {
                error_log('SH Mapping: API request failed for leaderboard categories, using cached data');
                return $cached_data;
            }
        }

        return [];
    }
}

if (!function_exists('shMappingSportsLeaguesConverterSingle')) {
    function shMappingSportsLeaguesConverterSingle($hierarchy_id, $source) {
        $sport_id = null;
        $league_id = null;
        $sports_mapped = [];

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_mapped = $helpers->getMappingData();
            if (!is_array($sports_mapped) || count($sports_mapped) == 0) return [];

            if (!strpos($hierarchy_id, '-')) {
                $sport_id = $hierarchy_id;
            } else {
                list($sport_id, $league_id) = explode('-', $hierarchy_id);
            }

            if (!isset($sports_mapped[$sport_id]['sports'])) return [];

            if (empty($league_id)) {
                return [
                    'sport' => $sports_mapped[$sport_id]['name'],
                    'sport_id' => implode(',', $sports_mapped[$sport_id]['sports'][$source]),
                ];
            } else {
                return [
                    'sport' => $sports_mapped[$sport_id]['name'],
                    'sport_id' => implode(',', $sports_mapped[$sport_id]['sports'][$source]),
                    'league' => $sports_mapped[$sport_id]['leagues'][$league_id]['name'],
                    'league_id' => implode(',', $sports_mapped[$sport_id]['leagues'][$league_id][$source]),	
                ];
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'hierarchy_id' => $hierarchy_id,
                'source' => $source,
            ], true));
        }

        return [];
    }
}

if (!function_exists('shMappingHierarchyConverterSingle')) {
    function shMappingHierarchyConverterSingle($hierarchy_id = null) {
        $sport_id = null;
        $league_id = null;
        $sports_mapped = [];

        if (empty($hierarchy_id)) return [];

        try {
            $helpers = New Sh_Mapping_Helpers;
            $sports_mapped = $helpers->getMappingDataHierarchy();

            if (!is_array($sports_mapped) || count($sports_mapped) == 0) return [];

            if (!strpos($hierarchy_id, '-')) {
                $sport_id = $hierarchy_id;
            } else {
                list($sport_id, $league_id) = explode('-', $hierarchy_id);
            }

            if (!isset($sports_mapped[$sport_id])) return [];

            if (empty($league_id)) {
                return [
                    'sport' => $sports_mapped[$sport_id]['group_name'],
                    'sport_id' => preg_replace('/[^0-9]/', '', $sport_id),
                ];
            } else {
                return [
                    'sport' => $sports_mapped[$sport_id]['group_name'],
                    'sport_id' => preg_replace('/[^0-9]/', '', $sport_id),
                    'league' => $sports_mapped[$sport_id]['items'][$league_id]['group_name'],
                    'league_id' => preg_replace('/[^0-9]/', '', $league_id),	
                ];
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
                'hierarchy_id' => $hierarchy_id,
            ], true));
        }

        return [];
    }
}

if (!function_exists('getSporstHierarchyOptions')) {
    function getSporstHierarchyOptions() {
        global $shmapping;
        $shmapping = (!empty($shmapping)) ? $shmapping : Sh_Mapping::instance();
        $shmapping->helpers->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $hierarchy_options = [];
        $key = 'sh_mapping_sports_hierarchy_options';

        // Try to get from Redis cache first
        if ($shmapping->helpers->initRedis()) {
            $cached_data = $shmapping->helpers->getRedis($key);
    
            if (!empty($cached_data)) {
                $unserialize_data = unserialize($cached_data);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        try {
            // Get hierarchy data
            $hierarchy_data = $shmapping->helpers->getSportsHierarchyData();
            
            if (is_array($hierarchy_data) && count($hierarchy_data) > 0) {
                $hierarchy_options = processSportsHierarchyOptions($hierarchy_data);
            } else {
                error_log('SH Mapping: No hierarchy data received from helper');
            }
        } catch (Exception $e) {
            error_log('SH Mapping: Exception in getSporstHierarchyOptions: ' . $e->getMessage());
        }

        // Cache the results if we have data
        if ($shmapping->helpers->initRedis() && is_array($hierarchy_options) && count($hierarchy_options) > 0) {
            $shmapping->helpers->setRedis($key, serialize($hierarchy_options), SHMAPPING_REDIS_TIMEOUT);
        }

        // Return cached/fallback data if API fails
        return !empty($hierarchy_options) ? $hierarchy_options : [];
    }
}

if (!function_exists('processSportsHierarchyOptions')) {
    /**
     * Process the sports hierarchy data from API response
     * 
     * @param array $hierarchy_data The raw hierarchy data from API
     * @return array Processed options for select dropdown
     */
    function processSportsHierarchyOptions($hierarchy_data) {
        $options = [];
        
        foreach ($hierarchy_data as $group) {
            // Add the main group
            $options[$group['id']] = strtoupper($group['group_name']);
            
            // Process nested groups if they exist
            if (isset($group['groups']) && is_array($group['groups'])) {
                foreach ($group['groups'] as $subgroup) {                   
                    $options[$subgroup['id']] = strtoupper($subgroup['group_name']);
                    
                    // Process third level groups if they exist
                    if (isset($subgroup['groups']) && is_array($subgroup['groups'])) {
                        foreach ($subgroup['groups'] as $thirdlevel) {                           
                            $options[$thirdlevel['id']] = strtoupper($thirdlevel['group_name']);
                        }
                    }
                }
            }
        }
        
        return $options;
    }
}
