<?php

// Exit if accessed directly.
if (!defined('ABSPATH')) exit;

/**
 * Class Sh_Mapping_Helpers
 *
 * This class contains repetitive functions that
 * are used globally within the plugin.
 *
 * @package        SHMAPPING
 * @subpackage    Classes/Sh_Mapping_Helpers
 * @author        IT
 * @since        1.0.0
 */

class Sh_Mapping_Helpers {
    private $redis_cache;
    private $redis_cache_group;

    function __construct() {
        $this->redis_cache = $this->initRedis();
        $this->setRedisGroup();
    }

    public function initRedis() {
        global $wp_object_cache;

        if (isset($wp_object_cache->redis) && $wp_object_cache->redis instanceof \Redis) {
            return true;
        }

        return false;
    }

    public function setRedisGroup($group = '') {
        $this->redis_cache_group = $group;
    }

    public function getRedis($key) {
        return wp_cache_get($key, $this->redis_cache_group);
    }

    public function setRedis($key, $data, $expire = 0) {
        wp_cache_set($key, $data, $this->redis_cache_group, $expire);

        return true;
    }

    public function getRedisKeyMapping() {
        return REDIS_KEY_SHMAPPING;
    }

    public function getRedisKeySportsLeaguesOptions() {
        return 'sh_mapping_sports_leagues_options';
    }

    public function getRedisKeySportsLeaguesTitles() {
        return 'sh_mapping_sports_leagues_titles';
    }

    public function getRedisKeySportsLeaguesAllData() {
        return 'sh_mapping_sports_leagues_all_data';
    }

    public function getMappingData()
    {
        $sports_mapped = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);

        if ($this->initRedis()) {
            $key = $this->getRedisKeyMapping();
            $mapping = $this->getRedis($key);
    
            if (!empty($mapping)) {
                $unserialize_data = unserialize($mapping);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameData();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && is_serialized($data)) {
                    $data_unserialize = unserialize($data);
                    if (is_array($data_unserialize)) {
                        return unserialize($data);
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        $mapping_url = $this->getRequestMapping();
        $execute_request_mapping = $this->exeRequest($mapping_url, $file_path_mapped);
        
        if ($execute_request_mapping) {
            $parse_results_mapping = $this->parseFeedFile($file_path_mapped);
            
            if (!empty($parse_results_mapping) && isset($parse_results_mapping['data-content']) && is_array($parse_results_mapping['data-content']) && isset($parse_results_mapping['data-content']['sports']) && isset($parse_results_mapping['data-content']['leagues']) && isset($parse_results_mapping['data-content']['teams'])) {
                $sports = $parse_results_mapping['data-content']['sports'];
                $leagues = $parse_results_mapping['data-content']['leagues'];
                $teams = $parse_results_mapping['data-content']['teams'];

                if (is_array($sports) && count($sports) > 0) {
                    foreach ($sports as $sport) {
                        $sports_mapped[$sport['id']] = [
                            'id' => $sport['id'],
                            'name' => $sport['name'],
                            'logo' => $sport['logo'],
                        ];

                        $sports_mapped[$sport['id']]['sports'] = [
                            'masterfeed' => $sport['field_level_media'],
                            'members' => $sport['handicappers'],
                        ];
    
                        if (is_array($leagues) && count($leagues) > 0) {
                            foreach ($leagues as $league) {
                                if (!isset($league['mapped_sport_id']) || (isset($league['mapped_sport_id']) && $league['mapped_sport_id'] != $sport['id'])) continue;
            
                                $sports_mapped[$sport['id']]['leagues'][$league['id']] = [
                                    'id' => $league['id'],
                                    'name' => $league['name'],
                                    'logo' => $league['logo'],
                                    'masterfeed' => $league['field_level_media'],
                                    'members' => $league['handicappers'],
                                ];

                                if (is_array($teams) && count($teams) > 0) {
                                    foreach ($teams as $team) {
                                        if (!isset($team['mapped_league_id']) || (isset($team['mapped_league_id']) && $team['mapped_league_id'] != $league['id'])) continue;
                    
                                        $sports_mapped[$sport['id']]['leagues'][$league['id']]['teams'][$team['id']] = [
                                            'id' => $team['id'],
                                            'name' => $team['name'],
                                            'logo' => $team['logo'],
                                            'masterfeed' => $team['field_level_media'],
                                            'members' => $team['handicappers'],
                                        ];
                                    }
                                }
                            }
                        }
                    }
                }

                if ($this->initRedis() && is_array($sports_mapped) && count($sports_mapped) > 0) {
                    $this->setRedis($key, serialize($sports_mapped), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        return $sports_mapped;
    }

    public function delMappingData()
    {
        if ($this->initRedis()) {
            try {
                $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
                $key_sports = $this->getRedisKeyMapping();
                $key_sports_options = $this->getRedisKeySportsLeaguesOptions();
                $key_sports_titles = $this->getRedisKeySportsLeaguesTitles();
                $this->setRedis($key_sports, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_sports_options, serialize([]), SHMAPPING_REDIS_TIMEOUT);
                $this->setRedis($key_sports_titles, serialize([]), SHMAPPING_REDIS_TIMEOUT);

                error_log('SH Mapping: Clear cache.');
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => $exception->getMessage(),
                    'keys' => [$key_sports, $key_sports_options, $key_sports_titles],
                ], true));
            }
        }

        try {
            unlink(SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameData());
            unlink(SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameSportsOptions());
            unlink(SHMAPPING_PLUGIN_LOG_DIR.'/'.$this->getFileNameTitles());
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
            ], true));
        }
    }

    private function getRequestMapping()
    {
        $api_request = '';
        $api_base = (!empty(get_field('sh_mapping_url', 'options')))
            ? get_field('sh_mapping_url', 'options') 
            : 'http://masterfeed.shnadmin.com/feed/mapped/get_data';
        $api_token = get_field('sh_mapping_token', 'options');

        
        $api_request = $api_base.'?key='.$api_token;

        return $api_request;
    }

    private function exeRequest($request, $file_path)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $request);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);

        if ($response === false) {
            curl_close($ch);
            error_log(sprintf(__('SH Mapping: ERROR: %s.', 'sh-mapping'), curl_error($ch)));
            WP_CLI::line(sprintf(__('ERROR: %s.', 'sh-mapping'), curl_error($ch)));
        } else {
            file_put_contents($file_path, $response);
            curl_close($ch);

            error_log(sprintf(__('JSON response saved to: %s', 'sh-mapping'), $file_path));

            return true;
        }

        return false;
    }

    private function parseFeedFile($file_path)
    {
        $file_contents = file_get_contents($file_path);

        if ($file_contents === false) {
            WP_CLI::line(__('SH Mapping Error: Failed to read the file.', 'sh-mapping'));

            return null;
        }

        $json_data = json_decode($file_contents, true);

        if ($json_data === null) {
            WP_CLI::line(__('SH Mapping Error: Failed to decode the JSON.', 'sh-mapping'));

            return null;
        }

        return $json_data;
    }

    public function getSports($data)
    {
        $sports_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeySportsLeaguesOptions();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);
    
            if (!empty($data_sports)) {
                $unserialize_data = unserialize($data_sports);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        foreach ($data as $sport_id => $sport) {
            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                $leagues = [];
                $leagues[$sport['id']] = $sport['name'] .' - '. __('Any Category');

                foreach ($sport['leagues'] as $league) {
                    $leagues[$sport['id'].'-'.$league['id']] = $league['name'];
                }
            } else {
                $leagues = $sport['name'];
            }

            if (is_array($leagues)) {
                $sports_data[$sport['name']] = $leagues;
            } else {
                $sports_data[$sport['id']] = $leagues;
            }
        }

        if ($this->initRedis() && is_array($sports_data) && count($sports_data) > 0) {
            $this->setRedis($key, serialize($sports_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_sports =  $this->getFileNameSportsOptions();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;
    
            file_put_contents($file_path_sports, serialize($sports_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $sports_data;
    }

    public function getSportsTitles($data)
    {
        $sport_titles = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeySportsLeaguesTitles();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);
    
            if (!empty($data_sports)) {
                $unserialize_data = unserialize($data_sports);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        foreach ($data as $sport_id => $sport) {
            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                $sport_titles[$sport['id']] = $sport['name'];

                foreach ($sport['leagues'] as $league) {
                    $sport_titles[$sport['id'].'-'.$league['id']] = $sport['name'].' | '.$league['name'];
                }
            } else {
                $sport_titles[$sport['id']] = $sport['name'];
            }
        }

        if ($this->initRedis() && is_array($sport_titles) && count($sport_titles) > 0) {
            $this->setRedis($key, serialize($sport_titles), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }
    
            $filename_titles =  $this->getFileNameTitles();
            $file_path_titles = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_titles;
    
            file_put_contents($file_path_titles, serialize($sport_titles));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: '.$exception->getMessage(),
            ], true));
        }

        return $sport_titles;
    }

    public function getFileNameData()
    {
        return 'SH_MAPPING.json';
    }

    public function getFileNameSportsOptions()
    {
        return 'SH_MAPPING_SPORTS_OPTIONS.json';
    }

    public function getFileNameTitles()
    {
        return 'SH_MAPPING_TITLES.json';
    }
}
