<?php

// Exit if accessed directly.

use function WP_CLI\Utils\is_json;

if (!defined('ABSPATH')) exit;

/**
 * Class Sh_Mapping_Helpers
 *
 * This class contains repetitive functions that
 * are used globally within the plugin.
 *
 * @package        SHMAPPING
 * @subpackage    Classes/Sh_Mapping_Helpers
 * @author        IT
 * @since        1.0.0
 */

class Sh_Mapping_Helpers
{
    private $redis_cache;
    private $redis_cache_group;

    function __construct()
    {
        $this->redis_cache = $this->initRedis();
        $this->setRedisGroup();
    }

    public function initRedis()
    {
        global $wp_object_cache;

        if (isset($wp_object_cache->redis) && $wp_object_cache->redis instanceof \Redis) {
            return true;
        }

        return false;
    }

    public function setRedisGroup($group = '')
    {
        $this->redis_cache_group = $group;
    }

    public function getRedis($key)
    {
        return wp_cache_get($key, $this->redis_cache_group);
    }

    public function setRedis($key, $data, $expire = 0)
    {
        wp_cache_set($key, $data, $this->redis_cache_group, $expire);

        return true;
    }

    public function deleteRedis($key)
    {
        return wp_cache_delete($key, $this->redis_cache_group);
    }

    public function getRedisKeyMapping()
    {
        return REDIS_KEY_SHMAPPING;
    }

    public function getRedisKeySportsLeaguesOptions()
    {
        return 'sh_mapping_sports_leagues_options';
    }

    public function getRedisKeySportsOnly()
    {
        return 'sh_mapping_sports_only';
    }

    public function getRedisKeySportsLeaguesTitles()
    {
        return 'sh_mapping_sports_leagues_titles';
    }

    public function getRedisKeySportsLeaguesAllData()
    {
        return 'sh_mapping_sports_leagues_all_data';
    }

    public function getRedisKeyLeaguesDirect()
    {
        return 'sh_mapping_leagues_direct';
    }

    public function getRedisKeyTeamsDirect()
    {
        return 'sh_mapping_teams_direct';
    }

    public function getRedisKeySeasonsDirect()
    {
        return 'sh_mapping_sasons_direct';
    }

    public function getRedisKeySportsDirect()
    {
        return 'sh_mapping_sports_direct';
    }

    public function getRedisKeyLeaguesSportDirect()
    {
        return 'sh_mapping_leagues_sport_direct';
    }

    public function getRedisKeyHierarchySports()
    {
        return 'sh_mapping_hierarchy_sports';
    }

    public function getRedisKeyHandicappers()
    {
        return 'sh_mapping_handicappers';
    }

    public function getRedisKeyMemberships()
    {
        return 'sh_mapping_memberships';
    }

    public function getRedisKeyMembershipsLogic()
    {
        return 'sh_mapping_memberships_logic';
    }

    public function getRedisKeyLeaderboardData()
    {
        return 'sh_mapping_leaderboard_data';
    }

    public function getRedisKeyLeaderboardSports()
    {
        return 'sh_mapping_leaderboard_sports';
    }

    public function getRedisKeyLeaderboardCategories()
    {
        return 'sh_mapping_leaderboard_categories';
    }

    public function getRedisKeyHierarchySportsLeagues()
    {
        return 'sh_mapping_hierarchy_sports_leagues';
    }

    public function getMappingData()
    {
        // CRITICAL: Prevent API calls from web requests to avoid timeouts
        // Only WP-CLI (cron jobs) should make fresh API calls
        $is_web_request = !defined('WP_CLI') || !WP_CLI;
        $emergency_mode = get_transient('sh_mapping_emergency_mode');

        $sports_mapped = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);

        if ($this->initRedis()) {
            $key = $this->getRedisKeyMapping();
            $mapping = $this->getRedis($key);

            if (!empty($mapping)) {
                // TRY JSON FIRST (new format)
                if ($this->isJson($mapping)) {
                    $unserialize_data = json_decode($mapping, true);
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                    // FALLBACK TO SERIALIZED (old format)
                    $unserialize_data = @unserialize($mapping);
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameData();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        // Try file cache first
        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && $this->isJson($data)) {
                    $data_unserialize = json_decode($data, true);
                    if (is_array($data_unserialize) && count($data_unserialize) > 0) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        // PREVENT API CALLS FROM WEB REQUESTS
        // Only CLI/cron should make API calls to prevent timeouts
        if ($is_web_request || $emergency_mode) {
            error_log('SH Mapping: Preventing API call to mapping data from web request (use WP-CLI: wp sh_mapping pull)');
            return $sports_mapped;
        }

        // Only reach here if called from WP-CLI
        @set_time_limit(60);

        $mapping_url = $this->getRequestMapping();
        if ($mapping_url === null) {
            error_log(__('SH Mapping: Cannot fetch mapping data due to missing API token.', 'sh-mapping'));
            return $sports_mapped;
        }

        $execute_request_mapping = $this->exeRequest($mapping_url, $file_path_mapped);

        if ($execute_request_mapping) {
            $parse_results_mapping = $this->parseFeedFile($file_path_mapped);

            if (!empty($parse_results_mapping) && isset($parse_results_mapping['data-content']) && is_array($parse_results_mapping['data-content']) && isset($parse_results_mapping['data-content']['sports']) && isset($parse_results_mapping['data-content']['leagues']) && isset($parse_results_mapping['data-content']['teams'])) {
                $sports = $parse_results_mapping['data-content']['sports'];
                $leagues = $parse_results_mapping['data-content']['leagues'];
                $teams = $parse_results_mapping['data-content']['teams'];

                if (is_array($sports) && count($sports) > 0) {
                    foreach ($sports as $key => $sport) {
                        // Validate required fields exist
                        if (!isset($sport['id']) || !isset($sport['name'])) {
                            error_log(sprintf(__('SH Mapping: Skipping invalid sport data - missing id or name. Key: %s', 'sh-mapping'), $key));
                            continue;
                        }

                        $sports_mapped[$sport['id']] = [
                            'id' => $sport['id'],
                            'sports_direct' => isset($sport['sports_direct']) ? $sport['sports_direct'] : '',
                            'name' => $sport['name'],
                            'logo' => isset($sport['logo']) ? $sport['logo'] : '',
                        ];

                        $sports_mapped[$sport['id']]['sports'] = [
                            'masterfeed' => isset($sport['field_level_media']) ? $sport['field_level_media'] : null,
                            'members' => isset($sport['handicappers']) ? $sport['handicappers'] : null,
                            'goal_serve' => isset($sport['goal_serve']) ? $sport['goal_serve'] : null,
                        ];

                        if (is_array($leagues) && count($leagues) > 0) {
                            foreach ($leagues as $league_key => $league) {
                                if (!isset($league['mapped_sport_id']) || (isset($league['mapped_sport_id']) && $league['mapped_sport_id'] != $sport['id'])) continue;

                                // Validate required fields exist
                                if (!isset($league['id']) || !isset($league['name'])) {
                                    error_log(sprintf(__('SH Mapping: Skipping invalid league data - missing id or name. Key: %s', 'sh-mapping'), $league_key));
                                    continue;
                                }

                                $sports_mapped[$sport['id']]['leagues'][$league['id']] = [
                                    'id' => $league['id'],
                                    'sports_direct' => isset($league['sports_direct']) ? $league['sports_direct'] : '',
                                    'name' => $league['name'],
                                    'logo' => isset($league['logo']) ? $league['logo'] : '',
                                    'masterfeed' => isset($league['field_level_media']) ? $league['field_level_media'] : null,
                                    // Using new ppp IDs instead of handicappers, but keeping the 'members' key
                                    // to minimize impact on RDG Blocks
                                    'members' => isset($league['ppp']) ? $league['ppp'] : null,
                                    'goal_serve' => isset($league['goal_serve']) ? $league['goal_serve'] : null,
                                ];

                                if (is_array($teams) && count($teams) > 0) {
                                    foreach ($teams as $team_key => $team) {
                                        if (!isset($team['mapped_league_id']) || (isset($team['mapped_league_id']) && $team['mapped_league_id'] != $league['id'])) continue;

                                        // Validate required fields exist
                                        if (!isset($team['id']) || !isset($team['name'])) {
                                            error_log(sprintf(__('SH Mapping: Skipping invalid team data - missing id or name. Key: %s', 'sh-mapping'), $team_key));
                                            continue;
                                        }

                                        $sports_mapped[$sport['id']]['leagues'][$league['id']]['teams'][$team['id']] = [
                                            'id' => $team['id'],
                                            'sports_direct' => isset($team['sports_direct']) ? $team['sports_direct'] : '',
                                            'name' => $team['name'],
                                            'logo' => isset($team['logo']) ? $team['logo'] : '',
                                            'masterfeed' => isset($team['field_level_media']) ? $team['field_level_media'] : null,
                                            'members' => isset($team['handicappers']) ? $team['handicappers'] : null,
                                        ];
                                    }
                                }
                            }
                        }
                    }
                }

                if ($this->initRedis() && is_array($sports_mapped) && count($sports_mapped) > 0) {
                    $this->setRedis($key, serialize($sports_mapped), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        return $sports_mapped;
    }

    public function getMappingLeaderboardData()
    {
        $sports_mapped = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyLeaderboardData();

        if ($this->initRedis()) {
            $leaderboard_data = $this->getRedis($key);

            if (!empty($leaderboard_data) && $this->isJson($leaderboard_data)) {
                $unserialize_data = json_decode($leaderboard_data, true);

                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameLeaderboard();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && $this->isJson($data)) {
                    $data_unserialize = json_decode($data, true);
                    if (is_array($data_unserialize)) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        $sports_url = $this->getRequestLeaderboardSports();
        $categories_url = $this->getRequestLeaderboardCategories();

        if ($sports_url === null || $categories_url === null) {
            error_log(__('SH Mapping: Cannot fetch leaderboard data due to missing API token.', 'sh-mapping'));
            return $sports_mapped;
        }

        $execute_request = $this->exeRequest($sports_url, $file_path_mapped);

        if ($execute_request) {
            $leaderboard_sports = $this->parseFeedFile($file_path_mapped);

            if (is_array($leaderboard_sports) && count($leaderboard_sports) > 0) {
                foreach ($leaderboard_sports as $sport) {
                    $sports_mapped[$sport['id']] = [
                        'id' => $sport['id'],
                        'name' => $sport['sport_name'],
                        'categories' => []
                    ];

                    $category_url = $categories_url . '&sport=' . $sport['id'];
                    $category_data = $this->exeRequestSimple($category_url);

                    // verify that the response indicates that the endpoint is deprecated
                    if (is_array($category_data) && isset($category_data['deprecated'])) {
                        error_log(sprintf(__('SH Mapping: Category endpoint is deprecated for sport ID: %s. Message: %s', 'sh-mapping'), $sport['id'], $category_data['deprecated']));
                        continue; //jump to the next sport
                    }

                    // verify that category_data is an array
                    if (!is_array($category_data)) {
                        error_log(sprintf(__('SH Mapping: Category data is not an array for sport ID: %s', 'sh-mapping'), $sport['id']));
                        continue; // skip to the next sport
                    }

                    if (is_array($category_data) && count($category_data) > 0) {
                        foreach ($category_data as $category) {
                            $sports_mapped[$sport['id']]['categories'][$category['id']] = [
                                'id' => $category['id'],
                                'name' => $category['description'],
                            ];
                        }
                    }
                }

                if (file_exists($file_path_mapped)) unlink($file_path_mapped);
                file_put_contents($file_path_mapped, json_encode($sports_mapped));

                if ($this->initRedis() && is_array($sports_mapped) && count($sports_mapped) > 0) {
                    $this->setRedis($key, json_encode($sports_mapped), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        return $sports_mapped;
    }

    public function getMappingDataHandicappers()
    {
        // CRITICAL: Prevent API calls from web requests to avoid timeouts
        // Only WP-CLI (cron jobs) should make fresh API calls
        $is_web_request = !defined('WP_CLI') || !WP_CLI;
        $emergency_mode = get_transient('sh_mapping_emergency_mode');

        $handicappers_mapped = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyHandicappers();

        // Try Redis cache first
        if ($this->initRedis()) {
            $handicappers = $this->getRedis($key);

            if (!empty($handicappers)) {
                // TRY JSON FIRST (new format)
                if ($this->isJson($handicappers)) {
                    $unserialize_data = json_decode($handicappers, true);
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                    // FALLBACK TO SERIALIZED (old format)
                    $unserialize_data = @unserialize($handicappers);
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                }
            }
        }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameHandicappers();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        // Try file cache
        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && $this->isJson($data)) {
                    $data_unserialize = json_decode($data, true);

                    if (is_array($data_unserialize) && count($data_unserialize) > 0) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        // PREVENT API CALLS FROM WEB REQUESTS
        // Only CLI/cron should make API calls to prevent timeouts
        if ($is_web_request || $emergency_mode) {
            error_log('SH Mapping: Preventing API call to handicappers from web request (use WP-CLI: wp sh_mapping pull)');
            return $handicappers_mapped;
        }

        // Only reach here if called from WP-CLI
        @set_time_limit(60);

        $mapping_url = $this->getRequestMappingHandicappers();
        if ($mapping_url === null) {
            error_log(__('SH Mapping: Cannot fetch handicappers data due to missing API token.', 'sh-mapping'));
            return $handicappers_mapped;
        }

        $response_handicappers = $this->exeRequest($mapping_url, $file_path_mapped);

        if ($response_handicappers) {
            $handicappers = $this->parseFeedFile($file_path_mapped);

            if (is_array($handicappers) && count($handicappers) > 0) {
                foreach ($handicappers as $key => $handicapper) {
                    // Validate required fields exist
                    if (!isset($handicapper['id']) || !isset($handicapper['name'])) {
                        error_log(sprintf(__('SH Mapping: Skipping invalid handicapper data - missing id or name. Key: %s', 'sh-mapping'), $key));
                        continue;
                    }

                    $handicappers_mapped[$handicapper['id']] = $handicapper['name'];
                }

                asort($handicappers_mapped);

                if (file_exists($file_path_mapped)) unlink($file_path_mapped);
                file_put_contents($file_path_mapped, json_encode($handicappers_mapped));

                if ($this->initRedis() && is_array($handicappers_mapped) && count($handicappers_mapped) > 0) {
                    $this->setRedis($key, serialize($handicappers_mapped), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        return $handicappers_mapped;
    }

    public function getMappingDataHierarchy()
    {
        // CRITICAL: Prevent API calls from web requests to avoid timeouts
        // Only WP-CLI (cron jobs) should make fresh API calls
        $is_web_request = !defined('WP_CLI') || !WP_CLI;
        $emergency_mode = get_transient('sh_mapping_emergency_mode');

        $clear_hierarchies = [];

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameDataHierarchy();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        // Try file cache first
        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && $this->isJson($data)) {
                    $data_unserialize = json_decode($data, true);
                    if (is_array($data_unserialize) && count($data_unserialize) > 0) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        // PREVENT API CALLS FROM WEB REQUESTS
        // Only CLI/cron should make API calls to prevent timeouts
        if ($is_web_request || $emergency_mode) {
            error_log('SH Mapping: Preventing API call to hierarchy from web request (use WP-CLI: wp sh_mapping pull)');
            return $clear_hierarchies;
        }

        // Only reach here if called from WP-CLI
        @set_time_limit(60);

        $mapping_url = $this->getRequestMappinghierarchy();
        if ($mapping_url === null) {
            error_log(__('SH Mapping: Cannot fetch hierarchy data due to missing API token.', 'sh-mapping'));
            return $clear_hierarchies;
        }

        $hierarchy_request = $this->exeRequestSimple($mapping_url);

        if (!empty($hierarchy_request) && is_array($hierarchy_request) && count($hierarchy_request) > 0) {
            $order_hierarchies = $this->orderHierarchies($hierarchy_request);

            if (!empty($order_hierarchies) && is_array($order_hierarchies)) {
                $clear_hierarchies = $this->clearHierarchy($order_hierarchies);

                file_put_contents($file_path_mapped, json_encode($clear_hierarchies));

                error_log('SH Mapping: Successfully processed hierarchy with ' . count($clear_hierarchies) . ' groups');
            } else {
                error_log('SH Mapping: Failed to order hierarchy structure');
            }
        } else {
            error_log('SH Mapping: No hierarchy data received from API');
        }

        return $clear_hierarchies;
    }

    public function getMappingMembershipsData()
    {
        // CRITICAL: Prevent API calls from web requests to avoid timeouts
        // Only WP-CLI (cron jobs) should make fresh API calls
        $is_web_request = !defined('WP_CLI') || !WP_CLI;
        $emergency_mode = get_transient('sh_mapping_emergency_mode');

        $memberships_mapped = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyMemberships();

        // Try Redis cache first
        if ($this->initRedis()) {
            $memberships = $this->getRedis($key);

            if (!empty($memberships)) {
                // Try JSON first (new format)
                if ($this->isJson($memberships)) {
                    $unserialize_data = json_decode($memberships, true);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                // Fallback to serialized (old format)
                $unserialize_data = @unserialize($memberships);
                
                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
            
            // Delete corrupted data from Redis
            error_log('SH Mapping: Corrupted memberships cache data in Redis for key: ' . $key . ' - Deleting...');
            $this->deleteRedis($key);
        }
    }

    if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
        wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
    }

    $filename_data =  $this->getFileNameDataMemberships();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        // Try file cache
        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data)) {
                    $data_decode = json_decode($data, true);

                    if (is_array($data_decode) && count($data_decode) > 0) {
                        return $data_decode;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        // PREVENT API CALLS FROM WEB REQUESTS
        // Only CLI/cron should make API calls to prevent timeouts
        if ($is_web_request || $emergency_mode) {
            error_log('SH Mapping: Preventing API call to memberships from web request (use WP-CLI: wp sh_mapping pull)');
            return $memberships_mapped;
        }

        // Only reach here if called from WP-CLI
        @set_time_limit(60);

        $memberships_url = $this->getRequestMembership();
        if ($memberships_url === null) {
            error_log(__('SH Mapping: Cannot fetch memberships data due to missing API token.', 'sh-mapping'));
            return $memberships_mapped;
        }

        $execute_request_mapping = $this->exeRequest($memberships_url, $file_path_mapped);

        if ($execute_request_mapping) {
            $memberships = $this->parseFeedFile($file_path_mapped);

            if (is_array($memberships) && count($memberships) > 0) {
                foreach ($memberships as $membership) {
                    $memberships_mapped[$membership['id']] = $membership;
                }

                if ($this->initRedis() && is_array($memberships_mapped) && count($memberships_mapped) > 0) {
                    $this->setRedis($key, serialize($memberships_mapped), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        return $memberships_mapped;
    }

    public function orderHierarchies($items, $parentId = null)
    {
        // Efficient iterative approach to prevent infinite recursion
        if ($parentId === null) {
            // Index items by parent_id for fast O(1) lookup
            $by_parent = [];
            $existing_ids = [];

            foreach ($items as $item) {
                // Normalize null to empty string for consistent comparison
                $parent = $item['id_parent_group_external'] ?? '';
                $group_id = $item['id_group_external'];

                if (!isset($by_parent[$parent])) {
                    $by_parent[$parent] = [];
                }
                $by_parent[$parent][] = $item;
                $existing_ids[$group_id] = true;
            }

            // Find root elements (parent_id doesn't exist in the dataset)
            $root_parents = [];
            foreach (array_keys($by_parent) as $parent_id) {
                if (!isset($existing_ids[$parent_id])) {
                    $root_parents[] = $parent_id;
                }
            }

            if (empty($root_parents)) {
                error_log('SH Mapping: No root elements found in hierarchy');
                return [];
            }

            // Build hierarchy from roots with cycle protection
            $result = [];
            $visited = []; // Track processed elements to prevent infinite loops

            foreach ($root_parents as $root_parent) {
                if (isset($by_parent[$root_parent])) {
                    foreach ($by_parent[$root_parent] as $root_item) {
                        $result[] = $this->buildHierarchy($root_item, $by_parent, $visited);
                    }
                }
            }

            return $result;
        }

        // For backward compatibility with old calls
        return [];
    }

    private function buildHierarchy($item, &$by_parent, &$visited)
    {
        $group_id = $item['id_group_external'];

        // Cycle protection: if we already visited this ID, don't process it again
        if (isset($visited[$group_id])) {
            error_log('SH Mapping: Circular reference detected in group_id=' . $group_id);
            $item['groups'] = [];
            return $item;
        }

        // Mark as visited
        $visited[$group_id] = true;
        $item['groups'] = [];

        // Find children of this item
        if (isset($by_parent[$group_id])) {
            foreach ($by_parent[$group_id] as $child) {
                $item['groups'][] = $this->buildHierarchy($child, $by_parent, $visited);
            }
        }

        return $item;
    }

    public function clearHierarchy($data)
    {
        $result = [];

        foreach ($data as $item) {
            $key = $item['id'];
            $group_name = trim($item['group_name']);
            $group = ['group_name' => $group_name, 'items' => []];

            if (isset($item['groups'])) {
                $group['items'] = $this->clearHierarchy($item['groups']);
            }

            if (isset($item['leagues'])) {
                foreach ($item['leagues'] as $league) {
                    // Use the level 2 group ID (league['id']) instead of the league ID (league_data['id'])
                    // For example: Use 269 (level 2 group) instead of 3548 (league ID)
                    if (isset($league['id_league']) && is_array($league['id_league'])) {
                        foreach ($league['id_league'] as $league_data) {
                            $league_key = $league['id']; // Changed from $league_data['id'] to $league['id']
                            // Use alias if available, otherwise use description
                            $league_name = !empty($league_data['alias'])
                                ? $league_data['alias']
                                : $league_data['description'];
                            $group['leagues']['L' . $league_key] = $league_name;
                        }
                    }
                }
            }

            $result['G' . $key] = $group;
        }

        return $result;
    }

    public function getMappingHierarchySports($hierarchy_data)
    {
        $hierarchy_sports = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyHierarchySports();

        if (!is_array($hierarchy_data) || count($hierarchy_data) == 0) return [];
        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);

        foreach ($hierarchy_data as $hierarchy_key => $hierarchy) {
            $hierarchy_sports[$hierarchy_key] = $hierarchy['group_name'];
        }

        asort($hierarchy_sports);

        if ($this->initRedis() && is_array($hierarchy_sports) && count($hierarchy_sports) > 0) {
            $this->setRedis($key, json_encode($hierarchy_sports), SHMAPPING_REDIS_TIMEOUT);
        }

        return $hierarchy_sports;
    }

    public function getLeagues($array, &$hierarchy_leagues)
    {
        if (isset($array['leagues']) && is_array($array['leagues']) && count($array['leagues']) > 0) {
            foreach ($array['leagues'] as $key => $league) {
                $hierarchy_leagues[$key] = $league;
            }
        }
        if (isset($array['items'])) {
            foreach ($array['items'] as $group) {
                $this->getLeagues($group, $hierarchy_leagues);
            }
        }
    }

    public function getMappingHierarchyFilter($hierarchy_data, $id)
    {
        if (!is_array($hierarchy_data) || count($hierarchy_data) == 0 || empty($id)) return [];

        $hierarchies = [];
        $indexes = explode('-', $id);
        $end = $indexes[count($indexes) - 1];

        $this->navigateArrayRecursively($hierarchy_data, $indexes, $end, $hierarchies);

        return $hierarchies;
    }

    public function navigateArrayRecursively($array, $indexes, $end, &$hierarchies)
    {
        $currentIndex = array_shift($indexes);

        if (isset($array[$currentIndex])) {
            if ($currentIndex != $end) {
                return $this->navigateArrayRecursively($array[$currentIndex]['items'], $indexes, $end, $hierarchies);
            }

            $items = [];
            $leagues = [];

            // Check if there are direct leagues first (more specific)
            if (
                isset($array[$currentIndex]['leagues']) &&
                is_array($array[$currentIndex]['leagues']) &&
                count($array[$currentIndex]['leagues']) > 0
            ) {
                $leagues = $array[$currentIndex]['leagues'];
            }

            // Only include items (child groups) if there are no direct leagues
            // to avoid duplicates when both exist with the same names
            if (
                empty($leagues) && isset($array[$currentIndex]['items']) &&
                is_array($array[$currentIndex]['items']) &&
                count($array[$currentIndex]['items']) > 0
            ) {
                foreach ($array[$currentIndex]['items'] as $key => $item) {
                    $items[$key . '-child'] = $item['group_name'];
                }
            }

            // If we have leagues, use only leagues; otherwise use items
            $hierarchies = !empty($leagues) ? $leagues : $items;

            // Special case: if we have both items and leagues with different content,
            // we might want to include both (for future flexibility)
            // For now, prioritize leagues to avoid duplicates
        }

        return [];
    }

    public function getMappingDataPlayers()
    {
        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameDataPlayers();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data)) {
                    $data_unserialize = json_decode($data, true);

                    return $data_unserialize;
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return [];
    }

    public function loadMappingDataPlayers()
    {
        $players_mapped = [];

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameDataPlayers();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        $mapping_url = $this->getRequestMappingPlayers();
        if ($mapping_url === null) {
            error_log(__('SH Mapping: Cannot fetch players data due to missing API token.', 'sh-mapping'));
            return $players_mapped;
        }

        $request_mapping = $this->exeRequestSimple($mapping_url);

        if (is_array($request_mapping) && count($request_mapping) > 0) {
            if (is_array($request_mapping[0]['data-content'][0][0]['players']) && count($request_mapping[0]['data-content'][0][0]['players']) > 0) {
                $players = $request_mapping[0]['data-content'][0][0]['players'];

                foreach ($players as $player) {
                    $players_mapped[$player['id']] = [
                        'id' => $player['id'],
                        'first_name' => $player['first_name'],
                        'last_name' => $player['last_name'],
                    ];
                }

                if (is_array($players_mapped) && count($players_mapped) > 0) {
                    file_put_contents($file_path_mapped, json_encode($players_mapped));
                }
            }
        }

        return $players_mapped;
    }

    public function getMappingDataSeasons($data)
    {
        // CRITICAL: Prevent API calls from web requests to avoid timeouts
        // Only WP-CLI (cron jobs) should make fresh API calls
        $is_web_request = !defined('WP_CLI') || !WP_CLI;
        $emergency_mode = get_transient('sh_mapping_emergency_mode');

        $seasons_array = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);

        if ($this->initRedis()) {
            $key = $this->getRedisKeySeasonsDirect();
            $seasons_data = $this->getRedis($key);

            if (!empty($seasons_data)) {
                // Try JSON first (new format)
                if ($this->isJson($seasons_data)) {
                    $unserialize_data = json_decode($seasons_data, true);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                // Fallback to serialized (old format)
                $unserialize_data = @unserialize($seasons_data);
                
                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
            
            // Delete corrupted data from Redis
            error_log('SH Mapping: Corrupted seasons cache data in Redis for key: ' . $key . ' - Deleting...');
            $this->deleteRedis($key);
        }
    }

        if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameSeasonsDirect();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        // Try file cache first
        try {
            if (file_exists($file_path_mapped)) {
                $data_file = file_get_contents($file_path_mapped);

                if (!empty($data_file) && $this->isJson($data_file)) {
                    $data_unserialize = json_decode($data_file, true);
                    if (is_array($data_unserialize) && count($data_unserialize) > 0) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        // PREVENT API CALLS FROM WEB REQUESTS
        // Only CLI/cron should make API calls to prevent timeouts
        if ($is_web_request || $emergency_mode) {
            error_log('SH Mapping: Preventing API call to seasons data from web request (use WP-CLI: wp sh_mapping pull)');
            return $seasons_array;
        }

        // Only reach here if called from WP-CLI
        @set_time_limit(60);

        $mapping_url = $this->getRequestMappingSeasons();
        if ($mapping_url === null) {
            error_log(__('SH Mapping: Cannot fetch seasons data due to missing API token.', 'sh-mapping'));
            return $seasons_array;
        }

        $execute_request_mapping_seasons = $this->exeRequest($mapping_url, $file_path_mapped);

        if ($execute_request_mapping_seasons) {
            $parse_results_mapping = $this->parseFeedFile($file_path_mapped);

            if (!empty($parse_results_mapping) && isset($parse_results_mapping[0]['data-content'][0][0]['seasons']) && is_array($parse_results_mapping[0]['data-content'][0][0]['seasons'])) {
                $seasons = $parse_results_mapping[0]['data-content'][0][0]['seasons'];

                foreach ($seasons as $season) {
                    $seasons_array[$season['league_id']][$season['id']] = $season['name'];
                }

                if ($this->initRedis() && is_array($seasons_array) && count($seasons_array) > 0) {
                    $this->setRedis($key, serialize($seasons_array), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        try {
            file_put_contents($file_path_mapped, json_encode($seasons_array));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $seasons_array;
    }

    public function getMappingDataSports()
    {
        // CRITICAL: Prevent API calls from web requests to avoid timeouts
        // Only WP-CLI (cron jobs) should make fresh API calls
        $is_web_request = !defined('WP_CLI') || !WP_CLI;
        $emergency_mode = get_transient('sh_mapping_emergency_mode');

        $sports_data = [];
        $key = $this->getRedisKeySportsDirect();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);

            if (!empty($data_sports)) {
                // Try JSON first (new format)
                if ($this->isJson($data_sports)) {
                    $unserialize_data = json_decode($data_sports, true);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                // Fallback to serialized (old format)
                $unserialize_data = @unserialize($data_sports);
                
                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
            
            // Delete corrupted data from Redis
            error_log('SH Mapping: Corrupted sports direct cache data in Redis for key: ' . $key . ' - Deleting...');
            $this->deleteRedis($key);
        }
    }

    if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
        wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
    }

        $filename_data =  $this->getFileNameSportsDirect();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        // Try file cache first
        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && $this->isJson($data)) {
                    $data_unserialize = json_decode($data, true);
                    if (is_array($data_unserialize) && count($data_unserialize) > 0) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        // PREVENT API CALLS FROM WEB REQUESTS
        // Only CLI/cron should make API calls to prevent timeouts
        if ($is_web_request || $emergency_mode) {
            error_log('SH Mapping: Preventing API call to sports data from web request (use WP-CLI: wp sh_mapping pull)');
            return $sports_data;
        }

        // Only reach here if called from WP-CLI
        @set_time_limit(60);

        $mapping_url = $this->getRequestMappingSports();
        if ($mapping_url === null) {
            error_log(__('SH Mapping: Cannot fetch sports data due to missing API token.', 'sh-mapping'));
            return $sports_data;
        }

        $execute_request_mapping = $this->exeRequest($mapping_url, $file_path_mapped);

        if ($execute_request_mapping) {
            $parse_results_mapping = $this->parseFeedFile($file_path_mapped);

            if (!empty($parse_results_mapping) && isset($parse_results_mapping[0]['data-content'][0][0]['sports']) && is_array($parse_results_mapping[0]['data-content'][0][0]['sports']) && count($parse_results_mapping[0]['data-content'][0][0]['sports']) > 0) {
                $sports = $parse_results_mapping[0]['data-content'][0][0]['sports'];

                foreach ($sports as $sport) {
                    if (!is_array($sport) || !isset($sport['id']) || !isset($sport['name'])) {
                        continue;
                    }
                    $sports_data[$sport['id']] = $this->getCapitalName($sport['name']);
                }

                asort($sports_data);

                if ($this->initRedis() && is_array($sports_data) && count($sports_data) > 0) {
                    $this->setRedis($key, serialize($sports_data), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        try {
            file_put_contents($file_path_mapped, json_encode($sports_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $sports_data;
    }

    public function getMappingDataLeagues()
    {
        // CRITICAL: Prevent API calls from web requests to avoid timeouts
        // Only WP-CLI (cron jobs) should make fresh API calls
        $is_web_request = !defined('WP_CLI') || !WP_CLI;
        $emergency_mode = get_transient('sh_mapping_emergency_mode');

        $leagues_data = [];
        $key = $this->getRedisKeyLeaguesSportDirect();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);

            if (!empty($data_sports)) {
                // Try JSON first (new format)
                if ($this->isJson($data_sports)) {
                    $unserialize_data = json_decode($data_sports, true);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                // Fallback to serialized (old format)
                $unserialize_data = @unserialize($data_sports);
                
                if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                    return $unserialize_data;
                }
            }
            
            // Delete corrupted data from Redis
            error_log('SH Mapping: Corrupted leagues direct cache data in Redis for key: ' . $key . ' - Deleting...');
            $this->deleteRedis($key);
        }
    }

    if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
            wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
        }

        $filename_data =  $this->getFileNameLeaguesSportDirect();
        $file_path_mapped = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_data;

        // Try file cache first
        try {
            if (file_exists($file_path_mapped)) {
                $data = file_get_contents($file_path_mapped);

                if (!empty($data) && $this->isJson($data)) {
                    $data_unserialize = json_decode($data, true);
                    if (is_array($data_unserialize) && count($data_unserialize) > 0) {
                        return $data_unserialize;
                    }
                }
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        // PREVENT API CALLS FROM WEB REQUESTS
        // Only CLI/cron should make API calls to prevent timeouts
        if ($is_web_request || $emergency_mode) {
            error_log('SH Mapping: Preventing API call to leagues data from web request (use WP-CLI: wp sh_mapping pull)');
            return $leagues_data;
        }

        // Only reach here if called from WP-CLI
        @set_time_limit(60);

        $mapping_url = $this->getRequestMappingLeagues();
        if ($mapping_url === null) {
            error_log(__('SH Mapping: Cannot fetch leagues data due to missing API token.', 'sh-mapping'));
            return $leagues_data;
        }

        $execute_request_mapping = $this->exeRequest($mapping_url, $file_path_mapped);

        if ($execute_request_mapping) {
            $parse_results_mapping = $this->parseFeedFile($file_path_mapped);

            if (!empty($parse_results_mapping) && isset($parse_results_mapping[0]['data-content'][0][0]['leagues']) && is_array($parse_results_mapping[0]['data-content'][0][0]['leagues']) && count($parse_results_mapping[0]['data-content'][0][0]['leagues']) > 0) {
                $leagues = $parse_results_mapping[0]['data-content'][0][0]['leagues'];

                foreach ($leagues as $league) {
                    if (!is_array($league) || !isset($league['sport_id']) || !isset($league['id']) || !isset($league['name'])) {
                        continue;
                    }
                    $leagues_data[$league['sport_id']][] = [
                        'id' => $league['id'],
                        'name' => $league['name'],
                        'nickname' => $league['nickname'] ?? '',
                    ];
                }

                asort($leagues_data);

                if ($this->initRedis() && is_array($leagues_data) && count($leagues_data) > 0) {
                    $this->setRedis($key, serialize($leagues_data), SHMAPPING_REDIS_TIMEOUT);
                }
            }
        }

        try {
            file_put_contents($file_path_mapped, json_encode($leagues_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $leagues_data;
    }

    private function getCapitalName($name)
    {
        if ($name == 'horseracing') {
            $name = 'horse racing';
        }

        return ucwords($name);
    }

    public function delMappingData()
    {
        if ($this->initRedis()) {
            try {
                $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
                $key_sports = $this->getRedisKeyMapping();
                $key_sports_options = $this->getRedisKeySportsLeaguesOptions();
                $key_sports_titles = $this->getRedisKeySportsLeaguesTitles();
                $key_sports_direct = $this->getRedisKeySportsDirect();
                $key_leagues_sport_direct = $this->getRedisKeyLeaguesSportDirect();
                $key_leagues_direct = $this->getRedisKeyLeaguesDirect();
                $key_teams_direct = $this->getRedisKeyTeamsDirect();
                $key_seasons_direct = $this->getRedisKeySeasonsDirect();
                $key_sports_only = $this->getRedisKeySportsOnly();
                $hierarchy_sports = $this->getRedisKeyHierarchySports();
                $key_handicappers = $this->getRedisKeyHandicappers();
                $key_memberships = $this->getRedisKeyMemberships();
                $key_memberships_logic = $this->getRedisKeyMembershipsLogic();
                $key_leaderboard = $this->getRedisKeyLeaderboardData();
                $key_leaderboard_sports = $this->getRedisKeyLeaderboardSports();
                $key_leaderboard_categories = $this->getRedisKeyLeaderboardCategories();
                $key_hierarchy_sports_leagues = $this->getRedisKeyHierarchySportsLeagues();

                // DELETE keys instead of setting to empty arrays
                $this->deleteRedis($key_sports);
                $this->deleteRedis($key_sports_options);
                $this->deleteRedis($key_sports_titles);
                $this->deleteRedis($key_sports_direct);
                $this->deleteRedis($key_leagues_direct);
                $this->deleteRedis($key_teams_direct);
                $this->deleteRedis($key_seasons_direct);
                $this->deleteRedis($key_leagues_sport_direct);
                $this->deleteRedis($key_sports_only);
                $this->deleteRedis($hierarchy_sports);
                $this->deleteRedis($key_handicappers);
                $this->deleteRedis($key_memberships);
                $this->deleteRedis($key_memberships_logic);
                $this->deleteRedis($key_leaderboard);
                $this->deleteRedis($key_leaderboard_sports);
                $this->deleteRedis($key_leaderboard_categories);
                $this->deleteRedis($key_hierarchy_sports_leagues);
            } catch (\Exception $exception) {
                error_log(print_r([
                    'message' => $exception->getMessage(),
                    'keys' => [
                        $key_sports,
                        $key_sports_options,
                        $key_sports_titles,
                        $key_teams_direct,
                        $key_leagues_direct,
                        $key_seasons_direct,
                        $key_sports_direct,
                        $key_leagues_sport_direct,
                        $key_sports_only,
                        $hierarchy_sports,
                        $key_handicappers,
                        $key_memberships,
                        $key_memberships_logic,
                        $key_leaderboard,
                        $key_leaderboard_sports,
                        $key_leaderboard_categories,
                        $key_hierarchy_sports_leagues
                    ],
                ], true));
            }
        }

        try {
            $cache_data = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameData();
            if (file_exists($cache_data)) {
                unlink($cache_data);
            }

            $cache_sports_options = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameSportsOptions();
            if (file_exists($cache_sports_options)) {
                unlink($cache_sports_options);
            }

            $cache_sports_titles = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameTitles();
            if (file_exists($cache_sports_titles)) {
                unlink($cache_sports_titles);
            }

            $cache_sports_direct = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameSportsDirect();
            if (file_exists($cache_sports_direct)) {
                unlink($cache_sports_direct);
            }

            $cache_leagues_sport_direct = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameLeaguesSportDirect();
            if (file_exists($cache_leagues_sport_direct)) {
                unlink($cache_leagues_sport_direct);
            }

            $cache_leagues_direct = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameLeaguesDirect();
            if (file_exists($cache_leagues_direct)) {
                unlink($cache_leagues_direct);
            }

            $cache_teams_direct = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameTeamsDirect();
            if (file_exists($cache_teams_direct)) {
                unlink($cache_teams_direct);
            }

            $cache_seasons_direct = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameSeasonsDirect();
            if (file_exists($cache_seasons_direct)) {
                unlink($cache_seasons_direct);
            }

            $cache_sports_only = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameSportsOnly();
            if (file_exists($cache_sports_only)) {
                unlink($cache_sports_only);
            }

            $cache_hierarchy = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameDataHierarchy();
            if (file_exists($cache_hierarchy)) {
                unlink($cache_hierarchy);
            }

            $cache_handicappers = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameHandicappers();
            if (file_exists($cache_handicappers)) {
                unlink($cache_handicappers);
            }

            $cache_memberships = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameDataMemberships();
            if (file_exists($cache_memberships)) {
                unlink($cache_memberships);
            }

            $cache_memberships_logic = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileMembersLogic();
            if (file_exists($cache_memberships_logic)) {
                unlink($cache_memberships_logic);
            }

            $cache_leaderboard = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameLeaderboard();
            if (file_exists($cache_leaderboard)) {
                unlink($cache_leaderboard);
            }

            $cache_leaderboard_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameLeaderboardSports();
            if (file_exists($cache_leaderboard_sports)) {
                unlink($cache_leaderboard_sports);
            }

            $cache_leaderboard_categories = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameLeaderboardCategories();
            if (file_exists($cache_leaderboard_categories)) {
                unlink($cache_leaderboard_categories);
            }

            $cache_hierarchy_sports_leagues = SHMAPPING_PLUGIN_LOG_DIR . '/' . $this->getFileNameHierarchySportsLeagues();
            if (file_exists($cache_hierarchy_sports_leagues)) {
                unlink($cache_hierarchy_sports_leagues);
            }
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => $exception->getMessage(),
            ], true));
        }
    }

    public function getRequestMapping()
    {
        $api_base = (!empty(get_field('sh_mapping_url', 'options')))
            ? get_field('sh_mapping_url', 'options')
            : 'http://masterfeed.shnadmin.com/feed/mapped/get_data';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');

        if (!$this->isValidApiToken($api_token, 'masterfeed')) {
            return null;
        }

        return $api_base . '?key=' . $api_token;
    }

    public function getRequestMappinghierarchy()
    {
        $api_base = (!empty(get_field('sh_mapping_url_hierarchy', 'options')))
            ? get_field('sh_mapping_url_hierarchy', 'options')
            : 'https://members.sportshub.com/api/get-hierarchy';
        $api_token = get_field('sh_mapping_members_token', 'option');

        if (!$this->isValidApiToken($api_token, 'members')) {
            return null;
        }

        return $api_base . '?api_token=' . $api_token;
    }

    private function getRequestMappingPlayers()
    {
        $api_base = (!empty(get_field('sh_mapping_url_players', 'options')))
            ? get_field('sh_mapping_url_players', 'options')
            : 'http://masterfeed.shnadmin.com/feed/sportdirect/players';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');

        if (!$this->isValidApiToken($api_token, 'masterfeed')) {
            return null;
        }

        return $api_base . '?key=' . $api_token;
    }

    private function getRequestMappingSports()
    {
        $api_base = (!empty(get_field('sh_mapping_url_sports', 'options')))
            ? get_field('sh_mapping_url_sports', 'options')
            : 'http://masterfeed.shnadmin.com/feed/sportdirect/sports';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');

        if (!$this->isValidApiToken($api_token, 'masterfeed')) {
            return null;
        }

        return $api_base . '?key=' . $api_token;
    }

    private function getRequestMappingLeagues()
    {
        $api_base = (!empty(get_field('sh_mapping_url_leagues', 'options')))
            ? get_field('sh_mapping_url_leagues', 'options')
            : 'http://masterfeed.shnadmin.com/feed/sportdirect/leagues';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');

        if (!$this->isValidApiToken($api_token, 'masterfeed')) {
            return null;
        }

        return $api_base . '?key=' . $api_token;
    }

    private function getRequestMappingSeasons()
    {
        $api_base = (!empty(get_field('sh_mapping_url_seasons', 'options')))
            ? get_field('sh_mapping_url_seasons', 'options')
            : 'http://masterfeed.shnadmin.com/feed/sportdirect/seasons';
        $api_token = get_field('sh_mapping_masterfeed_token', 'option');

        if (!$this->isValidApiToken($api_token, 'masterfeed')) {
            return null;
        }

        return $api_base . '?key=' . $api_token;
    }

    private function getRequestMappingHandicappers()
    {
        $api_base = (!empty(get_field('sh_mapping_url_handicappers', 'options')))
            ? get_field('sh_mapping_url_handicappers', 'options')
            : 'https://members.sportshub.com/api/handicappers/data';
        $api_token = get_field('sh_mapping_members_token', 'option');

        if (!$this->isValidApiToken($api_token, 'members')) {
            return null;
        }

        return $api_base . '?api_token=' . $api_token;
    }

    private function getRequestMembership()
    {
        $api_base = (!empty(get_field('sh_mapping_membership_url', 'options')))
            ? get_field('sh_mapping_membership_url', 'options')
            : 'https://members.sportshub.com/api/membership-info';
        $api_token = get_field('sh_mapping_members_token', 'option');

        if (!$this->isValidApiToken($api_token, 'members')) {
            return null;
        }

        return $api_base . '?api_token=' . $api_token;
    }

    private function getRequestLeaderboardSports()
    {
        $api_base = (!empty(get_field('sh_mapping_leaderboard_sports_url', 'options')))
            ? get_field('sh_mapping_leaderboard_sports_url', 'options')
            : 'https://members.sportshub.com/api/leaderboardData/sports';
        $api_token = get_field('sh_mapping_members_token', 'option');

        if (!$this->isValidApiToken($api_token, 'members')) {
            return null;
        }

        return $api_base . '?api_token=' . $api_token;
    }

    private function getRequestLeaderboardCategories()
    {
        $api_base = (!empty(get_field('sh_mapping_leaderboard_categories_url', 'options')))
            ? get_field('sh_mapping_leaderboard_categories_url', 'options')
            : 'https://members.sportshub.com/api/leaderboardData/categories';
        $api_token = get_field('sh_mapping_members_token', 'option');

        if (!$this->isValidApiToken($api_token, 'members')) {
            return null;
        }

        return $api_base . '?api_token=' . $api_token;
    }

    private function exeRequest($request, $file_path)
    {
        $ch = curl_init();
        $this->configureCurlOptions($ch, $request);
        $response = curl_exec($ch);

        if ($response === false) {
            $curl_error = curl_error($ch);
            $curl_errno = curl_errno($ch);
            curl_close($ch);

            // Check for timeout errors specifically
            if ($curl_errno === CURLE_OPERATION_TIMEDOUT) {
                $error_message = sprintf(__('SH Mapping: Request timeout after 30 seconds for URL: %s', 'sh-mapping'), $request);
            } else {
                $error_message = sprintf(__('SH Mapping: cURL ERROR: %s (Code: %d)', 'sh-mapping'), $curl_error, $curl_errno);
            }

            error_log($error_message);
            if (defined('WP_CLI') && WP_CLI) {
                WP_CLI::line($error_message);
            }
        } else {
            if ($this->isJson($response)) {
                $response_json = json_decode($response, true);

                if (isset($response_json['error'])) {
                    $error_message = $this->formatApiErrorMessage($response_json['error'], $request);

                    error_log(print_r([
                        'response_json' => $response_json,
                        'request' => $request,
                        'formatted_message' => $error_message,
                    ], true));

                    return false;
                }
            }

            file_put_contents($file_path, $response);
            curl_close($ch);

            return true;
        }

        return false;
    }

    private function exeRequestSimple($request)
    {
        $response_json = [];
        $ch = curl_init();
        $this->configureCurlOptions($ch, $request);
        $response = curl_exec($ch);

        if ($response === false) {
            $curl_error = curl_error($ch);
            $curl_errno = curl_errno($ch);
            curl_close($ch);

            // Check for timeout errors specifically
            if ($curl_errno === CURLE_OPERATION_TIMEDOUT) {
                $error_message = sprintf(__('SH Mapping: Request timeout after 30 seconds for URL: %s', 'sh-mapping'), $request);
            } else {
                $error_message = sprintf(__('SH Mapping: cURL ERROR: %s (Code: %d)', 'sh-mapping'), $curl_error, $curl_errno);
            }

            error_log($error_message);
            WP_CLI::line($error_message);
        } else {
            if ($this->isJson($response)) {
                $response_json = json_decode($response, true);

                // verify that the result is an array
                if (!is_array($response_json)) {
                    error_log(sprintf(__('SH Mapping: Response is not a valid array for URL: %s', 'sh-mapping'), $request));
                    return [];
                }

                if (isset($response_json['error'])) {
                    $error_message = $this->formatApiErrorMessage($response_json['error'], $request);

                    error_log(print_r([
                        'response_json' => $response_json,
                        'request' => $request,
                        'formatted_message' => $error_message,
                    ], true));

                    return [];
                }
            } else {
                // the response is not valid JSON
                error_log(sprintf(__('SH Mapping: Response is not valid JSON for URL: %s', 'sh-mapping'), $request));
                return [];
            }

            curl_close($ch);

            return $response_json;
        }

        return $response_json;
    }

    function isJson($string)
    {
        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }

    private function isValidApiToken($token, $api_type)
    {
        if (empty($token) || !is_string($token) || trim($token) === '') {
            error_log(sprintf(
                __('SH Mapping: Missing or empty API token for %s. Please configure the token in WordPress Admin > SH Mapping > Settings.', 'sh-mapping'),
                $api_type
            ));
            return false;
        }

        return true;
    }

    private function formatApiErrorMessage($error, $request_url)
    {
        if ($error === 'Unauthenticated.') {
            $api_type = $this->determineApiTypeFromUrl($request_url);
            return sprintf(
                __('SH Mapping: Authentication failed for %s API. Please check your API token configuration in WordPress Admin > SH Mapping > Settings.', 'sh-mapping'),
                $api_type
            );
        }

        return sprintf(__('SH Mapping: API Error: %s', 'sh-mapping'), $error);
    }

    private function configureCurlOptions($ch, $url, $timeout = 15, $connect_timeout = 5)
    {
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeout); // 15 seconds - increased for slow APIs
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $connect_timeout); // 5 seconds to connect
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 5);
        curl_setopt($ch, CURLOPT_USERAGENT, 'SH-Mapping-Plugin/' . SHMAPPING_VERSION);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: application/json',
            'Content-Type: application/json'
        ]);

        // SSL configuration - more secure for production
        if (strpos($url, 'https://') === 0) {
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        } else {
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        }
    }

    private function determineApiTypeFromUrl($url)
    {
        if (strpos($url, 'masterfeed.shnadmin.com') !== false) {
            return 'Masterfeed';
        } elseif (strpos($url, 'members.sportshub.com') !== false) {
            return 'Members';
        }

        return 'Unknown';
    }

    private function parseFeedFile($file_path)
    {
        $file_contents = file_get_contents($file_path);

        if ($file_contents === false) {
            error_log(__('SH Mapping Error: Failed to read the file.', 'sh-mapping'));

            return null;
        }

        $json_data = json_decode($file_contents, true);

        if ($json_data === null) {
            error_log(__('SH Mapping Error: Failed to decode the JSON.', 'sh-mapping'));

            return null;
        }

        return $json_data;
    }

    public function getSports($data)
    {
        $sports_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeySportsLeaguesOptions();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);

            if (!empty($data_sports)) {
                // Try JSON first (new format)
                if ($this->isJson($data_sports)) {
                    $unserialize_data = json_decode($data_sports, true);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                    // Fallback to serialized data (old format)
                    $unserialize_data = @unserialize($data_sports);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            // Validate that $sport is an array
            if (!is_array($sport)) {
                error_log('SH Mapping: Invalid sport data at index ' . $sport_id . ' in getSports() - Expected array, got ' . gettype($sport));
                continue;
            }

            if (!isset($sport['id']) || !isset($sport['name'])) {
                error_log('SH Mapping: Missing "id" or "name" key in sport data at index ' . $sport_id . ' in getSports()');
                continue;
            }

            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                $leagues = [];
                $leagues[$sport['id']] = $sport['name'] . ' - ' . __('Any Category');

                foreach ($sport['leagues'] as $league) {
                    if (!is_array($league) || !isset($league['id']) || !isset($league['name'])) {
                        continue;
                    }
                    $leagues[$sport['id'] . '-' . $league['id']] = $league['name'];
                }
            } else {
                $leagues = $sport['name'];
            }

            if (is_array($leagues)) {
                $sports_data[$sport['name']] = $leagues;
            } else {
                $sports_data[$sport['id']] = $leagues;
            }
        }

        if ($this->initRedis() && is_array($sports_data) && count($sports_data) > 0) {
            $this->setRedis($key, json_encode($sports_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }

            $filename_sports =  $this->getFileNameSportsOptions();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;

            file_put_contents($file_path_sports, json_encode($sports_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $sports_data;
    }

    public function getSportsOnly($data)
    {
        $sports_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeySportsOnly();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);

            if (!empty($data_sports)) {
                // Try JSON first (new format)
                if ($this->isJson($data_sports)) {
                    $unserialize_data = json_decode($data_sports, true);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                    // Fallback to serialized data (old format)
                    $unserialize_data = @unserialize($data_sports);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            // Validate that $sport is an array
            if (!is_array($sport)) {
                error_log('SH Mapping: Invalid sport data at index ' . $sport_id . ' in getSportsOnly() - Expected array, got ' . gettype($sport));
                continue;
            }

            if (!isset($sport['id']) || !isset($sport['name'])) {
                error_log('SH Mapping: Missing "id" or "name" key in sport data at index ' . $sport_id . ' in getSportsOnly()');
                continue;
            }

            $sports_data[$sport['id']] = $sport['name'];
        }

        if ($this->initRedis() && is_array($sports_data) && count($sports_data) > 0) {
            $this->setRedis($key, json_encode($sports_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }

            $filename_sports =  $this->getFileNameSportsOnly();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;

            file_put_contents($file_path_sports, json_encode($sports_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $sports_data;
    }

    public function getSportsTitles($data)
    {
        $sport_titles = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeySportsLeaguesTitles();

        if ($this->initRedis()) {
            $data_sports = $this->getRedis($key);

            if (!empty($data_sports)) {
                // Try JSON first (new format)
                if ($this->isJson($data_sports)) {
                    $unserialize_data = json_decode($data_sports, true);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                    // Fallback to serialized data (old format)
                    $unserialize_data = @unserialize($data_sports);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            // Validate that $sport is an array
            if (!is_array($sport)) {
                error_log('SH Mapping: Invalid sport data at index ' . $sport_id . ' in getSportsTitles() - Expected array, got ' . gettype($sport));
                continue;
            }

            if (!isset($sport['id']) || !isset($sport['name'])) {
                error_log('SH Mapping: Missing "id" or "name" key in sport data at index ' . $sport_id . ' in getSportsTitles()');
                continue;
            }

            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                $sport_titles[$sport['id']] = $sport['name'];

                foreach ($sport['leagues'] as $league) {
                    if (!is_array($league) || !isset($league['id']) || !isset($league['name'])) {
                        continue;
                    }
                    $sport_titles[$sport['id'] . '-' . $league['id']] = $sport['name'] . ' | ' . $league['name'];
                }
            } else {
                $sport_titles[$sport['id']] = $sport['name'];
            }
        }

        if ($this->initRedis() && is_array($sport_titles) && count($sport_titles) > 0) {
            $this->setRedis($key, json_encode($sport_titles), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }

            $filename_titles =  $this->getFileNameTitles();
            $file_path_titles = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_titles;

            file_put_contents($file_path_titles, json_encode($sport_titles));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $sport_titles;
    }

    public function getSportsLeaguesDirect($data)
    {
        $leagues_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyLeaguesDirect();

        if ($this->initRedis()) {
            $data_leagues = $this->getRedis($key);

            if (!empty($data_leagues)) {
                // Try JSON first (new format)
                if ($this->isJson($data_leagues)) {
                    $unserialize_data = json_decode($data_leagues, true);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                    // Fallback to serialized data (old format)
                    $unserialize_data = @unserialize($data_leagues);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            // Validate that $sport is an array
            if (!is_array($sport)) {
                error_log('SH Mapping: Invalid sport data at index ' . $sport_id . ' in getSportsLeaguesDirect() - Expected array, got ' . gettype($sport));
                continue;
            }

            if (!isset($sport['name'])) {
                error_log('SH Mapping: Missing "name" key in sport data at index ' . $sport_id . ' in getSportsLeaguesDirect()');
                continue;
            }

            $sport_name = $sport['name'];
            $leagues = [];

            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                foreach ($sport['leagues'] as $league_id => $league) {
                    if (isset($league['sports_direct'][0]) && !empty($league['sports_direct'][0])) {
                        $leagues[$league['sports_direct'][0]] = $league['name'];
                    }
                }
            }

            if (is_array($leagues) && count($leagues) > 0) {
                $leagues_data[$sport_name] = $leagues;
            }
        }

        if ($this->initRedis() && is_array($leagues_data) && count($leagues_data) > 0) {
            $this->setRedis($key, json_encode($leagues_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }

            $filename_sports =  $this->getFileNameLeaguesDirect();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;

            file_put_contents($file_path_sports, json_encode($leagues_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $leagues_data;
    }

    public function getSportsTeamsDirect($data)
    {
        $teams_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyTeamsDirect();

        if ($this->initRedis()) {
            $data_teams = $this->getRedis($key);

            if (!empty($data_teams)) {
                // Try JSON first (new format)
                if ($this->isJson($data_teams)) {
                    $unserialize_data = json_decode($data_teams, true);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                } else {
                    // Fallback to serialized data (old format)
                    $unserialize_data = @unserialize($data_teams);
                    
                    if (is_array($unserialize_data) && count($unserialize_data) > 0) {
                        return $unserialize_data;
                    }
                }
            }
        }

        if (!is_array($data) || count($data) == 0) {
            return [];
        }

        foreach ($data as $sport_id => $sport) {
            // Validate that $sport is an array
            if (!is_array($sport)) {
                error_log('SH Mapping: Invalid sport data at index ' . $sport_id . ' in getSportsTeamsDirect() - Expected array, got ' . gettype($sport));
                continue;
            }

            if (isset($sport['leagues']) && is_array($sport['leagues']) && count($sport['leagues']) > 0) {
                foreach ($sport['leagues'] as $league_id => $league) {
                    if (!is_array($league)) {
                        continue;
                    }

                    $teams = [];
                    $legue_id = (isset($league['sports_direct']) && is_array($league['sports_direct']) && count($league['sports_direct']) > 0)
                        ? $league['sports_direct'][0]
                        : $league['id'];

                    if (isset($league['teams']) && is_array($league['teams']) && count($league['teams']) > 0) {
                        foreach ($league['teams'] as $team_id => $team) {
                            if (!is_array($team)) {
                                continue;
                            }

                            if (isset($team['sports_direct'][0]) && !empty($team['sports_direct'][0]) && isset($team['name'])) {
                                $teams[$team['sports_direct'][0]] = $team['name'];
                            }
                        }
                    }

                    if (is_array($teams) && count($teams) > 0) {
                        $teams_data[$legue_id] = $teams;
                    }
                }
            }
        }

        if ($this->initRedis() && is_array($teams_data) && count($teams_data) > 0) {
            $this->setRedis($key, json_encode($teams_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }

            $filename_sports =  $this->getFileNameTeamsDirect();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;

            file_put_contents($file_path_sports, json_encode($teams_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $teams_data;
    }

    public function getLeaguesNames($data)
    {
        $leagues_array = [];

        foreach ($data as $league => $seasons) {
            foreach ($seasons as $id => $name) {
                $leagues_array[$id] = $name;
            }
        }

        return $leagues_array;
    }

    public function getLeaguesSport($leagues, $sport_id)
    {
        if (empty($sport_id) || !isset($leagues[$sport_id]) || !is_array($leagues[$sport_id]) || count($leagues[$sport_id]) == 0) return [];

        $leagues_array = [];
        foreach ($leagues[$sport_id] as $league) {
            if (!is_array($league) || !isset($league['id']) || !isset($league['name'])) {
                continue;
            }
            $leagues_array[$league['id']] = $league['name'];
        }

        asort($leagues_array);

        return $leagues_array;
    }

    public function getTeamsLeague($teams, $league_id)
    {
        if (empty($league_id) || !isset($teams[$league_id]) || !is_array($teams[$league_id]) || count($teams[$league_id]) == 0) return [];

        $teams_array = (isset($teams[$league_id])) ? $teams[$league_id] : [];
        asort($teams_array);

        return $teams_array;
    }

    public function getSeasonsLeague($seasons, $league_id)
    {
        if (empty($league_id) || !isset($seasons[$league_id]) || !is_array($seasons[$league_id]) || count($seasons[$league_id]) == 0) return [];

        $seasons_array = (isset($seasons[$league_id])) ? $seasons[$league_id] : [];
        asort($seasons_array);

        return $seasons_array;
    }

    public function getMembersLogic($configs)
    {
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);

        $memberships = [];
        $memberships_mapped = $this->getMappingMembershipsData();

        if (is_array($memberships_mapped) && count($memberships_mapped) > 0) {
            foreach ($memberships_mapped as $membership_key => $membership_data) {
                // Validate required fields exist
                if (!isset($membership_data['id']) || !isset($membership_data['name'])) {
                    error_log(sprintf(__('SH Mapping: Skipping invalid membership data - missing id or name. Key: %s', 'sh-mapping'), $membership_key));
                    continue;
                }

                $memberships[$membership_data['id']] = [
                    'name' => $membership_data['name'],
                    'url' => isset($membership_data['url']) ? $membership_data['url'] : '',
                ];
            }
        }

        if (!is_array($memberships) || count($memberships) == 0) return [];

        $packages = [];
        foreach ($configs as $config_key => $config) {
            // Validate config has required fields
            if (!isset($config['key']) || !isset($config['value'])) {
                error_log(sprintf(__('SH Mapping: Skipping invalid config - missing key or value. Config key: %s', 'sh-mapping'), $config_key));
                continue;
            }

            // Validate membership exists for this config value
            if (!isset($memberships[$config['value']])) {
                error_log(sprintf(__('SH Mapping: Skipping config - membership not found for value: %s', 'sh-mapping'), $config['value']));
                continue;
            }

            $key_new = $config['key'];
            if (empty($config['key'])) $key_new = 0;
            $data = $memberships[$config['value']];
            $packages[$key_new][] = $data;
        }

        if (!is_array($packages) || count($packages) == 0) return [];

        $key = $this->getRedisKeyMembershipsLogic();
        if ($this->initRedis()) {
            $this->setRedis($key, json_encode($packages), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }

            $filename_members =  $this->getFileMembersLogic();
            $file_path_members = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_members;

            file_put_contents($file_path_members, json_encode($packages));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $packages;
    }

    public function getLeaderboardSports($data)
    {
        if (!is_array($data) || count($data) == 0) return;

        $sports_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyLeaderboardSports();

        // Extract sports data from the API response structure
        $sports_array = isset($data['key']) && is_array($data['key']) ? $data['key'] : $data;

        foreach ($sports_array as $sport_id => $sport) {
            // Validate that sport is an array with required fields
            if (is_array($sport) && isset($sport['name']) && !empty($sport['name']) && isset($sport['id'])) {
                $sports_data[$sport['id']] = $sport['name'];
            } else {
                error_log(sprintf(__('SH Mapping: Skipping invalid leaderboard sport data - missing id or name. Sport array key: %s', 'sh-mapping'), $sport_id));
            }
        }

        if ($this->initRedis() && is_array($sports_data) && count($sports_data) > 0) {
            $this->setRedis($key, json_encode($sports_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }

            $filename_sports =  $this->getFileNameLeaderboardSports();
            $file_path_sports = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_sports;
            file_put_contents($file_path_sports, json_encode($sports_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $sports_data;
    }

    public function getLeaderboardCategories($data)
    {
        if (!is_array($data) || count($data) == 0) return [];

        $categories_data = [];
        $this->setRedisGroup(REDIS_GROUP_SHMAPPING);
        $key = $this->getRedisKeyLeaderboardCategories();

        // Extract sports data from the API response structure
        $sports_data = isset($data['key']) && is_array($data['key']) ? $data['key'] : $data;

        foreach ($sports_data as $sport_id => $sport) {
            // Validate that sport is an array with required fields
            if (!is_array($sport) || !isset($sport['id']) || !isset($sport['name'])) {
                error_log(sprintf(__('SH Mapping: Skipping invalid sport data - missing id or name. Sport array key: %s', 'sh-mapping'), $sport_id));
                continue;
            }

            if (isset($sport['categories']) && is_array($sport['categories']) && count($sport['categories']) > 0) {
                $categories = [];
                $categories[$sport['id']] = $sport['name'] . ' - ' . __('Any Category');

                foreach ($sport['categories'] as $category) {
                    // Validate category has required fields
                    if (isset($category['id']) && isset($category['name'])) {
                        $categories[$sport['id'] . '-' . $category['id']] = $category['name'];
                    } else {
                        error_log(sprintf(__('SH Mapping: Skipping invalid category data for sport %s - missing id or name', 'sh-mapping'), $sport['name']));
                    }
                }
            } else {
                $categories = $sport['name'];
            }

            if (is_array($categories)) {
                $categories_data[$sport['name']] = $categories;
            } else {
                $categories_data[$sport['id']] = $categories;
            }
        }

        if ($this->initRedis() && is_array($categories_data) && count($categories_data) > 0) {
            $this->setRedis($key, json_encode($categories_data), SHMAPPING_REDIS_TIMEOUT);
        }

        try {
            if (!file_exists(SHMAPPING_PLUGIN_LOG_DIR)) {
                wp_mkdir_p(SHMAPPING_PLUGIN_LOG_DIR);
            }

            $filename_categories =  $this->getFileNameLeaderboardCategories();
            $file_path_categories = SHMAPPING_PLUGIN_LOG_DIR . '/' . $filename_categories;
            file_put_contents($file_path_categories, json_encode($categories_data));
        } catch (\Exception $exception) {
            error_log(print_r([
                'message' => 'SH Mapping: ' . $exception->getMessage(),
            ], true));
        }

        return $categories_data;
    }

    /**
     * Get sports hierarchy options data from API
     * 
     * @return array Hierarchy data from API
     */
    public function getSportsHierarchyData()
    {
        try {
            // Get API URL using private method
            $api_url = $this->getRequestMappinghierarchy();

            if ($api_url === null) {
                error_log(__('SH Mapping: Cannot fetch hierarchy options data due to missing API token.', 'sh-mapping'));
                return [];
            }

            // Make API request using private method
            $hierarchy_data = $this->exeRequestSimple($api_url);

            if (is_array($hierarchy_data) && count($hierarchy_data) > 0) {
                return $hierarchy_data;
            } else {
                error_log('SH Mapping Helper: No hierarchy data received from API');
                return [];
            }
        } catch (Exception $e) {
            error_log('SH Mapping Helper: Exception in getSportsHierarchyData: ' . $e->getMessage());
            return [];
        }
    }

    public function getFileNameData()
    {
        return 'SH_MAPPING.json';
    }

    public function getFileNameDataHierarchy()
    {
        return 'SH_MAPPING_HIERARCHY.json';
    }

    public function getFileNameDataPlayers()
    {
        return 'SH_MAPPING_PLAYERS.json';
    }

    public function getFileNameDataMemberships()
    {
        return 'SH_MAPPING_MEMBERSHIPS.json';
    }

    public function getFileNameSportsOptions()
    {
        return 'SH_MAPPING_SPORTS_OPTIONS.json';
    }

    public function getFileNameSportsOnly()
    {
        return 'SH_MAPPING_SPORTS_ONLY.json';
    }

    public function getFileNameTitles()
    {
        return 'SH_MAPPING_TITLES.json';
    }

    public function getFileNameLeaguesDirect()
    {
        return 'SH_MAPPING_LEAGUES_DIRECT.json';
    }

    public function getFileNameTeamsDirect()
    {
        return 'SH_MAPPING_TEAMS_DIRECT.json';
    }

    public function getFileNameSeasonsDirect()
    {
        return 'SH_MAPPING_SEASONS_DIRECT.json';
    }

    public function getFileNameSportsDirect()
    {
        return 'SH_MAPPING_SPORTS_DIRECT.json';
    }

    public function getFileNameLeaguesSportDirect()
    {
        return 'SH_MAPPING_LEAGUES_SPORT_DIRECT.json';
    }

    public function getFileNameHandicappers()
    {
        return 'SH_MAPPING_HANDICAPPERS.json';
    }

    public function getFileMembersLogic()
    {
        return 'SH_MAPPING_MEMBERS_LOGIC.json';
    }

    public function getFileNameLeaderboard()
    {
        return 'SH_MAPPING_LEADERBOARD.json';
    }

    public function getFileNameLeaderboardSports()
    {
        return 'SH_MAPPING_LEADERBOARD_SPORTS.json';
    }

    public function getFileNameLeaderboardCategories()
    {
        return 'SH_MAPPING_LEADERBOARD_CATEGORIES.json';
    }

    public function getFileNameHierarchySportsLeagues()
    {
        return 'SH_MAPPING_HIERARCHY_SPORTS_LEAGUES.json';
    }
}
