<?php

/**
 * Manage reviews entity and DB management
 *
 * @link       https://www.wpcostarica.com
 * @since      1.0.0
 *
 * @package    blocks/directory
 * @subpackage blocks/directory/models
 * @author     WPCR <mario@wpcostarica.com>
 */

class WpcrReviewEntity
{
    private $current_page = 0;
    private $items_per_page = 10;
    private $total_items = 0;
    private $attributes = [];

    public function __construct($items_per_page = 10, $current_page = 0, $attributes = []) {
        $this->items_per_page = $items_per_page;
        $this->current_page = $current_page;
        $this->attributes = $attributes;
    }

    public function get_term_query() {
        global $wpdb;
        $terms = [];
        $terms['show'] = '';
        $terms['join'] = '';
        $terms['and'] = '';
        $terms['posts'] = '';

        if ('latest' == $this->attributes['show']) {
            $terms['show'] = 'latest';

            if ('include' == $this->attributes['select_category_type'] && !empty($this->attributes['category_include'])) {
                $terms['join'] = ' LEFT JOIN '.$wpdb->term_relationships.' tr ON (p.ID = tr.object_id) ';
                $terms['and'] = ' (tr.term_taxonomy_id IN ('.implode(',', $this->attributes['category_include']).')) AND ';
            } elseif ('exclude' == $this->attributes['select_category_type'] && !empty($this->attributes['category_exclude'])) {
                $terms['and'] = ' (p.ID NOT IN (SELECT object_id FROM '.$wpdb->term_relationships.' tr WHERE tr.term_taxonomy_id IN ('.implode(',', $this->attributes['category_exclude']).'))) AND ';
            }
        } elseif ('specific' == $this->attributes['show']) {
            $terms['show'] = 'specific';
            $terms['posts'] = implode(',', $this->attributes['posts']);
        }

        return $terms;
    }

    /**
     * Get a page of reviews items.
     *
     * @since 1.0.0
     */
    public function get_paged_reviews($letter = false) {
        global $wpdb;
        $reviews = [];
        $tems = $this->get_term_query();

        if ('specific' == $tems['show']) {
            if ($letter == '#') {
                $query_base = "FROM $wpdb->posts p WHERE p.ID IN (".$tems['posts'].") AND p.post_title REGEXP '^[0-9]'";
            } elseif ($letter === false) {
                $query_base = "FROM $wpdb->posts p WHERE p.ID IN (".$tems['posts'].")";
            } else {
                $query_base = "FROM $wpdb->posts p WHERE p.ID IN (".$tems['posts'].") AND p.post_title LIKE '$letter%'";
            }
        } else {
            if ($letter == '#') {
                $query_base = "FROM $wpdb->posts p ".$tems['join']." WHERE ".$tems['and']." p.post_type = 'reviews' AND p.post_status = 'publish' AND p.post_title REGEXP '^[0-9]'";
            } elseif ($letter === false) {
                $query_base = "FROM $wpdb->posts p ".$tems['join']." WHERE ".$tems['and']." p.post_type = 'reviews' AND p.post_status = 'publish'";
            } else {
                $query_base = "FROM $wpdb->posts p ".$tems['join']." WHERE ".$tems['and']." p.post_type = 'reviews' AND p.post_status = 'publish' AND p.post_title LIKE '$letter%'";
            }
        }

        if (empty($tems['show']) || 'latest' == $tems['show']) {
            $review_count = $wpdb->get_var("SELECT COUNT(p.ID) $query_base");
        } else {
            $review_count = count(explode(',', $tems['posts']));
        }

        $this->total_items = (int) $review_count;

        // Make sure current page is valid
        $pages = (int) ceil($this->total_items / $this->items_per_page);

        // Get the items for the current page

        if($this->current_page == 1) {
            $offset = 0;
        } else {
            if ($pages == $this->current_page) {
                $offset = ($this->current_page * $this->items_per_page) - $this->items_per_page;
            } else {
                $offset = ($this->current_page - 1) * $this->items_per_page;
            }
        }

        $order = ($letter === false) ? 'ORDER BY p.post_date DESC' : 'ORDER BY p.post_title ASC';
        $review_ids = $wpdb->get_col("SELECT p.ID $query_base $order LIMIT $offset, $this->items_per_page");

        if (count($review_ids)) {
            // Get the posts objects for the current page
            $reviews = get_posts([
                'include' => $review_ids, 
                'post_type' => 'reviews', 
                'orderby' => ($letter === false) ? 'ASC' : 'none'
            ]);
        }

        return $reviews;
    }

    public function get_letter_items_for_bar() {
        global $wpdb;
        $tems = $this->get_term_query();
        $letters = get_transient('reviews_letter_for_bar');

        if (false === ($letters = get_transient('reviews_letter_for_bar'))) {
            $letters = array('#' => false);
            $search = str_split('0123456789abcdefghijklmnopqrstuvwxyz');

            foreach ($search as $letter) {
                if ('specific' == $tems['show']) {
                    $item = $wpdb->get_var( "SELECT ID FROM $wpdb->posts p WHERE p.ID IN (".$tems['posts'].") AND p.post_title LIKE '$letter%'" );
                } else {
                    $item = $wpdb->get_var( "SELECT ID FROM $wpdb->posts p ".$tems['join']." WHERE ".$tems['and']." p.post_type = 'reviews' AND p.post_status = 'publish' AND p.post_title LIKE '$letter%'" );
                }

                if (is_numeric($letter)) {
                    $letters['#'] = $letters['#'] || (bool)$item;
                } else {
                    $letters[strtoupper($letter)] = (bool)$item;
                }
            }

            set_transient( 'reviews_letter_for_bar', $letters, 60*15 );
        }

        return $letters;
    }

    public function get_paginator() {
        // Number of pages
        $pages = (int) ceil($this->total_items / $this->items_per_page);

        // Calculate the rest of the info
        $first = $pages > 1 ? 0 : false;
        $last = $pages > 1 ? $pages : false;
        $prev = $pages > 1 && ($this->current_page > 0) ? ($this->current_page - 1) : false;
        $next = $pages > 1 && ($this->current_page < $pages ) ? ($this->current_page + 1) : false;

        return [
            'total' => $this->total_items,
            'pages' => $pages,
            'current' => $this->current_page,
            'first' => $first,
            'last' => $last,
            'prev' => $prev,
            'next' => $next
        ];
    }
}
