<?php
// Service Worker Options Page
function shn_base_sw_options_page()
{
    add_menu_page(
        'SHN-Base Service Worker Options',
        'SHN-Base SW',
        'manage_options',
        'shn-base-sw-options',
        'shn_base_sw_options_page_content',
        'dashicons-admin-settings',
        50
    );
}
add_action('admin_menu', 'shn_base_sw_options_page');

function shn_base_sw_options_page_content()
{
    if (!current_user_can('manage_options')) {
        return;
    }

    settings_errors('shn_base_sw_messages');
?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        <form action="options.php" method="post">
            <?php
            settings_fields('shn_base_sw_options');
            do_settings_sections('shn-base-sw-options');
            submit_button('Save Settings');
            ?>
        </form>
    </div>
<?php
}

function shn_base_sw_register_settings()
{
    register_setting('shn_base_sw_options', 'shn_base_sw_options', 'shn_base_sw_validate_options');

    add_settings_section('sw_settings_section', 'Service Worker Settings', 'shn_base_sw_section_callback', 'shn-base-sw-options');

    add_settings_field('cache_name', 'Cache Name', 'shn_base_sw_cache_name_callback', 'shn-base-sw-options', 'sw_settings_section');
    add_settings_field('cache_expiration', 'Cache Expiration (days)', 'shn_base_sw_cache_expiration_callback', 'shn-base-sw-options', 'sw_settings_section');
    add_settings_field('enable_sw', 'Enable Service Worker', 'shn_base_sw_enable_callback', 'shn-base-sw-options', 'sw_settings_section');
}
add_action('admin_init', 'shn_base_sw_register_settings');

function shn_base_sw_section_callback($args)
{
    echo '<p>Configure the service worker settings for optimal caching performance.</p>';
}

function shn_base_sw_cache_name_callback($args)
{
    $options = get_option('shn_base_sw_options');
    $cache_name = isset($options['cache_name']) ? esc_attr($options['cache_name']) : 'shn-base-cache-v1';
    echo '<input type="text" name="shn_base_sw_options[cache_name]" value="' . $cache_name . '" />';
    echo '<p class="description">The name of the cache used by the service worker.</p>';
}

function shn_base_sw_cache_expiration_callback($args)
{
    $options = get_option('shn_base_sw_options');
    $cache_expiration = isset($options['cache_expiration']) ? intval($options['cache_expiration']) : 30;
    echo '<input type="number" name="shn_base_sw_options[cache_expiration]" value="' . $cache_expiration . '" />';
    echo '<p class="description">The number of days after which the cache will expire.</p>';
}

function shn_base_sw_enable_callback($args)
{
    $options = get_option('shn_base_sw_options');
    $enable_sw = isset($options['enable_sw']) ? $options['enable_sw'] : 1;
    echo '<input type="checkbox" name="shn_base_sw_options[enable_sw]" value="1" ' . checked(1, $enable_sw, false) . ' />';
    echo '<p class="description">Enable or disable the service worker functionality.</p>';
}

function shn_base_sw_validate_options($input)
{
    $validated = array();
    $validated['cache_name'] = sanitize_text_field($input['cache_name']);
    $validated['cache_expiration'] = intval($input['cache_expiration']);
    $validated['enable_sw'] = isset($input['enable_sw']) ? 1 : 0;
    return $validated;
}

function shn_base_generate_sw()
{
    $options = get_option('shn_base_sw_options');
    $cache_name = isset($options['cache_name']) ? esc_attr($options['cache_name']) : 'shn-base-cache-v1';
    $cache_expiration = isset($options['cache_expiration']) ? intval($options['cache_expiration']) : 30;

    $sw_content = "
    const CACHE_NAME = '" . $cache_name . "';
    const CACHE_EXPIRATION = " . $cache_expiration . ";
    const urlsToCache = [
        '/',
        '/wp-content/themes/shn-base/dist/css/',
        '/wp-content/themes/shn-base/dist/js/'
    ];

    self.addEventListener('install', function(event) {
        event.waitUntil(
            caches.open(CACHE_NAME)
                .then(function(cache) {
                    console.log('SHN-Base: Opened cache');
                    return cache.addAll(urlsToCache);
                })
        );
    });

    self.addEventListener('fetch', function(event) {
        event.respondWith(
            caches.match(event.request)
                .then(function(response) {
                    if (response) {
                        return response;
                    }
                    return fetch(event.request);
                })
        );
    });

    self.addEventListener('activate', function(event) {
        const cacheWhitelist = [CACHE_NAME];
        event.waitUntil(
            caches.keys().then(function(cacheNames) {
                return Promise.all(
                    cacheNames.map(function(cacheName) {
                        if (cacheWhitelist.indexOf(cacheName) === -1) {
                            return caches.delete(cacheName);
                        }
                    })
                );
            })
        );
    });
    ";

    file_put_contents(ABSPATH . 'sw.js', $sw_content);
}

add_action('update_option_shn_base_sw_options', 'shn_base_generate_sw');
