<?php
/**
 * Plugin Name: Smarter Blocks
 * Description: Simple Gutenberg blocks with ACF support and PHP templates
 * Version: 1.0.4
 * Author: RDG Corp
 * License: GPL v2 or later
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class SmarterBlocks {
    
    private $blocks_path;
    private $blocks = [];
    
    public function __construct() {
        $this->blocks_path = plugin_dir_path(__FILE__) . 'blocks/';
        add_action('acf/init', [$this, 'register_blocks']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        
        // Load ACF fields early
        add_action('acf/init', [$this, 'load_acf_fields'], 5);
        
        // Discover blocks early for admin assets
        add_action('init', [$this, 'discover_blocks_early'], 5);
    }
    
    /**
     * Load ACF fields for all blocks
     */
    public function load_acf_fields() {
        if (!function_exists('acf_add_local_field_group')) {
            return;
        }
        
        // Discover blocks first
        $this->discover_blocks();
        
        // Load ACF fields for each block
        foreach ($this->blocks as $block_name => $block_data) {
            if (file_exists($block_data['fields_file'])) {
                require_once $block_data['fields_file'];
            }
        }
    }
    
    /**
     * Discover blocks early for admin assets
     */
    public function discover_blocks_early() {
        $this->discover_blocks();
    }
    
    /**
     * Discover and register all blocks
     */
    public function register_blocks() {
        if (!function_exists('acf_register_block_type')) {
            return;
        }
        
        // Discover blocks
        $this->discover_blocks();
        
        // Register each block
        foreach ($this->blocks as $block_name => $block_data) {
            $this->register_single_block($block_name, $block_data);
        }
    }
    
    /**
     * Discover blocks in the blocks directory
     */
    private function discover_blocks() {
        if (!is_dir($this->blocks_path)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SmarterBlocks: Blocks directory not found: ' . $this->blocks_path);
            }
            return;
        }
        
        $block_dirs = glob($this->blocks_path . '*', GLOB_ONLYDIR);

        foreach ($block_dirs as $block_dir) {
            $block_name = basename($block_dir);
            $this->load_block($block_name, $block_dir);
        }
    }
    
    /**
     * Load a single block
     */
    private function load_block($block_name, $block_dir) {
        $config_file = $block_dir . '/config.php';
        $template_file = $block_dir . '/template.php';
        $fields_file = $block_dir . '/fields.php';
        $css_file = $block_dir . '/style.css';
        $js_file = $block_dir . '/script.js';
        
        // Check if block has required files
        if (!file_exists($config_file) || !file_exists($template_file)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SmarterBlocks: Missing required files for block ' . $block_name);
                error_log('SmarterBlocks: config.php exists: ' . (file_exists($config_file) ? 'YES' : 'NO'));
                error_log('SmarterBlocks: template.php exists: ' . (file_exists($template_file) ? 'YES' : 'NO'));
            }
            return;
        }
        
        // Include config file
        require_once $config_file;
        
        // Get block configuration
        $config_function = "smarter_blocks_" . str_replace('-', '_', $block_name) . "_config";
        if (!function_exists($config_function)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SmarterBlocks: Config function not found: ' . $config_function);
            }
            return;
        }
        
        $config = call_user_func($config_function);
        
        $this->blocks[$block_name] = [
            'config' => $config,
            'dir' => $block_dir,
            'template' => $template_file,
            'fields_file' => $fields_file,
            'css_file' => $css_file,
            'js_file' => $js_file,
        ];
    }
    
    /**
     * Register a single block
     */
    private function register_single_block($block_name, $block_data) {
        $config = $block_data['config'];
        $template = $block_data['template'];
        
        // Prepare block configuration
        $acf_config = [
            'name' => $config['name'] ?? 'acf/' . str_replace('_', '-', $block_name),
            'title' => $config['title'],
            'description' => $config['description'] ?? '',
            'category' => $config['category'] ?? 'widgets',
            'icon' => $config['icon'] ?? 'smiley',
            'keywords' => $config['keywords'] ?? [],
            'supports' => $config['supports'] ?? [],
            'example' => $config['example'] ?? [],
            'render_callback' => [$this, 'render_block'],
            'render_template' => $template,
            'mode' => $config['mode'] ?? 'preview',
            'align' => $config['align'] ?? '',
            'align_text' => $config['align_text'] ?? '',
            'align_content' => $config['align_content'] ?? '',
            'multiple' => $config['multiple'] ?? true,
        ];
        
        
        // Register the block
        acf_register_block_type($acf_config);
    }
    
    /**
     * Render block callback
     */
    public function render_block($block, $content = '', $is_preview = false, $post_id = 0) {
        // Get block name
        $block_name = '';
        if (isset($block['name'])) {
            $block_name = str_replace('acf/', '', $block['name']);
            // Convert back to directory name format (with hyphens)
            $block_name = str_replace('_', '-', $block_name);
        }
        
        // Find block data
        $block_data = $this->blocks[$block_name] ?? null;
        if (!$block_data) {
            return '';
        }
        
        // Get ACF fields
        $fields = get_fields();
        
        // Include template
        include $block_data['template'];
    }
    
    /**
     * Enqueue assets
     */
    public function enqueue_assets() {
        
        // In admin, always enqueue assets for all blocks to support editor
        if (is_admin()) {
            foreach ($this->blocks as $block_name => $block_data) {
                $this->enqueue_block_assets($block_name, $block_data);
            }
            return;
        }
        
        // Check if any blocks are present on the page (frontend only)
        if (!$this->has_blocks_on_page()) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SmarterBlocks: No blocks found on page, skipping asset enqueue');
            }
            return;
        }
        
        // Enqueue CSS and JS for blocks present on the page
        foreach ($this->blocks as $block_name => $block_data) {
            if ($this->is_block_on_page($block_name)) {
                $this->enqueue_block_assets($block_name, $block_data);
            }
        }
    }
    
    /**
     * Check if any blocks are present on the page
     */
    private function has_blocks_on_page() {
        global $post;
        
        if (!$post) {
            return false;
        }
        
        foreach ($this->blocks as $block_name => $block_data) {
            $block_pattern = '<!-- wp:acf/' . $block_name;
            if (strpos($post->post_content, $block_pattern) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Check if specific block is present on the page
     */
    private function is_block_on_page($block_name) {
        global $post;
        
        if (!$post) {
            return false;
        }
        
        $block_pattern = '<!-- wp:acf/' . $block_name;
        
        return strpos($post->post_content, $block_pattern) !== false;
    }
    
    /**
     * Enqueue assets for a specific block
     */
    private function enqueue_block_assets($block_name, $block_data) {
    
        if (!empty($block_data['css_file']) && file_exists($block_data['css_file'])) {
            $handle = "smarter-blocks-{$block_name}";
            $ver    = filemtime($block_data['css_file']);
    
            wp_register_style(
                $handle,
                plugin_dir_url(__FILE__) . "blocks/{$block_name}/style.css",
                [],
                $ver
            );
            wp_enqueue_style($handle);
    
            $editor_css = dirname($block_data['css_file']) . '/editor.css';
            if (is_admin() && file_exists($editor_css)) {
                add_action('enqueue_block_editor_assets', function () use ($editor_css, $block_name) {
                    $handle_editor = "smarter-blocks-editor-{$block_name}";
                    wp_register_style(
                        $handle_editor,
                        plugin_dir_url(__FILE__) . "blocks/{$block_name}/editor.css",
                        ['wp-edit-blocks'],
                        filemtime($editor_css)
                    );
                    wp_enqueue_style($handle_editor);

                });
            }
        }
    }
}

// Initialize plugin
new SmarterBlocks();

