/**
 * Testimonials Block JavaScript
 */

(function() {
    'use strict';
    
    function initTestimonials() {
        const testimonialsSections = document.querySelectorAll('.testimonials-section');
        
        testimonialsSections.forEach(function(section) {
            const content = section.querySelector('.testimonials-section__content');
            const layout = content.classList.contains('testimonials-section__content--slider') ? 'slider' : 'grid';
            
            if (layout === 'slider') {
                initSlider(content);
            }
            
            // Add animation on scroll
            const observer = new IntersectionObserver(function(entries) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        entry.target.style.opacity = '1';
                        entry.target.style.transform = 'translateY(0)';
                    }
                });
            });
            
            // Add initial styles for animation
            section.style.opacity = '0';
            section.style.transform = 'translateY(20px)';
            section.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
            
            observer.observe(section);
        });
    }
    
    function initSlider(content) {
        const items = content.querySelectorAll('.testimonial-item');
        let currentIndex = 0;
        
        if (items.length <= 1) {
            return;
        }
        
        // Create navigation dots
        const dotsContainer = document.createElement('div');
        dotsContainer.className = 'testimonials-slider__dots';
        dotsContainer.style.cssText = `
            display: flex;
            justify-content: center;
            gap: 10px;
            margin-top: 30px;
        `;
        
        items.forEach(function(item, index) {
            const dot = document.createElement('button');
            dot.className = 'testimonials-slider__dot';
            dot.style.cssText = `
                width: 12px;
                height: 12px;
                border-radius: 50%;
                border: none;
                background: #ddd;
                cursor: pointer;
                transition: background 0.3s ease;
            `;
            
            if (index === 0) {
                dot.style.background = '#007cba';
            }
            
            dot.addEventListener('click', function() {
                showSlide(index);
            });
            
            dotsContainer.appendChild(dot);
        });
        
        content.appendChild(dotsContainer);
        
        // Create navigation arrows
        const prevButton = document.createElement('button');
        prevButton.className = 'testimonials-slider__arrow testimonials-slider__arrow--prev';
        prevButton.innerHTML = '‹';
        prevButton.style.cssText = `
            position: absolute;
            left: 10px;
            top: 50%;
            transform: translateY(-50%);
            background: rgba(0, 124, 186, 0.8);
            color: white;
            border: none;
            width: 40px;
            height: 40px;
            border-radius: 50%;
            font-size: 20px;
            cursor: pointer;
            transition: background 0.3s ease;
        `;
        
        const nextButton = document.createElement('button');
        nextButton.className = 'testimonials-slider__arrow testimonials-slider__arrow--next';
        nextButton.innerHTML = '›';
        nextButton.style.cssText = `
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            background: rgba(0, 124, 186, 0.8);
            color: white;
            border: none;
            width: 40px;
            height: 40px;
            border-radius: 50%;
            font-size: 20px;
            cursor: pointer;
            transition: background 0.3s ease;
        `;
        
        prevButton.addEventListener('click', function() {
            const newIndex = currentIndex > 0 ? currentIndex - 1 : items.length - 1;
            showSlide(newIndex);
        });
        
        nextButton.addEventListener('click', function() {
            const newIndex = currentIndex < items.length - 1 ? currentIndex + 1 : 0;
            showSlide(newIndex);
        });
        
        content.appendChild(prevButton);
        content.appendChild(nextButton);
        
        // Set up content container for slider
        content.style.position = 'relative';
        content.style.minHeight = '300px';
        
        function showSlide(index) {
            // Hide all items
            items.forEach(function(item) {
                item.style.display = 'none';
                item.style.opacity = '0';
                item.style.transform = 'translateX(20px)';
                item.style.transition = 'opacity 0.3s ease, transform 0.3s ease';
            });
            
            // Show current item
            items[index].style.display = 'block';
            setTimeout(function() {
                items[index].style.opacity = '1';
                items[index].style.transform = 'translateX(0)';
            }, 50);
            
            // Update dots
            const dots = dotsContainer.querySelectorAll('.testimonials-slider__dot');
            dots.forEach(function(dot, dotIndex) {
                dot.style.background = dotIndex === index ? '#007cba' : '#ddd';
            });
            
            currentIndex = index;
        }
        
        // Auto-advance slider
        setInterval(function() {
            const newIndex = currentIndex < items.length - 1 ? currentIndex + 1 : 0;
            showSlide(newIndex);
        }, 5000);
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initTestimonials);
    } else {
        initTestimonials();
    }
})();
