<?php
/**
 * Plugin Name: Smarter Blocks
 * Description: Simple Gutenberg blocks with ACF support and PHP templates
 * Version: 1.0.12
 * Author: RDG Corp
 * License: GPL v2 or later
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class SmarterBlocks {
    
    private $blocks_path;
    private $blocks = [];
    
    public function __construct() {
        $this->blocks_path = plugin_dir_path(__FILE__) . 'blocks/';
        add_action('acf/init', [$this, 'register_blocks']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        
        // Load ACF fields early
        add_action('acf/init', [$this, 'load_acf_fields'], 5);
        
        // Discover blocks early for admin assets
        add_action('init', [$this, 'discover_blocks_early'], 5);
        
        // Add settings page
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
        
        // Add settings link to plugins page
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'add_settings_link']);
        
    }
    
    /**
     * Load ACF fields for all blocks
     */
    public function load_acf_fields() {
        if (!function_exists('acf_add_local_field_group')) {
            return;
        }
        
        // Discover blocks first
        $this->discover_blocks();
        
        // Load ACF fields for each block
        foreach ($this->blocks as $block_name => $block_data) {
            if (file_exists($block_data['fields_file'])) {
                require_once $block_data['fields_file'];
            }
        }
    }
    
    /**
     * Discover blocks early for admin assets
     */
    public function discover_blocks_early() {
        $this->discover_blocks();
    }
    
    /**
     * Discover and register all blocks
     */
    public function register_blocks() {
        if (!function_exists('acf_register_block_type')) {
            return;
        }
        
        // Discover blocks
        $this->discover_blocks();
        
        // Register each block
        foreach ($this->blocks as $block_name => $block_data) {
            $this->register_single_block($block_name, $block_data);
        }
    }
    
    /**
     * Discover blocks in the blocks directory
     */
    private function discover_blocks() {
        if (!is_dir($this->blocks_path)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SmarterBlocks: Blocks directory not found: ' . $this->blocks_path);
            }
            return;
        }
        
        $block_dirs = glob($this->blocks_path . '*', GLOB_ONLYDIR);

        foreach ($block_dirs as $block_dir) {
            $block_name = basename($block_dir);
            $this->load_block($block_name, $block_dir);
        }
    }
    
    /**
     * Load a single block
     */
    private function load_block($block_name, $block_dir) {
        $config_file = $block_dir . '/config.php';
        $template_file = $block_dir . '/template.php';
        $fields_file = $block_dir . '/fields.php';
        $css_file = $block_dir . '/style.css';
        $js_file = $block_dir . '/script.js';
        
        // Check if block has required files
        if (!file_exists($config_file) || !file_exists($template_file)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SmarterBlocks: Missing required files for block ' . $block_name);
                error_log('SmarterBlocks: config.php exists: ' . (file_exists($config_file) ? 'YES' : 'NO'));
                error_log('SmarterBlocks: template.php exists: ' . (file_exists($template_file) ? 'YES' : 'NO'));
            }
            return;
        }
        
        // Include config file
        require_once $config_file;
        
        // Get block configuration
        $config_function = "smarter_blocks_" . str_replace('-', '_', $block_name) . "_config";
        if (!function_exists($config_function)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SmarterBlocks: Config function not found: ' . $config_function);
            }
            return;
        }
        
        $config = call_user_func($config_function);
        
        $this->blocks[$block_name] = [
            'config' => $config,
            'dir' => $block_dir,
            'template' => $template_file,
            'fields_file' => $fields_file,
            'css_file' => $css_file,
            'js_file' => $js_file,
        ];
    }
    
    /**
     * Register a single block
     */
    private function register_single_block($block_name, $block_data) {
        $config = $block_data['config'];
        $template = $block_data['template'];
        
        // Prepare block configuration
        $acf_config = [
            'name' => $config['name'] ?? 'acf/' . str_replace('_', '-', $block_name),
            'title' => $config['title'],
            'description' => $config['description'] ?? '',
            'category' => $config['category'] ?? 'widgets',
            'icon' => $config['icon'] ?? 'smiley',
            'keywords' => $config['keywords'] ?? [],
            'supports' => $config['supports'] ?? [],
            'example' => $config['example'] ?? [],
            'render_callback' => [$this, 'render_block'],
            'render_template' => $template,
            'mode' => $config['mode'] ?? 'preview',
            'align' => $config['align'] ?? '',
            'align_text' => $config['align_text'] ?? '',
            'align_content' => $config['align_content'] ?? '',
            'multiple' => $config['multiple'] ?? true,
        ];
        
        
        // Register the block
        acf_register_block_type($acf_config);
    }
    
    /**
     * Render block callback
     */
    public function render_block($block, $content = '', $is_preview = false, $post_id = 0) {
        // Get block name
        $block_name = '';
        if (isset($block['name'])) {
            $block_name = str_replace('acf/', '', $block['name']);
            // Convert back to directory name format (with hyphens)
            $block_name = str_replace('_', '-', $block_name);
        }
        
        // Find block data
        $block_data = $this->blocks[$block_name] ?? null;
        if (!$block_data) {
            return '';
        }
        
        // Get ACF fields
        $fields = get_fields();
        
        // Include template
        include $block_data['template'];
    }
    
    /**
     * Enqueue global assets
     */
    private function enqueue_global_assets() {
        $global_css = plugin_dir_path(__FILE__) . 'assets/css/global.css';
        if (file_exists($global_css)) {
            wp_register_style(
                'smarter-blocks-global',
                plugin_dir_url(__FILE__) . 'assets/css/global.css',
                [],
                filemtime($global_css)
            );
            wp_enqueue_style('smarter-blocks-global');
        }
    }
    
    /**
     * Enqueue assets
     */
    public function enqueue_assets() {
        // Enqueue global CSS
        $this->enqueue_global_assets();
        
        // In admin, always enqueue assets for all blocks to support editor
        if (is_admin()) {
            foreach ($this->blocks as $block_name => $block_data) {
                $this->enqueue_block_assets($block_name, $block_data);
            }
            return;
        }
        
        // Check if any blocks are present on the page (frontend only)
        if (!$this->has_blocks_on_page()) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SmarterBlocks: No blocks found on page, skipping asset enqueue');
            }
            return;
        }
        
        // Enqueue CSS and JS for blocks present on the page
        foreach ($this->blocks as $block_name => $block_data) {
            if ($this->is_block_on_page($block_name)) {
                $this->enqueue_block_assets($block_name, $block_data);
            }
        }
    }
    
    /**
     * Check if any blocks are present on the page
     */
    private function has_blocks_on_page() {
        global $post;
        
        if (!$post) {
            return false;
        }
        
        foreach ($this->blocks as $block_name => $block_data) {
            $block_pattern = '<!-- wp:acf/' . $block_name;
            if (strpos($post->post_content, $block_pattern) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Check if specific block is present on the page
     */
    private function is_block_on_page($block_name) {
        global $post;
        
        if (!$post) {
            return false;
        }
        
        $block_pattern = '<!-- wp:acf/' . $block_name;
        
        return strpos($post->post_content, $block_pattern) !== false;
    }
    
    /**
     * Enqueue assets for a specific block
     */
    private function enqueue_block_assets($block_name, $block_data) {
    
        if (!empty($block_data['css_file']) && file_exists($block_data['css_file'])) {
            $handle = "smarter-blocks-{$block_name}";
            $ver    = filemtime($block_data['css_file']);
    
            wp_register_style(
                $handle,
                plugin_dir_url(__FILE__) . "blocks/{$block_name}/style.css",
                [],
                $ver
            );
            wp_enqueue_style($handle);
    
            $editor_css = dirname($block_data['css_file']) . '/editor.css';
            if (is_admin() && file_exists($editor_css)) {
                add_action('enqueue_block_editor_assets', function () use ($editor_css, $block_name) {
                    $handle_editor = "smarter-blocks-editor-{$block_name}";
                    wp_register_style(
                        $handle_editor,
                        plugin_dir_url(__FILE__) . "blocks/{$block_name}/editor.css",
                        ['wp-edit-blocks'],
                        filemtime($editor_css)
                    );
                    wp_enqueue_style($handle_editor);

                });
            }
        }

        // Enqueue JavaScript files
        if (!empty($block_data['js_file']) && file_exists($block_data['js_file'])) {
            $handle_js = "smarter-blocks-{$block_name}-js";
            $ver_js = filemtime($block_data['js_file']);
            
            wp_register_script(
                $handle_js,
                plugin_dir_url(__FILE__) . "blocks/{$block_name}/script.js",
                [],
                $ver_js,
                true
            );
            wp_enqueue_script($handle_js);
        }
    }
    
    /**
     * Add settings page to admin menu
     */
    public function add_settings_page() {
        add_menu_page(
            'Smarter Blocks',
            'Smarter Blocks',
            'manage_options',
            'smarter-blocks-settings',
            [$this, 'settings_page_callback'],
            'dashicons-block-default',
            30
        );
        
        // Add submenu for User Signup settings
        add_submenu_page(
            'smarter-blocks-settings',
            'User Signup Settings',
            'User Signup',
            'manage_options',
            'smarter-blocks-user-signup',
            [$this, 'user_signup_settings_callback']
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        // User Signup Settings
        register_setting('smarter_blocks_user_signup', 'smarter_blocks_user_signup_form_action');
        register_setting('smarter_blocks_user_signup', 'smarter_blocks_user_signup_form_method');
        register_setting('smarter_blocks_user_signup', 'smarter_blocks_user_signup_google_client_id');
        register_setting('smarter_blocks_user_signup', 'smarter_blocks_user_signup_apple_client_id');
        register_setting('smarter_blocks_user_signup', 'smarter_blocks_user_signup_google_analytics_id');
        
        // reCAPTCHA Settings
        register_setting('smarter_blocks_user_signup', 'smarter_blocks_recaptcha_site_key');
        register_setting('smarter_blocks_user_signup', 'smarter_blocks_recaptcha_secret_key');
        register_setting('smarter_blocks_user_signup', 'smarter_blocks_recaptcha_enabled');
        
        // Custom Google Analytics Events
        register_setting('smarter_blocks_user_signup', 'smarter_blocks_custom_ga_events');
    }
    
    /**
     * Main settings page callback
     */
    public function settings_page_callback() {
        ?>
        <div class="wrap">
            <h1><span class="dashicons dashicons-block-default"></span> Smarter Blocks</h1>
            
            <div class="smarter-blocks-dashboard">                
                <div class="postbox">
                    <div class="inside">
                        <h2 class="hndle">Welcome to Smarter Blocks</h2>
                        <h3>Plugin Information</h3>
                        <table class="form-table">
                            <tr>
                                <th scope="row">Version</th>
                                <td>1.0.12</td>
                            </tr>
                            <tr>
                                <th scope="row">Author</th>
                                <td>RDG Corp</td>
                            </tr>
                            <tr>
                                <th scope="row">License</th>
                                <td>GPL v2 or later</td>
                            </tr>
                            <tr>
                                <th scope="row">Requirements</th>
                                <td>WordPress 6.0+, PHP 8.3+, ACF Pro (recommended)</td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
        .smarter-blocks-dashboard .postbox {
            margin: 20px 0;
        }
        .smarter-blocks-dashboard .inside {
            padding: 20px;
        }
        .smarter-blocks-dashboard h3 {
            margin-top: 20px;
            margin-bottom: 10px;
        }
        .smarter-blocks-dashboard ul {
            margin-left: 20px;
        }
        .smarter-blocks-dashboard li {
            margin-bottom: 5px;
        }
        </style>
        <?php
    }
    
    /**
     * User Signup settings page callback
     */
    public function user_signup_settings_callback() {
        ?>
        <div class="wrap">
            <h1 style="display: flex; align-items: center; gap: 8px;"><span class="dashicons dashicons-admin-users"></span> User Signup Block Settings</h1>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('smarter_blocks_user_signup');
                do_settings_sections('smarter_blocks_user_signup');
                ?>
                
                <div class="postbox">
                    <h2 class="hndle">Form Configuration</h2>
                    <div class="inside">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="smarter_blocks_user_signup_form_action">Form Action URL</label>
                                </th>
                                <td>
                                    <input type="url" 
                                           id="smarter_blocks_user_signup_form_action" 
                                           name="smarter_blocks_user_signup_form_action" 
                                           value="<?php echo esc_attr(get_option('smarter_blocks_user_signup_form_action', '')); ?>" 
                                           class="regular-text" 
                                           placeholder="https://example.com/register" />
                                    <p class="description">URL where the registration form will be sent</p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="smarter_blocks_user_signup_form_method">Form Method</label>
                                </th>
                                <td>
                                    <select id="smarter_blocks_user_signup_form_method" name="smarter_blocks_user_signup_form_method">
                                        <option value="post" <?php selected(get_option('smarter_blocks_user_signup_form_method', 'post'), 'post'); ?>>POST</option>
                                        <option value="get" <?php selected(get_option('smarter_blocks_user_signup_form_method', 'post'), 'get'); ?>>GET</option>
                                    </select>
                                    <p class="description">HTTP method to send the form</p>
                                </td>
                            </tr>
                            
                        </table>
                    </div>
                </div>
                
                
                <div class="postbox">
                    <h2 class="hndle">Social Login Configuration</h2>
                    <div class="inside">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="smarter_blocks_user_signup_google_client_id">Google Client ID</label>
                                </th>
                                <td>
                                    <input type="text" 
                                           id="smarter_blocks_user_signup_google_client_id" 
                                           name="smarter_blocks_user_signup_google_client_id" 
                                           value="<?php echo esc_attr(get_option('smarter_blocks_user_signup_google_client_id', '')); ?>" 
                                           class="regular-text" 
                                           placeholder="123456789-abcdefg.apps.googleusercontent.com" />
                                    <p class="description">Google Client ID for OAuth authentication</p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="smarter_blocks_user_signup_apple_client_id">Apple Client ID</label>
                                </th>
                                <td>
                                    <input type="text" 
                                           id="smarter_blocks_user_signup_apple_client_id" 
                                           name="smarter_blocks_user_signup_apple_client_id" 
                                           value="<?php echo esc_attr(get_option('smarter_blocks_user_signup_apple_client_id', '')); ?>" 
                                           class="regular-text" 
                                           placeholder="com.example.app" />
                                    <p class="description">Apple Client ID for OAuth authentication</p>
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>
                
                <div class="postbox">
                    <h2 class="hndle">Google Analytics Configuration</h2>
                    <div class="inside">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="smarter_blocks_user_signup_google_analytics_id">Google Analytics ID</label>
                                </th>
                                <td>
                                    <input type="text" 
                                           id="smarter_blocks_user_signup_google_analytics_id" 
                                           name="smarter_blocks_user_signup_google_analytics_id" 
                                           value="<?php echo esc_attr(get_option('smarter_blocks_user_signup_google_analytics_id', '')); ?>" 
                                           class="regular-text" 
                                           placeholder="G-XXXXXXXXXX" />
                                    <p class="description">ID de Google Analytics para tracking de eventos del formulario (ej: G-XXXXXXXXXX)</p>
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>
                
                <div class="postbox">
                    <h2 class="hndle">reCAPTCHA Configuration</h2>
                    <div class="inside">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="smarter_blocks_recaptcha_enabled">Enable reCAPTCHA</label>
                                </th>
                                <td>
                                    <input type="checkbox" 
                                           id="smarter_blocks_recaptcha_enabled" 
                                           name="smarter_blocks_recaptcha_enabled" 
                                           value="1" 
                                           <?php checked(get_option('smarter_blocks_recaptcha_enabled', 0), 1); ?> />
                                    <p class="description">Enable reCAPTCHA protection for the signup form</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="smarter_blocks_recaptcha_site_key">reCAPTCHA Site Key</label>
                                </th>
                                <td>
                                    <input type="text" 
                                           id="smarter_blocks_recaptcha_site_key" 
                                           name="smarter_blocks_recaptcha_site_key" 
                                           value="<?php echo esc_attr(get_option('smarter_blocks_recaptcha_site_key', '')); ?>" 
                                           class="regular-text" 
                                           placeholder="6LcXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" />
                                    <p class="description">Your reCAPTCHA Site Key (public key)</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="smarter_blocks_recaptcha_secret_key">reCAPTCHA Secret Key</label>
                                </th>
                                <td>
                                    <input type="password" 
                                           id="smarter_blocks_recaptcha_secret_key" 
                                           name="smarter_blocks_recaptcha_secret_key" 
                                           value="<?php echo esc_attr(get_option('smarter_blocks_recaptcha_secret_key', '')); ?>" 
                                           class="regular-text" 
                                           placeholder="6LcXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" />
                                    <p class="description">Your reCAPTCHA Secret Key (private key)</p>
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>
                
                <div class="postbox">
                    <h2 class="hndle">Custom Google Analytics Events</h2>
                    <div class="inside">
                        <p>Configure custom Google Analytics events with your own JavaScript code. Each event has a trigger name and custom code that will be executed.</p>
                        
                        <div id="custom-ga-events-repeater">
                            <?php 
                            $custom_ga_events = get_option('smarter_blocks_custom_ga_events', []);
                            
                            // Default events if none exist
                            if (empty($custom_ga_events)) {
                                $custom_ga_events = [
                                    [
                                        'name' => 'form_start',
                                        'code' => "dataLayer.push({\n  event: 'sign_up_start',\n  method: 'email',\n  form_id: 'signup_main_v1',\n  form_variant: 'control'\n});"
                                    ],
                                    [
                                        'name' => 'form_submit_success',
                                        'code' => "dataLayer.push({\n  event: 'sign_up',\n  method: 'email',\n  form_id: 'signup_main_v1',\n  form_variant: 'control'\n});"
                                    ],
                                    [
                                        'name' => 'conversion',
                                        'code' => "gtag('event', 'conversion', {\n  send_to: 'AW-CONVERSION_ID/CONVERSION_LABEL',\n  value: 1.0,\n  currency: 'USD'\n});"
                                    ]
                                ];
                            }
                            
                            foreach ($custom_ga_events as $index => $event): ?>
                            <div class="custom-ga-event-row" data-index="<?php echo $index; ?>">
                                <table class="form-table">
                                    <tr>
                                        <th scope="row">
                                            <label>Event Trigger Name</label>
                                        </th>
                                        <td>
                                            <input type="text" 
                                                   name="smarter_blocks_custom_ga_events[<?php echo $index; ?>][name]" 
                                                   value="<?php echo esc_attr($event['name'] ?? ''); ?>" 
                                                   class="regular-text" 
                                                   placeholder="form_start" />
                                            <p class="description">The trigger name that will be called from JavaScript</p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row">
                                            <label>Custom JavaScript Code</label>
                                        </th>
                                        <td>
                                            <textarea name="smarter_blocks_custom_ga_events[<?php echo $index; ?>][code]" 
                                                      rows="8" 
                                                      class="large-text code" 
                                                      placeholder="dataLayer.push({
  event: 'sign_up',
  method: 'email',
  form_id: 'signup_main_v1',
  form_variant: 'control'
});"><?php echo esc_textarea($event['code'] ?? ''); ?></textarea>
                                            <p class="description">Custom JavaScript code that will be executed when this event is triggered</p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"></th>
                                        <td>
                                            <button type="button" class="button button-secondary remove-custom-ga-event">Remove Event</button>
                                        </td>
                                    </tr>
                                </table>
                                <hr style="margin: 20px 0;">
                            </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <div style="margin-top: 20px;">
                            <button type="button" id="add-custom-ga-event" class="button button-secondary">Add New Event</button>
                        </div>
                        
                        <div class="notice notice-info ga-help-section">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                                <strong>How to use Google Analytics Events</strong>
                                <button type="button" class="ga-help-close" style="background: none; border: none; font-size: 18px; cursor: pointer; color: #666;">✕</button>
                            </div>
                            
                            <div class="ga-help-content">
                                <ul>
                                    <li><strong>Event Trigger Name:</strong> The name that will be called from JavaScript</li>
                                    <li><strong>Custom JavaScript Code:</strong> Your custom code that will be executed (dataLayer.push, gtag, etc.)</li>
                                </ul>
                                
                                <h4>Available Built-in Triggers:</h4>
                                <ul>
                                    <li><code>form_start</code> - User starts filling the form</li>
                                    <li><code>form_click</code> - User clicks on the form</li>
                                    <li><code>form_submit_attempt</code> - User attempts to submit</li>
                                    <li><code>form_submit_success</code> - Form submitted successfully</li>
                                    <li><code>form_error</code> - Form validation errors</li>
                                    <li><code>social_login_click</code> - Social login button clicks</li>
                                    <li><code>conversion</code> - Conversion tracking</li>
                                </ul>
                                
                                <h4>Custom Triggers:</h4>
                                <p>You can create your own trigger names and call them manually from your custom JavaScript code using:</p>
                                <code>trackEvent('your_custom_trigger_name');</code>
                                
                                <h4>Example Custom Trigger:</h4>
                                <p>If you want to track when a user spends 30 seconds on the form, you could:</p>
                                <ol>
                                    <li>Create an event with trigger name: <code>form_30_seconds</code></li>
                                    <li>Add your custom code: <code>dataLayer.push({event: 'form_engagement', duration: 30});</code></li>
                                    <li>Call it from your code: <code>setTimeout(() => trackEvent('form_30_seconds'), 30000);</code></li>
                                </ol>
                            </div>
                        </div>
                    </div>
                </div>
                
                
                <?php submit_button('Save Settings', 'primary', 'submit', true, ['id' => 'submit']); ?>
            </form>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            let eventIndex = <?php echo count($custom_ga_events); ?>;
            
            // Add new custom event
            $('#add-custom-ga-event').on('click', function() {
                const newEventHtml = `
                    <div class="custom-ga-event-row" data-index="${eventIndex}">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label>Event Trigger Name</label>
                                </th>
                                <td>
                                    <input type="text" 
                                           name="smarter_blocks_custom_ga_events[${eventIndex}][name]" 
                                           value="" 
                                           class="regular-text" 
                                           placeholder="form_start" />
                                    <p class="description">The trigger name that will be called from JavaScript</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label>Custom JavaScript Code</label>
                                </th>
                                <td>
                                    <textarea name="smarter_blocks_custom_ga_events[${eventIndex}][code]" 
                                              rows="8" 
                                              class="large-text code" 
                                              placeholder="dataLayer.push({
  event: 'sign_up',
  method: 'email',
  form_id: 'signup_main_v1',
  form_variant: 'control'
});"></textarea>
                                    <p class="description">Custom JavaScript code that will be executed when this event is triggered</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"></th>
                                <td>
                                    <button type="button" class="button button-secondary remove-custom-ga-event">Remove Event</button>
                                </td>
                            </tr>
                        </table>
                        <hr style="margin: 20px 0;">
                    </div>
                `;
                
                $('#custom-ga-events-repeater').append(newEventHtml);
                eventIndex++;
            });
            
            // Remove custom event
            $(document).on('click', '.remove-custom-ga-event', function() {
                $(this).closest('.custom-ga-event-row').remove();
            });
            
            // Close help section
            $(document).on('click', '.ga-help-close', function() {
                $(this).closest('.ga-help-section').fadeOut();
            });
        });
        </script>
        <?php
    }
    
    /**
     * Add settings link to plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=smarter-blocks-settings') . '">Settings</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
}

new SmarterBlocks();

